import { BrowserRouter as Router, Routes, Route, Navigate, useLocation } from 'react-router-dom';
import { useEffect } from 'react';
import { AuthProvider, useAuth } from './context/AuthContext';
import { SocketProvider } from './context/SocketContext';
import { pageview } from './utils/analytics';
import { fetchCSRFToken } from './utils/csrf';
import Navbar from './components/Navbar';
import Footer from './components/Footer';
import Landing from './pages/Landing';
import SignIn from './pages/SignIn';
import SignUp from './pages/SignUp';
import ForgotPassword from './pages/ForgotPassword';
import ResetPassword from './pages/ResetPassword';
import VerifyEmail from './pages/VerifyEmail';
import Community from './pages/Community';
import Workflows from './pages/Workflows';
import Models from './pages/Models';
import ModelDetail from './pages/ModelDetail';
import Showcase from './pages/Showcase';
import Challenges from './pages/Challenges';
import Jobs from './pages/Jobs';
import VibeClasses from './pages/VibeClasses';
import Summit from './pages/Summit';
import Dashboard from './pages/Dashboard';
import ReferralDashboard from './pages/ReferralDashboard';
import Referrals from './pages/Referrals';
import Search from './pages/Search';
import Messages from './pages/Messages';
import Notifications from './pages/Notifications';
import Marketplace from './pages/Marketplace';
import ProductDetail from './pages/ProductDetail';
import Blog from './pages/Blog';
import BlogPost from './pages/BlogPost';
import Cart from './pages/Cart';
import Pricing from './pages/Pricing';
import Checkout from './pages/Checkout';
import CheckoutSuccess from './pages/CheckoutSuccess';
import SubscriptionSuccess from './pages/SubscriptionSuccess';
import CreatorDashboard from './pages/CreatorDashboard';
import JobPostPricing from './pages/JobPostPricing';
import JobAnalytics from './pages/JobAnalytics';
import ChallengeSponsorPricing from './pages/ChallengeSponsorPricing';
import ChallengeSponsorDashboard from './pages/ChallengeSponsorDashboard';
import SponsoredChallengeSuccess from './pages/SponsoredChallengeSuccess';
import CustomProfile from './pages/CustomProfile';
import PlatformAnalytics from './pages/PlatformAnalytics';
import AdminDashboard from './pages/AdminDashboard';
import Privacy from './pages/Privacy';
import Terms from './pages/Terms';
import CookiePolicy from './pages/CookiePolicy';
import DMCA from './pages/DMCA';
import RefundPolicy from './pages/RefundPolicy';
import WorkflowDemo from './pages/WorkflowDemo';
import OpenAIChallengeDemo from './pages/OpenAIChallengeDemo';

const ProtectedRoute = ({ children }) => {
  const { isAuthenticated, loading } = useAuth();

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return isAuthenticated ? children : <Navigate to="/signin" />;
};

// Analytics component to track page views
function AnalyticsTracker() {
  const location = useLocation();

  useEffect(() => {
    pageview(location.pathname + location.search);
  }, [location]);

  return null;
}

function AppContent() {
  const location = useLocation();

  useEffect(() => {
    pageview(location.pathname + location.search);
  }, [location]);

  // Initialize CSRF token on app mount
  useEffect(() => {
    const initCSRF = async () => {
      try {
        await fetchCSRFToken();
      } catch (error) {
        console.error('Failed to initialize CSRF token:', error);
      }
    };
    initCSRF();
  }, []);

  return (
    <div className="flex flex-col min-h-screen">
      <Navbar />
      <main className="flex-grow">
        <Routes>
            <Route path="/" element={<Landing />} />
            <Route path="/signin" element={<SignIn />} />
            <Route path="/signup" element={<SignUp />} />
            <Route path="/forgot-password" element={<ForgotPassword />} />
            <Route path="/reset-password/:resetToken" element={<ResetPassword />} />
            <Route path="/verify-email/:token" element={<VerifyEmail />} />
            <Route path="/community" element={<Community />} />
            <Route path="/workflows" element={<Workflows />} />
            <Route path="/models" element={<Models />} />
            <Route path="/showcase" element={<Showcase />} />
            <Route path="/models/:modelId" element={<ModelDetail />} />
            <Route path="/workflows/gpt4-chatbot" element={<WorkflowDemo />} />
            <Route path="/challenges" element={<Challenges />} />
            <Route path="/challenges/openai-innovation" element={<OpenAIChallengeDemo />} />
            <Route path="/challenges/sponsor/success" element={<SponsoredChallengeSuccess />} />
            <Route
              path="/challenges/:challengeId/sponsor"
              element={
                <ProtectedRoute>
                  <ChallengeSponsorPricing />
                </ProtectedRoute>
              }
            />
            <Route
              path="/challenges/:challengeId/dashboard"
              element={
                <ProtectedRoute>
                  <ChallengeSponsorDashboard />
                </ProtectedRoute>
              }
            />
            <Route path="/jobs" element={<Jobs />} />
            <Route path="/classes" element={<VibeClasses />} />
            <Route
              path="/jobs/:jobId/pricing"
              element={
                <ProtectedRoute>
                  <JobPostPricing />
                </ProtectedRoute>
              }
            />
            <Route
              path="/jobs/:jobId/analytics"
              element={
                <ProtectedRoute>
                  <JobAnalytics />
                </ProtectedRoute>
              }
            />
            <Route path="/summit" element={<Summit />} />
            <Route path="/search" element={<Search />} />
            <Route path="/pricing" element={<Pricing />} />
            <Route path="/blog" element={<Blog />} />
            <Route path="/blog/:slug" element={<BlogPost />} />
            <Route path="/marketplace" element={<Marketplace />} />
            <Route path="/marketplace/:id" element={<ProductDetail />} />
            <Route
              path="/cart"
              element={
                <ProtectedRoute>
                  <Cart />
                </ProtectedRoute>
              }
            />
            <Route
              path="/checkout"
              element={
                <ProtectedRoute>
                  <Checkout />
                </ProtectedRoute>
              }
            />
            <Route
              path="/checkout/success"
              element={
                <ProtectedRoute>
                  <CheckoutSuccess />
                </ProtectedRoute>
              }
            />
            <Route
              path="/subscription/success"
              element={
                <ProtectedRoute>
                  <SubscriptionSuccess />
                </ProtectedRoute>
              }
            />
            <Route
              path="/messages"
              element={
                <ProtectedRoute>
                  <Messages />
                </ProtectedRoute>
              }
            />
            <Route
              path="/notifications"
              element={
                <ProtectedRoute>
                  <Notifications />
                </ProtectedRoute>
              }
            />
            <Route
              path="/dashboard"
              element={
                <ProtectedRoute>
                  <Dashboard />
                </ProtectedRoute>
              }
            />
            <Route path="/referrals" element={<Referrals />} />
            <Route
              path="/referrals/dashboard"
              element={
                <ProtectedRoute>
                  <ReferralDashboard />
                </ProtectedRoute>
              }
            />
            <Route
              path="/creator/dashboard"
              element={
                <ProtectedRoute>
                  <CreatorDashboard />
                </ProtectedRoute>
              }
            />
            <Route
              path="/profile/customize"
              element={
                <ProtectedRoute>
                  <CustomProfile />
                </ProtectedRoute>
              }
            />
            <Route
              path="/analytics"
              element={
                <ProtectedRoute>
                  <PlatformAnalytics />
                </ProtectedRoute>
              }
            />
            <Route
              path="/admin"
              element={
                <ProtectedRoute>
                  <AdminDashboard />
                </ProtectedRoute>
              }
            />
            <Route path="/privacy" element={<Privacy />} />
            <Route path="/terms" element={<Terms />} />
            <Route path="/cookies" element={<CookiePolicy />} />
            <Route path="/dmca" element={<DMCA />} />
            <Route path="/refund-policy" element={<RefundPolicy />} />
          </Routes>
        </main>
        <Footer />
      </div>
  );
}

function App() {
  return (
    <Router>
      <AuthProvider>
        <SocketProvider>
          <AppContent />
        </SocketProvider>
      </AuthProvider>
    </Router>
  );
}

export default App;
