import { useState } from 'react';
import { FaTools, FaEye, FaBrain, FaLaugh, FaCopy, FaShare, FaCheckCircle, FaPlay, FaInfoCircle, FaRobot } from 'react-icons/fa';

export default function ModelCard({ 
  model, 
  viewMode = 'grid',
  isSelected = false,
  onToggleSelect,
  onOpenDetail,
  onTestModel 
}) {
  const [rating, setRating] = useState(model.rating || 0);
  const [copied, setCopied] = useState(false);
  const [imageError, setImageError] = useState(false);

  const stars = [1, 2, 3, 4, 5];

  const capabilityIcons = {
    tools: <FaTools className="text-blue-400" />,
    vision: <FaEye className="text-green-400" />,
    reasoning: <FaBrain className="text-purple-400" />,
    humor: <FaLaugh className="text-yellow-400" />,
  };

  const handleCopyLink = async () => {
    const shareUrl = model.socialLinks?.share || window.location.href;
    await navigator.clipboard.writeText(shareUrl);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  const handleImageError = () => {
    setImageError(true);
  };

  // Generate consistent color from model creator name
  const getColorFromString = (str) => {
    const colors = [
      { from: 'turquoise', to: 'cyan-500', icon: 'turquoise' },
      { from: 'purple', to: 'pink-500', icon: 'purple-400' },
      { from: 'blue-500', to: 'indigo-500', icon: 'blue-400' },
      { from: 'green-500', to: 'emerald-500', icon: 'green-400' },
      { from: 'orange-500', to: 'red-500', icon: 'orange-400' },
      { from: 'yellow-500', to: 'amber-500', icon: 'yellow-400' },
      { from: 'pink-500', to: 'rose-500', icon: 'pink-400' },
      { from: 'indigo-500', to: 'violet-500', icon: 'indigo-400' },
      { from: 'teal-500', to: 'cyan-400', icon: 'teal-400' },
      { from: 'red-500', to: 'orange-600', icon: 'red-400' },
    ];
    
    // Simple hash function
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      hash = ((hash << 5) - hash) + str.charCodeAt(i);
      hash = hash & hash; // Convert to 32bit integer
    }
    
    return colors[Math.abs(hash) % colors.length];
  };

  const modelColor = getColorFromString(model.creator || model.model || 'default');

  // Fallback icon component
  const CompanyIcon = () => {
    if (imageError || !model.companyIcon) {
      return (
        <div className={`w-14 h-14 rounded-md bg-gradient-to-br from-${modelColor.from}/20 to-${modelColor.to}/20 flex items-center justify-center`}>
          <FaRobot className={`text-2xl text-${modelColor.icon}`} />
        </div>
      );
    }
    return (
      <img 
        src={model.companyIcon} 
        alt={model.creator} 
        className="w-14 h-14 rounded-md object-cover shadow-inner bg-gray-800"
        onError={handleImageError}
      />
    );
  };

  // Fallback icon for list view
  const CompanyIconList = () => {
    if (imageError || !model.companyIcon) {
      return (
        <div className={`w-16 h-16 rounded-md bg-gradient-to-br from-${modelColor.from}/20 to-${modelColor.to}/20 flex items-center justify-center`}>
          <FaRobot className={`text-3xl text-${modelColor.icon}`} />
        </div>
      );
    }
    return (
      <img 
        src={model.companyIcon} 
        alt={model.creator} 
        className="w-16 h-16 rounded-md object-cover shadow-inner bg-gray-800"
        onError={handleImageError}
      />
    );
  };

  // List view rendering
  if (viewMode === 'list') {
    return (
      <div className={`card glass-effect flex items-center gap-6 p-6 relative ${isSelected ? 'border-2 border-turquoise' : ''}`}>
        {onToggleSelect && (
          <button
            onClick={() => onToggleSelect(model)}
            className={`absolute top-4 right-4 ${isSelected ? 'text-turquoise' : 'text-gray-600'}`}
          >
            <FaCheckCircle size={24} />
          </button>
        )}
        
        <CompanyIconList />
        
        <div className="flex-1">
          <h3 className="text-xl font-semibold text-white">{model.model}</h3>
          <div className="text-sm text-gray-400">{model.creator}</div>
        </div>

        <div className="flex items-center gap-6">
          <div className="text-center">
            <div className="text-sm text-gray-400">Parameters</div>
            <div className="text-lg font-bold text-turquoise">{model.paramSize || '—'}</div>
          </div>
          <div className="text-center">
            <div className="text-sm text-gray-400">Context</div>
            <div className="text-lg font-bold text-turquoise">{model.context || '—'}</div>
          </div>
          <div className="flex items-center gap-1">
            {'★'.repeat(model.rating || 0)}
            <span className="text-gray-600">{'★'.repeat(5 - (model.rating || 0))}</span>
          </div>
        </div>

        <div className="flex items-center gap-2">
          {onOpenDetail && (
            <button
              onClick={() => onOpenDetail(model)}
              className="px-4 py-2 bg-gray-800 hover:bg-gray-700 rounded-lg flex items-center gap-2"
            >
              <FaInfoCircle /> Details
            </button>
          )}
          {onTestModel && (
            <button
              onClick={() => onTestModel(model)}
              className="btn-primary px-4 py-2 flex items-center gap-2"
            >
              <FaPlay /> Test
            </button>
          )}
        </div>
      </div>
    );
  }

  // Grid view rendering (default)
  return (
    <div className={`card glass-effect relative overflow-hidden border-2 hover:scale-105 transition-transform ${isSelected ? 'border-turquoise' : ''}`} style={{ borderImage: 'linear-gradient(90deg,#06b6d4,#f6c84c) 1' }}>
      <div className="absolute -top-4 -right-12 rotate-45 bg-gradient-to-r from-turquoise to-gold text-vibe-black px-6 py-1 text-sm font-bold shadow-md">PREMIUM</div>

      {onToggleSelect && (
        <button
          onClick={() => onToggleSelect(model)}
          className={`absolute top-4 right-4 ${isSelected ? 'text-turquoise' : 'text-gray-600'}`}
        >
          <FaCheckCircle size={20} />
        </button>
      )}

      <div className="flex items-center gap-4">
        <CompanyIcon />
        <div className="flex-1">
          <h3 className="text-xl font-semibold text-white leading-tight">{model.model}</h3>
          <div className="text-sm text-gray-400">{model.creator} • {model.paramSize || '—'}</div>
        </div>
        <div className="text-right">
          <div className="text-sm text-gray-300">{model.context}</div>
          <div className="text-xs text-gray-500">Max {model.maxOutput}</div>
        </div>
      </div>

      <div className="mt-4 flex flex-wrap gap-2">
        {(model.capabilities || []).map((c) => (
          <span key={c} className="flex items-center gap-1 text-xs px-2 py-1 rounded-full bg-white/6 text-gray-200">
            {capabilityIcons[c] || <FaTools className="text-gray-400" />}
            {c.charAt(0).toUpperCase() + c.slice(1)}
          </span>
        ))}
      </div>

      {model.benchmarks && (
        <div className="mt-4">
          <h4 className="text-sm font-semibold text-gray-300 mb-2">Benchmarks</h4>
          <div className="grid grid-cols-2 gap-2">
            {Object.entries(model.benchmarks).map(([key, value]) => (
              <div key={key} className="flex justify-between text-xs">
                <span className="text-gray-400">{key}:</span>
                <span className="text-white font-mono">{value}%</span>
              </div>
            ))}
          </div>
        </div>
      )}

      <div className="mt-4 flex items-center justify-between">
        <div className="flex items-center gap-2">
          {stars.map((s) => (
            <button
              key={s}
              aria-label={`Rate ${s}`}
              onClick={() => setRating(s)}
              className={`text-sm ${s <= rating ? 'text-gold' : 'text-gray-600'} hover:text-gold`}
            >
              ★
            </button>
          ))}
          <span className="text-xs text-gray-400 ml-2">{rating ? `${rating}/5` : 'Unrated'}</span>
        </div>

        <div className="flex items-center gap-2">
          {model.socialLinks?.twitter && (
            <a href={model.socialLinks.twitter} target="_blank" rel="noreferrer" className="text-turquoise hover:underline text-sm">
              <FaShare className="inline mr-1" />Twitter
            </a>
          )}
          <button onClick={handleCopyLink} className="text-gray-300 hover:text-white text-sm flex items-center gap-1">
            <FaCopy className="inline" />
            {copied ? 'Copied!' : 'Copy'}
          </button>
        </div>
      </div>

      {/* Action Buttons */}
      {(onOpenDetail || onTestModel) && (
        <div className="mt-4 flex gap-2">
          {onOpenDetail && (
            <button
              onClick={() => onOpenDetail(model)}
              className="flex-1 px-3 py-2 bg-gray-800 hover:bg-gray-700 rounded-lg flex items-center justify-center gap-2 text-sm"
            >
              <FaInfoCircle /> Details
            </button>
          )}
          {onTestModel && (
            <button
              onClick={() => onTestModel(model)}
              className="flex-1 btn-primary px-3 py-2 flex items-center justify-center gap-2 text-sm"
            >
              <FaPlay /> Test
            </button>
          )}
        </div>
      )}

      <div className="mt-4 text-sm text-gray-400">Cost multiplier: {model.costMultiplier ?? '—'}</div>
    </div>
  );
}
