import { useEffect, useRef, useState } from 'react';
import { Link, useNavigate, useLocation } from 'react-router-dom';
import { FaBars, FaTimes, FaUser, FaSignOutAlt, FaSearch, FaComments, FaChevronDown, FaTrophy, FaBriefcase, FaShoppingCart, FaCalendarAlt, FaDollarSign, FaLightbulb, FaCode, FaRobot, FaBlog, FaCrown, FaCog, FaChartLine } from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import NotificationBell from './NotificationBell';

const Navbar = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [activeDropdown, setActiveDropdown] = useState(null);
  const [showUserMenu, setShowUserMenu] = useState(false);
  const [mobileExpandedSections, setMobileExpandedSections] = useState({});
  const [searchExpanded, setSearchExpanded] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const { user, logout, isAuthenticated } = useAuth();
  const navigate = useNavigate();
  const location = useLocation();

  const desktopDropdownRef = useRef(null);
  const userMenuRef = useRef(null);
  const searchRef = useRef(null);

  useEffect(() => {
    setIsOpen(false);
    setActiveDropdown(null);
    setShowUserMenu(false);
    setSearchExpanded(false);
  }, [location.pathname]);

  useEffect(() => {
    const onPointerDown = (event) => {
      const target = event.target;
      if (!(target instanceof Node)) return;

      const clickedInDropdown = desktopDropdownRef.current?.contains(target);
      const clickedInUserMenu = userMenuRef.current?.contains(target);
      const clickedInSearch = searchRef.current?.contains(target);

      if (!clickedInDropdown) setActiveDropdown(null);
      if (!clickedInUserMenu) setShowUserMenu(false);
      if (!clickedInSearch) setSearchExpanded(false);
    };

    const onKeyDown = (event) => {
      if (event.key !== 'Escape') return;
      setActiveDropdown(null);
      setShowUserMenu(false);
      setSearchExpanded(false);
    };

    document.addEventListener('mousedown', onPointerDown);
    document.addEventListener('touchstart', onPointerDown);
    document.addEventListener('keydown', onKeyDown);
    return () => {
      document.removeEventListener('mousedown', onPointerDown);
      document.removeEventListener('touchstart', onPointerDown);
      document.removeEventListener('keydown', onKeyDown);
    };
  }, []);

  const handleLogout = () => {
    logout();
    navigate('/');
  };

  const handleSearchSubmit = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      navigate(`/search?q=${encodeURIComponent(searchQuery.trim())}`);
      setSearchExpanded(false);
      setSearchQuery('');
    }
  };

  const toggleMobileSection = (section) => {
    setMobileExpandedSections(prev => ({
      ...prev,
      [section]: !prev[section]
    }));
  };

  const isActivePath = (path) => location.pathname === path;

  const navGroups = {
    discover: {
      label: 'Discover',
      items: [
        { to: '/community', label: 'Community', icon: <FaLightbulb /> },
        { to: '/models', label: 'AI Models', icon: <FaRobot /> },
        { to: '/workflows', label: 'Workflows', icon: <FaCode /> },
        { to: '/blog', label: 'Blog', icon: <FaBlog /> },
      ]
    },
    compete: {
      label: 'Compete & Earn',
      items: [
        { to: '/challenges', label: 'Challenges', icon: <FaTrophy /> },
        { to: '/jobs', label: 'Jobs', icon: <FaBriefcase /> },
        { to: '/marketplace', label: 'Marketplace', icon: <FaShoppingCart /> },
      ]
    }
  };

  const standaloneLinks = [
    { to: '/summit', label: 'Summit', icon: <FaCalendarAlt /> },
    { to: '/classes', label: 'Classes', icon: <FaCrown /> },
    { to: '/pricing', label: 'Pricing', icon: <FaDollarSign /> },
  ];

  return (
    <nav className="bg-vibe-black/95 backdrop-blur-md fixed w-full z-50 border-b border-gray-800">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-20">
          {/* Logo */}
          <Link to="/" className="flex items-center space-x-2">
            <span className="text-2xl font-bold text-gradient whitespace-nowrap">Vibe Coding Bros©</span>
          </Link>

          {/* Desktop Navigation */}
          <div ref={desktopDropdownRef} className="hidden md:flex items-center space-x-6 md:ml-8 lg:ml-12">
            {/* Discover Dropdown */}
            <div ref={searchRef} className="relative">
              <button
                type="button"
                aria-haspopup="menu"
                aria-expanded={activeDropdown === 'discover'}
                onClick={() => setActiveDropdown(activeDropdown === 'discover' ? null : 'discover')}
                className="flex items-center space-x-1 text-gray-300 hover:text-turquoise transition-colors duration-200"
              >
                <span>Discover</span>
                <FaChevronDown size={12} />
              </button>
              {activeDropdown === 'discover' && (
                <div className="absolute top-full left-0 mt-2 w-56 bg-gray-900 border border-gray-800 rounded-lg shadow-xl py-2 z-50">
                  {navGroups.discover.items.map((link) => (
                    <Link
                      key={link.to}
                      to={link.to}
                      className={`flex items-center space-x-3 px-4 py-3 transition-colors duration-200 ${
                        isActivePath(link.to) 
                          ? 'bg-turquoise/10 text-turquoise border-l-2 border-turquoise' 
                          : 'text-gray-300 hover:bg-gray-800 hover:text-turquoise'
                      }`}
                      onClick={() => setActiveDropdown(null)}
                    >
                      <span className="text-lg">{link.icon}</span>
                      <span>{link.label}</span>
                    </Link>
                  ))}
                </div>
              )}
            </div>

            {/* Compete & Earn Dropdown */}
            <div className="relative">
              <button
                type="button"
                aria-haspopup="menu"
                aria-expanded={activeDropdown === 'compete'}
                onClick={() => setActiveDropdown(activeDropdown === 'compete' ? null : 'compete')}
                className="flex items-center space-x-1 text-gray-300 hover:text-turquoise transition-colors duration-200"
              >
                <span>Compete & Earn</span>
                <FaChevronDown size={12} />
              </button>
              {activeDropdown === 'compete' && (
                <div className="absolute top-full left-0 mt-2 w-56 bg-gray-900 border border-gray-800 rounded-lg shadow-xl py-2 z-50">
                  {navGroups.compete.items.map((link) => (
                    <Link
                      key={link.to}
                      to={link.to}
                      className={`flex items-center space-x-3 px-4 py-3 transition-colors duration-200 ${
                        isActivePath(link.to) 
                          ? 'bg-turquoise/10 text-turquoise border-l-2 border-turquoise' 
                          : 'text-gray-300 hover:bg-gray-800 hover:text-turquoise'
                      }`}
                      onClick={() => setActiveDropdown(null)}
                    >
                      <span className="text-lg">{link.icon}</span>
                      <span>{link.label}</span>
                    </Link>
                  ))}
                </div>
              )}
            </div>

            {/* Standalone Links */}
            {standaloneLinks.map((link) => (
              <Link
                key={link.to}
                to={link.to}
                className={`transition-colors duration-200 relative ${
                  isActivePath(link.to)
                    ? 'text-turquoise after:absolute after:bottom-[-8px] after:left-0 after:w-full after:h-0.5 after:bg-turquoise'
                    : 'text-gray-300 hover:text-turquoise'
                }`}
              >
                {link.label}
              </Link>
            ))}
            
            {/* Enhanced Search */}
            <div className="relative">
              {searchExpanded ? (
                <form onSubmit={handleSearchSubmit} className="flex items-center">
                  <input
                    type="text"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    placeholder="Search..."
                    className="w-48 lg:w-64 px-4 py-1.5 bg-gray-800 border border-gray-700 rounded-l-lg text-gray-300 placeholder-gray-500 focus:outline-none focus:border-turquoise transition-colors"
                    autoFocus
                    onBlur={() => {
                      if (!searchQuery) {
                        setTimeout(() => setSearchExpanded(false), 200);
                      }
                    }}
                  />
                  <button
                    type="submit"
                    className="px-3 py-1.5 bg-turquoise text-black rounded-r-lg hover:bg-turquoise/80 transition-colors"
                  >
                    <FaSearch size={16} />
                  </button>
                  <button
                    type="button"
                    onClick={() => {
                      setSearchExpanded(false);
                      setSearchQuery('');
                    }}
                    className="ml-2 text-gray-400 hover:text-gray-300"
                  >
                    <FaTimes size={16} />
                  </button>
                </form>
              ) : (
                <button
                  onClick={() => setSearchExpanded(true)}
                  className={`transition-colors duration-200 ${
                    isActivePath('/search') ? 'text-turquoise' : 'text-gray-300 hover:text-turquoise'
                  }`}
                >
                  <FaSearch size={18} />
                </button>
              )}
            </div>
            
            {isAuthenticated && (
              <>
                {/* Activity Hub - Combined Notifications & Messages */}
                <div className="flex items-center space-x-3 pl-3 border-l border-gray-700">
                  <NotificationBell />
                  <Link 
                    to="/messages" 
                    className={`relative transition-colors duration-200 ${
                      isActivePath('/messages') ? 'text-turquoise' : 'text-gray-300 hover:text-turquoise'
                    }`}
                  >
                    <FaComments size={18} />
                    {/* You can add unread count badge here similar to NotificationBell */}
                  </Link>
                </div>
              </>
            )}
            
            {/* Referral CTA */}
            <Link
              to="/referrals"
              className="hidden lg:flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-turquoise to-gold text-vibe-black font-semibold rounded-lg hover:opacity-90 transition-opacity"
            >
              <FaDollarSign size={16} />
              Earn Rewards
            </Link>

            {isAuthenticated ? (
              <div ref={userMenuRef} className="relative">
                <button
                  type="button"
                  aria-haspopup="menu"
                  aria-expanded={showUserMenu}
                  onClick={() => setShowUserMenu((prev) => !prev)}
                  className="flex items-center space-x-2 text-gray-300 hover:text-turquoise transition-colors"
                >
                  <div className="w-8 h-8 rounded-full bg-gradient-to-br from-turquoise to-gold flex items-center justify-center text-black font-bold">
                    {user?.username?.[0]?.toUpperCase()}
                  </div>
                  <span className="hidden lg:inline">{user?.username}</span>
                  {user?.subscriptionTier && user.subscriptionTier !== 'free' && (
                    <FaCrown className={user.subscriptionTier === 'creator' ? 'text-gold' : 'text-turquoise'} size={14} />
                  )}
                  <FaChevronDown size={12} />
                </button>
                
                {showUserMenu && (
                  <div className="absolute top-full right-0 mt-2 w-64 bg-gray-900 border border-gray-800 rounded-lg shadow-xl py-2 z-50">
                    {/* User Info Section */}
                    <div className="px-4 py-3 border-b border-gray-800">
                      <p className="text-sm text-gray-400">Signed in as</p>
                      <p className="font-semibold text-turquoise">{user?.email}</p>
                      {user?.subscriptionTier && (
                        <span className={`inline-block mt-1 text-xs px-2 py-1 rounded ${
                          user.subscriptionTier === 'creator' ? 'bg-gold/20 text-gold' :
                          user.subscriptionTier === 'pro' ? 'bg-turquoise/20 text-turquoise' :
                          'bg-gray-700 text-gray-300'
                        }`}>
                          {user.subscriptionTier.toUpperCase()} TIER
                        </span>
                      )}
                    </div>

                    {/* Main Navigation */}
                    <div className="py-2">
                      <Link
                        to="/dashboard"
                        className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                          isActivePath('/dashboard') 
                            ? 'bg-turquoise/10 text-turquoise' 
                            : 'text-gray-300 hover:bg-gray-800 hover:text-turquoise'
                        }`}
                        onClick={() => setShowUserMenu(false)}
                      >
                        <FaUser />
                        <span>Dashboard</span>
                      </Link>

                      <Link
                        to="/referrals/dashboard"
                        className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                          isActivePath('/referrals/dashboard') 
                            ? 'bg-turquoise/10 text-turquoise' 
                            : 'text-gray-300 hover:bg-gray-800 hover:text-turquoise'
                        }`}
                        onClick={() => setShowUserMenu(false)}
                      >
                        <FaUsers />
                        <span>Referrals</span>
                      </Link>
                      
                      {(user?.subscriptionTier === 'pro' || user?.subscriptionTier === 'creator') && (
                        <Link
                          to="/profile/customize"
                          className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                            isActivePath('/profile/customize') 
                              ? 'bg-turquoise/10 text-turquoise' 
                              : 'text-gray-300 hover:bg-gray-800 hover:text-turquoise'
                          }`}
                          onClick={() => setShowUserMenu(false)}
                        >
                          <FaCog />
                          <span>Customize Profile</span>
                        </Link>
                      )}
                    </div>

                    {/* Creator/Admin Section */}
                    {(user?.subscriptionTier === 'creator' || user?.isAdmin) && (
                      <>
                        <div className="border-t border-gray-800 my-2"></div>
                        <div className="py-2">
                          {user?.subscriptionTier === 'creator' && (
                            <>
                              <Link
                                to="/creator/dashboard"
                                className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                                  isActivePath('/creator/dashboard') 
                                    ? 'bg-gold/10 text-gold' 
                                    : 'text-gray-300 hover:bg-gray-800 hover:text-gold'
                                }`}
                                onClick={() => setShowUserMenu(false)}
                              >
                                <FaCrown />
                                <span>Creator Dashboard</span>
                              </Link>
                              <Link
                                to="/analytics"
                                className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                                  isActivePath('/analytics') 
                                    ? 'bg-gold/10 text-gold' 
                                    : 'text-gray-300 hover:bg-gray-800 hover:text-gold'
                                }`}
                                onClick={() => setShowUserMenu(false)}
                              >
                                <FaChartLine />
                                <span>Analytics</span>
                              </Link>
                            </>
                          )}
                          
                          {user?.isAdmin && (
                            <Link
                              to="/admin"
                              className={`flex items-center space-x-3 px-4 py-2 transition-colors ${
                                isActivePath('/admin') 
                                  ? 'bg-red-500/10 text-red-500' 
                                  : 'text-gray-300 hover:bg-gray-800 hover:text-red-500'
                              }`}
                              onClick={() => setShowUserMenu(false)}
                            >
                              <FaCog />
                              <span>Admin Dashboard</span>
                            </Link>
                          )}
                        </div>
                      </>
                    )}

                    {/* Logout */}
                    <div className="border-t border-gray-800 mt-2 pt-2">
                      <button
                        onClick={() => {
                          setShowUserMenu(false);
                          handleLogout();
                        }}
                        className="flex items-center space-x-3 px-4 py-2 w-full text-left text-gray-300 hover:bg-red-500/10 hover:text-red-500 transition-colors"
                      >
                        <FaSignOutAlt />
                        <span>Logout</span>
                      </button>
                    </div>
                  </div>
                )}
              </div>
            ) : (
              <div className="flex items-center space-x-4">
                <Link to="/signin" className="text-gray-300 hover:text-turquoise">
                  Sign In
                </Link>
                <Link to="/signup" className="btn-primary">
                  Sign Up
                </Link>
              </div>
            )}
          </div>

          {/* Mobile menu button */}
          <button
            onClick={() => setIsOpen(!isOpen)}
            className="md:hidden text-gray-300 hover:text-turquoise"
          >
            {isOpen ? <FaTimes size={24} /> : <FaBars size={24} />}
          </button>
        </div>
      </div>

      {/* Mobile Navigation */}
      {isOpen && (
        <div className="md:hidden bg-gray-900 border-t border-gray-800">
          <div className="px-4 pt-2 pb-4 space-y-1">
            {/* Discover Section */}
            <div>
              <button
                onClick={() => toggleMobileSection('discover')}
                className="flex items-center justify-between w-full py-3 text-gray-300 hover:text-turquoise transition-colors"
              >
                <span className="font-semibold">Discover</span>
                <FaChevronDown 
                  className={`transform transition-transform ${mobileExpandedSections.discover ? 'rotate-180' : ''}`} 
                  size={14} 
                />
              </button>
              {mobileExpandedSections.discover && (
                <div className="pl-4 space-y-1 mb-2">
                  {navGroups.discover.items.map((link) => (
                    <Link
                      key={link.to}
                      to={link.to}
                      className={`flex items-center space-x-3 py-2 transition-colors ${
                        isActivePath(link.to) 
                          ? 'text-turquoise font-semibold' 
                          : 'text-gray-400 hover:text-turquoise'
                      }`}
                      onClick={() => setIsOpen(false)}
                    >
                      <span>{link.icon}</span>
                      <span>{link.label}</span>
                    </Link>
                  ))}
                </div>
              )}
            </div>

            {/* Compete & Earn Section */}
            <div>
              <button
                onClick={() => toggleMobileSection('compete')}
                className="flex items-center justify-between w-full py-3 text-gray-300 hover:text-turquoise transition-colors"
              >
                <span className="font-semibold">Compete & Earn</span>
                <FaChevronDown 
                  className={`transform transition-transform ${mobileExpandedSections.compete ? 'rotate-180' : ''}`} 
                  size={14} 
                />
              </button>
              {mobileExpandedSections.compete && (
                <div className="pl-4 space-y-1 mb-2">
                  {navGroups.compete.items.map((link) => (
                    <Link
                      key={link.to}
                      to={link.to}
                      className={`flex items-center space-x-3 py-2 transition-colors ${
                        isActivePath(link.to) 
                          ? 'text-turquoise font-semibold' 
                          : 'text-gray-400 hover:text-turquoise'
                      }`}
                      onClick={() => setIsOpen(false)}
                    >
                      <span>{link.icon}</span>
                      <span>{link.label}</span>
                    </Link>
                  ))}
                </div>
              )}
            </div>

            <div className="border-t border-gray-800 my-2"></div>

            {/* Standalone Links */}
            {standaloneLinks.map((link) => (
              <Link
                key={link.to}
                to={link.to}
                className={`flex items-center space-x-3 py-3 transition-colors ${
                  isActivePath(link.to) 
                    ? 'text-turquoise font-semibold' 
                    : 'text-gray-300 hover:text-turquoise'
                }`}
                onClick={() => setIsOpen(false)}
              >
                <span>{link.icon}</span>
                <span>{link.label}</span>
              </Link>
            ))}
            
            {/* Mobile Search */}
            <Link
              to="/search"
              className={`flex items-center space-x-3 py-3 transition-colors ${
                isActivePath('/search') 
                  ? 'text-turquoise font-semibold' 
                  : 'text-gray-300 hover:text-turquoise'
              }`}
              onClick={() => setIsOpen(false)}
            >
              <FaSearch />
              <span>Search</span>
            </Link>
            
            {isAuthenticated && (
              <Link
                to="/messages"
                className={`flex items-center space-x-3 py-3 transition-colors ${
                  isActivePath('/messages') 
                    ? 'text-turquoise font-semibold' 
                    : 'text-gray-300 hover:text-turquoise'
                }`}
                onClick={() => setIsOpen(false)}
              >
                <FaComments />
                <span>Messages</span>
              </Link>
            )}

            {isAuthenticated && (
              <>
                <div className="border-t border-gray-800 my-2"></div>
                
                {/* User Section */}
                <div>
                  <button
                    onClick={() => toggleMobileSection('account')}
                    className="flex items-center justify-between w-full py-3 text-gray-300 hover:text-turquoise transition-colors"
                  >
                    <div className="flex items-center space-x-2">
                      <div className="w-8 h-8 rounded-full bg-gradient-to-br from-turquoise to-gold flex items-center justify-center text-black font-bold text-sm">
                        {user?.username?.[0]?.toUpperCase()}
                      </div>
                      <span className="font-semibold">{user?.username}</span>
                      {user?.subscriptionTier && user.subscriptionTier !== 'free' && (
                        <FaCrown className={user.subscriptionTier === 'creator' ? 'text-gold' : 'text-turquoise'} size={12} />
                      )}
                    </div>
                    <FaChevronDown 
                      className={`transform transition-transform ${mobileExpandedSections.account ? 'rotate-180' : ''}`} 
                      size={14} 
                    />
                  </button>
                  
                  {mobileExpandedSections.account && (
                    <div className="pl-4 space-y-1 mb-2">
                      {user?.subscriptionTier && (
                        <div className="py-2">
                          <span className={`inline-block text-xs px-2 py-1 rounded ${
                            user.subscriptionTier === 'creator' ? 'bg-gold/20 text-gold' :
                            user.subscriptionTier === 'pro' ? 'bg-turquoise/20 text-turquoise' :
                            'bg-gray-700 text-gray-300'
                          }`}>
                            {user.subscriptionTier.toUpperCase()} TIER
                          </span>
                        </div>
                      )}
                      
                      <Link
                        to="/dashboard"
                        className={`flex items-center space-x-3 py-2 transition-colors ${
                          isActivePath('/dashboard') 
                            ? 'text-turquoise font-semibold' 
                            : 'text-gray-400 hover:text-turquoise'
                        }`}
                        onClick={() => setIsOpen(false)}
                      >
                        <FaUser />
                        <span>Dashboard</span>
                      </Link>
                      
                      {(user?.subscriptionTier === 'pro' || user?.subscriptionTier === 'creator') && (
                        <Link
                          to="/profile/customize"
                          className={`flex items-center space-x-3 py-2 transition-colors ${
                            isActivePath('/profile/customize') 
                              ? 'text-turquoise font-semibold' 
                              : 'text-gray-400 hover:text-turquoise'
                          }`}
                          onClick={() => setIsOpen(false)}
                        >
                          <FaCog />
                          <span>Customize Profile</span>
                        </Link>
                      )}

                      {user?.subscriptionTier === 'creator' && (
                        <>
                          <div className="border-t border-gray-700 my-2"></div>
                          <Link
                            to="/creator/dashboard"
                            className={`flex items-center space-x-3 py-2 transition-colors ${
                              isActivePath('/creator/dashboard') 
                                ? 'text-gold font-semibold' 
                                : 'text-gray-400 hover:text-gold'
                            }`}
                            onClick={() => setIsOpen(false)}
                          >
                            <FaCrown />
                            <span>Creator Dashboard</span>
                          </Link>
                          <Link
                            to="/analytics"
                            className={`flex items-center space-x-3 py-2 transition-colors ${
                              isActivePath('/analytics') 
                                ? 'text-gold font-semibold' 
                                : 'text-gray-400 hover:text-gold'
                            }`}
                            onClick={() => setIsOpen(false)}
                          >
                            <FaChartLine />
                            <span>Analytics</span>
                          </Link>
                        </>
                      )}

                      {user?.isAdmin && (
                        <>
                          <div className="border-t border-gray-700 my-2"></div>
                          <Link
                            to="/admin"
                            className={`flex items-center space-x-3 py-2 transition-colors ${
                              isActivePath('/admin') 
                                ? 'text-red-500 font-semibold' 
                                : 'text-gray-400 hover:text-red-500'
                            }`}
                            onClick={() => setIsOpen(false)}
                          >
                            <FaCog />
                            <span>Admin Dashboard</span>
                          </Link>
                        </>
                      )}

                      <div className="border-t border-gray-700 my-2"></div>
                      <button
                        onClick={() => {
                          handleLogout();
                          setIsOpen(false);
                        }}
                        className="flex items-center space-x-3 py-2 w-full text-left text-gray-400 hover:text-red-500 transition-colors"
                      >
                        <FaSignOutAlt />
                        <span>Logout</span>
                      </button>
                    </div>
                  )}
                </div>
              </>
            )}

            {!isAuthenticated && (
              <>
                <div className="border-t border-gray-800 my-2"></div>
                <Link
                  to="/signin"
                  className="block py-3 text-gray-300 hover:text-turquoise transition-colors"
                  onClick={() => setIsOpen(false)}
                >
                  Sign In
                </Link>
                <Link
                  to="/signup"
                  className="block py-3 text-center bg-gradient-to-r from-turquoise to-gold text-black font-semibold rounded-lg"
                  onClick={() => setIsOpen(false)}
                >
                  Sign Up
                </Link>
              </>
            )}
          </div>
        </div>
      )}
    </nav>
  );
};

export default Navbar;
