import React, { useState, useEffect } from 'react';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';
import { FaCrown, FaLock, FaCheck, FaTimes } from 'react-icons/fa';

const PremiumFeatureGate = ({ feature, featureName, children, fallback }) => {
  const { user } = useAuth();
  const [hasAccess, setHasAccess] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    checkAccess();
  }, [user, feature]);

  const checkAccess = async () => {
    try {
      const response = await api.get('/premium/features');
      const featureAccess = response.data.features;
      
      // Navigate through nested feature object
      const keys = feature.split('.');
      let access = featureAccess;
      
      for (const key of keys) {
        if (access && typeof access === 'object') {
          access = access[key];
        } else {
          access = false;
          break;
        }
      }
      
      setHasAccess(!!access);
    } catch (error) {
      console.error('Failed to check feature access:', error);
      setHasAccess(false);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return <div className="animate-pulse bg-gray-800 rounded h-10"></div>;
  }

  if (hasAccess) {
    return <>{children}</>;
  }

  return (
    fallback || (
      <div className="card p-6 text-center bg-gradient-to-br from-gray-800 to-gray-900 border border-gray-700">
        <div className="flex justify-center mb-4">
          <div className="relative">
            <FaLock className="text-5xl text-gray-600" />
            <FaCrown className="text-2xl text-yellow-500 absolute -top-2 -right-2" />
          </div>
        </div>
        <h3 className="text-xl font-bold mb-2">Premium Feature</h3>
        <p className="text-gray-400 mb-4">
          {featureName || 'This feature'} is available for Pro and Creator members
        </p>
        <a href="/pricing" className="btn-primary inline-block">
          Upgrade to Unlock
        </a>
      </div>
    )
  );
};

export const FeatureComparison = () => {
  const features = [
    {
      category: 'Search',
      items: [
        { name: 'Basic Search', free: true, pro: true, creator: true },
        { name: 'Advanced Filters', free: false, pro: true, creator: true },
        { name: 'Saved Searches', free: false, pro: true, creator: true },
        { name: 'Search Alerts', free: false, pro: true, creator: true },
      ],
    },
    {
      category: 'Messaging',
      items: [
        { name: 'Basic Messaging', free: true, pro: true, creator: true },
        { name: 'Unlimited Messages', free: false, pro: true, creator: true },
        { name: 'File Attachments', free: false, pro: true, creator: true },
        { name: 'Voice Messages', free: false, pro: true, creator: true },
      ],
    },
    {
      category: 'Profile',
      items: [
        { name: 'Standard Profile', free: true, pro: true, creator: true },
        { name: 'Custom Theme', free: false, pro: true, creator: true },
        { name: 'Portfolio Showcase', free: false, pro: true, creator: true },
        { name: 'Verified Badge', free: false, pro: false, creator: true },
      ],
    },
    {
      category: 'Marketplace',
      items: [
        { name: 'Browse Products', free: true, pro: true, creator: true },
        { name: 'Purchase Products', free: true, pro: true, creator: true },
        { name: 'Sell Products', free: false, pro: false, creator: true },
        { name: 'Analytics Dashboard', free: false, pro: false, creator: true },
        { name: 'Priority Listing', free: false, pro: false, creator: true },
      ],
    },
    {
      category: 'Support',
      items: [
        { name: 'Community Support', free: true, pro: true, creator: true },
        { name: 'Priority Support', free: false, pro: true, creator: true },
        { name: 'Dedicated Manager', free: false, pro: false, creator: true },
      ],
    },
    {
      category: 'Content',
      items: [
        { name: 'Public Content', free: true, pro: true, creator: true },
        { name: 'Exclusive Access', free: false, pro: false, creator: true },
        { name: 'Early Access', free: false, pro: false, creator: true },
        { name: 'Private Workshops', free: false, pro: false, creator: true },
      ],
    },
    {
      category: 'API',
      items: [
        { name: 'API Access', free: false, pro: false, creator: true },
        { name: 'Webhook Support', free: false, pro: false, creator: true },
        { name: 'Rate Limit', free: '10/hr', pro: '100/hr', creator: '1000/hr' },
      ],
    },
  ];

  const CheckIcon = ({ value }) => {
    if (value === true) return <FaCheck className="text-green-500 mx-auto" />;
    if (value === false) return <FaTimes className="text-red-500 mx-auto" />;
    return <span className="text-sm text-gray-400">{value}</span>;
  };

  return (
    <div className="overflow-x-auto">
      <table className="w-full">
        <thead>
          <tr className="border-b border-gray-700">
            <th className="text-left py-4 px-4 font-bold">Features</th>
            <th className="text-center py-4 px-4 font-bold">Free</th>
            <th className="text-center py-4 px-4 font-bold">
              <span className="text-turquoise">Pro</span>
            </th>
            <th className="text-center py-4 px-4 font-bold">
              <span className="text-purple">Creator</span>
            </th>
          </tr>
        </thead>
        <tbody>
          {features.map((category, idx) => (
            <React.Fragment key={idx}>
              <tr className="bg-gray-800/50">
                <td colSpan="4" className="py-3 px-4 font-bold text-sm uppercase tracking-wide">
                  {category.category}
                </td>
              </tr>
              {category.items.map((item, itemIdx) => (
                <tr key={itemIdx} className="border-b border-gray-800 hover:bg-gray-800/30">
                  <td className="py-3 px-4">{item.name}</td>
                  <td className="text-center py-3 px-4">
                    <CheckIcon value={item.free} />
                  </td>
                  <td className="text-center py-3 px-4">
                    <CheckIcon value={item.pro} />
                  </td>
                  <td className="text-center py-3 px-4">
                    <CheckIcon value={item.creator} />
                  </td>
                </tr>
              ))}
            </React.Fragment>
          ))}
        </tbody>
      </table>
    </div>
  );
};

export default PremiumFeatureGate;
