import { Helmet } from 'react-helmet-async';

const StructuredData = ({ type, data }) => {
  const generateSchema = () => {
    switch (type) {
      case 'JobPosting':
        return {
          "@context": "https://schema.org",
          "@type": "JobPosting",
          "title": data.title,
          "description": data.description,
          "datePosted": data.createdAt,
          "validThrough": data.expiresAt,
          "employmentType": data.type?.toUpperCase(),
          "hiringOrganization": {
            "@type": "Organization",
            "name": data.company,
            "sameAs": data.companyWebsite,
            "logo": data.companyLogo
          },
          "jobLocation": {
            "@type": "Place",
            "address": {
              "@type": "PostalAddress",
              "addressLocality": data.location,
              "addressRegion": data.region,
              "addressCountry": data.country || "US"
            }
          },
          "baseSalary": data.salary ? {
            "@type": "MonetaryAmount",
            "currency": "USD",
            "value": {
              "@type": "QuantitativeValue",
              "value": data.salary,
              "unitText": "YEAR"
            }
          } : undefined,
          "applicantLocationRequirements": {
            "@type": "Country",
            "name": data.country || "US"
          }
        };

      case 'Course':
        return {
          "@context": "https://schema.org",
          "@type": "Course",
          "name": data.title,
          "description": data.description,
          "provider": {
            "@type": "Organization",
            "name": "Vibe Coding Bros",
            "sameAs": "https://vibecodingbros.com"
          },
          "offers": {
            "@type": "Offer",
            "category": "Paid",
            "price": data.price,
            "priceCurrency": "USD"
          },
          "hasCourseInstance": {
            "@type": "CourseInstance",
            "courseMode": "online",
            "courseWorkload": data.duration
          }
        };

      case 'Event':
        return {
          "@context": "https://schema.org",
          "@type": "Event",
          "name": data.title,
          "description": data.description,
          "startDate": data.startDate,
          "endDate": data.endDate,
          "location": data.isVirtual ? {
            "@type": "VirtualLocation",
            "url": data.eventUrl
          } : {
            "@type": "Place",
            "name": data.venue,
            "address": data.address
          },
          "organizer": {
            "@type": "Organization",
            "name": "Vibe Coding Bros",
            "url": "https://vibecodingbros.com"
          },
          "offers": data.price ? {
            "@type": "Offer",
            "price": data.price,
            "priceCurrency": "USD",
            "availability": "https://schema.org/InStock",
            "url": data.registrationUrl
          } : undefined
        };

      case 'Product':
        return {
          "@context": "https://schema.org",
          "@type": "Product",
          "name": data.title,
          "description": data.description,
          "image": data.images,
          "brand": {
            "@type": "Brand",
            "name": data.brand || "Vibe Coding Bros"
          },
          "offers": {
            "@type": "Offer",
            "price": data.price,
            "priceCurrency": "USD",
            "availability": "https://schema.org/InStock",
            "seller": {
              "@type": "Organization",
              "name": data.seller || "Vibe Coding Bros"
            }
          },
          "aggregateRating": data.rating ? {
            "@type": "AggregateRating",
            "ratingValue": data.rating.average,
            "reviewCount": data.rating.count
          } : undefined
        };

      case 'SoftwareApplication':
        return {
          "@context": "https://schema.org",
          "@type": "SoftwareApplication",
          "name": data.title,
          "description": data.description,
          "applicationCategory": "DeveloperApplication",
          "offers": {
            "@type": "Offer",
            "price": data.price || "0",
            "priceCurrency": "USD"
          },
          "operatingSystem": data.operatingSystem || "All",
          "downloadUrl": data.downloadUrl,
          "screenshot": data.screenshots,
          "aggregateRating": data.rating ? {
            "@type": "AggregateRating",
            "ratingValue": data.rating.average,
            "reviewCount": data.rating.count
          } : undefined
        };

      case 'Article':
        return {
          "@context": "https://schema.org",
          "@type": "Article",
          "headline": data.title,
          "description": data.description,
          "image": data.image,
          "datePublished": data.publishedAt,
          "dateModified": data.updatedAt,
          "author": {
            "@type": "Person",
            "name": data.author
          },
          "publisher": {
            "@type": "Organization",
            "name": "Vibe Coding Bros",
            "logo": {
              "@type": "ImageObject",
              "url": "https://vibecodingbros.com/icons/icon-512x512.png"
            }
          }
        };

      case 'BreadcrumbList':
        return {
          "@context": "https://schema.org",
          "@type": "BreadcrumbList",
          "itemListElement": data.items.map((item, index) => ({
            "@type": "ListItem",
            "position": index + 1,
            "name": item.name,
            "item": item.url
          }))
        };

      case 'FAQPage':
        return {
          "@context": "https://schema.org",
          "@type": "FAQPage",
          "mainEntity": data.questions.map(q => ({
            "@type": "Question",
            "name": q.question,
            "acceptedAnswer": {
              "@type": "Answer",
              "text": q.answer
            }
          }))
        };

      default:
        return null;
    }
  };

  const schema = generateSchema();
  
  if (!schema) return null;

  return (
    <Helmet>
      <script type="application/ld+json">
        {JSON.stringify(schema)}
      </script>
    </Helmet>
  );
};

export default StructuredData;
