import { useState, useRef, useEffect } from 'react';
import { FaPlus, FaTimes, FaPlay, FaCheckCircle, FaEdit, FaTrash } from 'react-icons/fa';

const VisualWorkflowBuilder = ({ steps = [], onChange, readOnly = false }) => {
  const [nodes, setNodes] = useState([]);
  const [connections, setConnections] = useState([]);
  const [draggingNode, setDraggingNode] = useState(null);
  const [dragOffset, setDragOffset] = useState({ x: 0, y: 0 });
  const [selectedNode, setSelectedNode] = useState(null);
  const [connecting, setConnecting] = useState(null);
  const canvasRef = useRef(null);

  // Initialize nodes from steps
  useEffect(() => {
    if (steps.length > 0 && nodes.length === 0) {
      const initialNodes = steps.map((step, index) => ({
        id: `node-${index}`,
        title: step.title,
        description: step.description || '',
        code: step.code || '',
        x: 100 + (index % 3) * 250,
        y: 100 + Math.floor(index / 3) * 180,
        status: 'pending' // pending, running, completed, error
      }));
      setNodes(initialNodes);

      // Create sequential connections
      const initialConnections = initialNodes.slice(0, -1).map((node, index) => ({
        from: node.id,
        to: initialNodes[index + 1].id
      }));
      setConnections(initialConnections);
    }
  }, [steps, nodes.length]);

  // Sync changes back to parent
  useEffect(() => {
    if (onChange && nodes.length > 0) {
      const updatedSteps = nodes.map(node => ({
        title: node.title,
        description: node.description,
        code: node.code
      }));
      onChange(updatedSteps);
    }
  // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [nodes]);

  const addNode = () => {
    const newNode = {
      id: `node-${Date.now()}`,
      title: 'New Step',
      description: '',
      code: '',
      x: 100 + (nodes.length % 3) * 250,
      y: 100 + Math.floor(nodes.length / 3) * 180,
      status: 'pending'
    };
    setNodes([...nodes, newNode]);
  };

  const updateNode = (id, updates) => {
    setNodes(nodes.map(node => node.id === id ? { ...node, ...updates } : node));
  };

  const deleteNode = (id) => {
    setNodes(nodes.filter(node => node.id !== id));
    setConnections(connections.filter(conn => conn.from !== id && conn.to !== id));
    if (selectedNode?.id === id) setSelectedNode(null);
  };

  const handleMouseDown = (e, node) => {
    if (readOnly) return;
    e.preventDefault();
    const rect = canvasRef.current.getBoundingClientRect();
    setDragOffset({
      x: e.clientX - rect.left - node.x,
      y: e.clientY - rect.top - node.y
    });
    setDraggingNode(node.id);
  };

  const handleMouseMove = (e) => {
    if (draggingNode && canvasRef.current) {
      const rect = canvasRef.current.getBoundingClientRect();
      const newX = e.clientX - rect.left - dragOffset.x;
      const newY = e.clientY - rect.top - dragOffset.y;
      updateNode(draggingNode, { x: newX, y: newY });
    }
  };

  const handleMouseUp = () => {
    setDraggingNode(null);
  };

  const startConnection = (nodeId) => {
    if (readOnly) return;
    setConnecting(nodeId);
  };

  const completeConnection = (targetId) => {
    if (connecting && connecting !== targetId) {
      const exists = connections.find(c => c.from === connecting && c.to === targetId);
      if (!exists) {
        setConnections([...connections, { from: connecting, to: targetId }]);
      }
    }
    setConnecting(null);
  };

  const deleteConnection = (from, to) => {
    setConnections(connections.filter(c => !(c.from === from && c.to === to)));
  };

  const getNodePosition = (nodeId) => {
    const node = nodes.find(n => n.id === nodeId);
    return node ? { x: node.x + 120, y: node.y + 45 } : { x: 0, y: 0 };
  };

  const statusColors = {
    pending: 'border-gray-600 bg-gray-800',
    running: 'border-blue-500 bg-blue-900/20 animate-pulse',
    completed: 'border-green-500 bg-green-900/20',
    error: 'border-red-500 bg-red-900/20'
  };

  return (
    <div className="relative">
      {/* Toolbar */}
      {!readOnly && (
        <div className="mb-4 flex items-center gap-3 p-3 bg-gray-800 rounded-lg">
          <button onClick={addNode} className="btn-primary flex items-center gap-2 text-sm">
            <FaPlus /> Add Step
          </button>
          <div className="text-xs text-gray-400">
            {nodes.length} steps • {connections.length} connections
          </div>
          <div className="text-xs text-gray-500 ml-auto">
            Drag nodes to reposition • Click connectors to link steps
          </div>
        </div>
      )}

      {/* Canvas */}
      <div
        ref={canvasRef}
        className="relative bg-gray-900 rounded-lg border-2 border-gray-700 overflow-auto"
        style={{ height: '600px', position: 'relative' }}
        onMouseMove={handleMouseMove}
        onMouseUp={handleMouseUp}
      >
        {/* SVG for connections */}
        <svg className="absolute inset-0 pointer-events-none" style={{ zIndex: 1 }}>
          {connections.map((conn, index) => {
            const fromPos = getNodePosition(conn.from);
            const toPos = getNodePosition(conn.to);
            const midX = (fromPos.x + toPos.x) / 2;
            
            return (
              <g key={index}>
                <path
                  d={`M ${fromPos.x} ${fromPos.y} Q ${midX} ${fromPos.y}, ${midX} ${(fromPos.y + toPos.y) / 2} T ${toPos.x} ${toPos.y}`}
                  stroke="#20B2AA"
                  strokeWidth="2"
                  fill="none"
                  markerEnd="url(#arrowhead)"
                />
                {!readOnly && (
                  <circle
                    cx={midX}
                    cy={(fromPos.y + toPos.y) / 2}
                    r="8"
                    fill="#ef4444"
                    className="cursor-pointer pointer-events-auto"
                    onClick={() => deleteConnection(conn.from, conn.to)}
                  />
                )}
              </g>
            );
          })}
          <defs>
            <marker
              id="arrowhead"
              markerWidth="10"
              markerHeight="10"
              refX="9"
              refY="3"
              orient="auto"
            >
              <polygon points="0 0, 10 3, 0 6" fill="#20B2AA" />
            </marker>
          </defs>
        </svg>

        {/* Nodes */}
        {nodes.map(node => (
          <div
            key={node.id}
            className={`absolute w-60 border-2 rounded-lg p-3 transition-all ${
              statusColors[node.status]
            } ${draggingNode === node.id ? 'cursor-grabbing shadow-2xl scale-105' : 'cursor-grab'} ${
              selectedNode?.id === node.id ? 'ring-2 ring-turquoise' : ''
            }`}
            style={{
              left: `${node.x}px`,
              top: `${node.y}px`,
              zIndex: draggingNode === node.id ? 10 : 2
            }}
            onMouseDown={(e) => handleMouseDown(e, node)}
            onClick={() => setSelectedNode(node)}
          >
            {/* Status Indicator */}
            <div className="absolute -top-2 -right-2 w-6 h-6 rounded-full bg-gray-900 flex items-center justify-center">
              {node.status === 'running' && <FaPlay className="text-blue-400" size={12} />}
              {node.status === 'completed' && <FaCheckCircle className="text-green-400" size={12} />}
              {node.status === 'error' && <FaTimes className="text-red-400" size={12} />}
            </div>

            {/* Node Header */}
            <div className="flex items-start justify-between mb-2">
              <input
                type="text"
                value={node.title}
                onChange={(e) => updateNode(node.id, { title: e.target.value })}
                className="flex-1 bg-transparent border-none text-white font-semibold text-sm focus:outline-none"
                readOnly={readOnly}
                onClick={(e) => e.stopPropagation()}
              />
              {!readOnly && (
                <button
                  onClick={(e) => { e.stopPropagation(); deleteNode(node.id); }}
                  className="text-gray-500 hover:text-red-400 ml-2"
                >
                  <FaTrash size={12} />
                </button>
              )}
            </div>

            {/* Node Description */}
            <textarea
              value={node.description}
              onChange={(e) => updateNode(node.id, { description: e.target.value })}
              className="w-full bg-gray-900/50 border border-gray-700 rounded px-2 py-1 text-xs text-gray-300 resize-none"
              rows="2"
              placeholder="Description..."
              readOnly={readOnly}
              onClick={(e) => e.stopPropagation()}
            />

            {/* Connection Points */}
            <div className="flex justify-between mt-2">
              <button
                className="w-4 h-4 rounded-full bg-turquoise hover:scale-125 transition-transform"
                onClick={(e) => { e.stopPropagation(); completeConnection(node.id); }}
                title="Input"
              />
              <button
                className="w-4 h-4 rounded-full bg-gold hover:scale-125 transition-transform"
                onClick={(e) => { e.stopPropagation(); startConnection(node.id); }}
                title="Output"
              />
            </div>
          </div>
        ))}

        {/* Empty State */}
        {nodes.length === 0 && (
          <div className="absolute inset-0 flex items-center justify-center">
            <div className="text-center text-gray-500">
              <FaPlus size={48} className="mx-auto mb-4 opacity-20" />
              <p>No workflow steps yet</p>
              {!readOnly && <p className="text-sm mt-2">Click &quot;Add Step&quot; to begin</p>}
            </div>
          </div>
        )}
      </div>

      {/* Node Editor Panel */}
      {selectedNode && !readOnly && (
        <div className="mt-4 p-4 bg-gray-800 rounded-lg border border-gray-700">
          <div className="flex items-center justify-between mb-3">
            <h3 className="font-semibold flex items-center gap-2">
              <FaEdit className="text-turquoise" />
              Edit Step: {selectedNode.title}
            </h3>
            <button onClick={() => setSelectedNode(null)} className="text-gray-400 hover:text-white">
              <FaTimes />
            </button>
          </div>

          <div className="space-y-3">
            <div>
              <label className="text-xs text-gray-400 mb-1 block">Code Snippet</label>
              <textarea
                value={selectedNode.code || ''}
                onChange={(e) => updateNode(selectedNode.id, { code: e.target.value })}
                className="w-full bg-gray-900 border border-gray-700 rounded px-3 py-2 text-sm text-gray-300 font-mono resize-none"
                rows="6"
                placeholder="// Add code snippet here..."
              />
            </div>

            <div>
              <label className="text-xs text-gray-400 mb-1 block">Status</label>
              <select
                value={selectedNode.status}
                onChange={(e) => updateNode(selectedNode.id, { status: e.target.value })}
                className="w-full bg-gray-900 border border-gray-700 rounded px-3 py-2 text-sm text-gray-300"
              >
                <option value="pending">Pending</option>
                <option value="running">Running</option>
                <option value="completed">Completed</option>
                <option value="error">Error</option>
              </select>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default VisualWorkflowBuilder;
