import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  FaUsers, FaBox, FaMoneyBillWave, FaBan, FaCheck, FaTimes,
  FaSearch, FaShieldAlt, FaExclamationTriangle, FaClock
} from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

const AdminDashboard = () => {
  const navigate = useNavigate();
  const { user } = useAuth();
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('overview');
  
  const [stats, setStats] = useState(null);
  const [users, setUsers] = useState([]);
  const [pendingContent, setPendingContent] = useState([]);
  const [pendingPayouts, setPendingPayouts] = useState([]);
  const [auditLogs, setAuditLogs] = useState([]);
  
  const [searchTerm, setSearchTerm] = useState('');
  const [filterRole, setFilterRole] = useState('');
  const [filterTier, setFilterTier] = useState('');

  useEffect(() => {
    // Check if user is admin
    if (!user?.isAdmin) {
      navigate('/dashboard');
      return;
    }

    fetchDashboardData();
  }, [user, navigate]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const response = await api.get('/admin/dashboard/stats');
      setStats(response.data);
      setAuditLogs(response.data.recentActivity || []);
    } catch (error) {
      console.error('Failed to fetch dashboard data:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchUsers = async () => {
    try {
      const params = new URLSearchParams();
      if (searchTerm) params.append('search', searchTerm);
      if (filterRole) params.append('role', filterRole);
      if (filterTier) params.append('subscriptionTier', filterTier);

      const response = await api.get(`/admin/users?${params}`);
      setUsers(response.data.users);
    } catch (error) {
      console.error('Failed to fetch users:', error);
    }
  };

  const fetchPendingContent = async () => {
    try {
      const response = await api.get('/admin/content/pending');
      setPendingContent(response.data.products || []);
    } catch (error) {
      console.error('Failed to fetch pending content:', error);
    }
  };

  const fetchPendingPayouts = async () => {
    try {
      const response = await api.get('/admin/payouts/pending');
      setPendingPayouts(response.data.payouts || []);
    } catch (error) {
      console.error('Failed to fetch pending payouts:', error);
    }
  };

  useEffect(() => {
    if (activeTab === 'users') fetchUsers();
    if (activeTab === 'content') fetchPendingContent();
    if (activeTab === 'payouts') fetchPendingPayouts();
  }, [activeTab, searchTerm, filterRole, filterTier]);

  const handleBanUser = async (userId, reason) => {
    if (!reason) reason = prompt('Reason for ban:');
    if (!reason) return;

    try {
      await api.put(`/admin/users/${userId}/ban`, { reason });
      fetchUsers();
      fetchDashboardData();
      alert('User banned successfully');
    } catch (error) {
      console.error('Failed to ban user:', error);
      alert('Failed to ban user');
    }
  };

  const handleUnbanUser = async (userId) => {
    try {
      await api.put(`/admin/users/${userId}/unban`);
      fetchUsers();
      fetchDashboardData();
      alert('User unbanned successfully');
    } catch (error) {
      console.error('Failed to unban user:', error);
      alert('Failed to unban user');
    }
  };

  const handleChangeRole = async (userId, newRole) => {
    try {
      await api.put(`/admin/users/${userId}/role`, { role: newRole });
      fetchUsers();
      alert('User role updated');
    } catch (error) {
      console.error('Failed to change role:', error);
      alert('Failed to change role');
    }
  };

  const handleApproveProduct = async (productId) => {
    try {
      await api.put(`/admin/products/${productId}/approve`);
      fetchPendingContent();
      fetchDashboardData();
      alert('Product approved');
    } catch (error) {
      console.error('Failed to approve product:', error);
      alert('Failed to approve product');
    }
  };

  const handleRejectProduct = async (productId) => {
    const reason = prompt('Reason for rejection:');
    if (!reason) return;

    try {
      await api.put(`/admin/products/${productId}/reject`, { reason });
      fetchPendingContent();
      fetchDashboardData();
      alert('Product rejected');
    } catch (error) {
      console.error('Failed to reject product:', error);
      alert('Failed to reject product');
    }
  };

  const handleApprovePayout = async (payoutId) => {
    try {
      await api.put(`/admin/payouts/${payoutId}/approve`);
      fetchPendingPayouts();
      fetchDashboardData();
      alert('Payout approved');
    } catch (error) {
      console.error('Failed to approve payout:', error);
      alert('Failed to approve payout');
    }
  };

  const handleRejectPayout = async (payoutId) => {
    const reason = prompt('Reason for rejection:');
    if (!reason) return;

    try {
      await api.put(`/admin/payouts/${payoutId}/reject`, { reason });
      fetchPendingPayouts();
      fetchDashboardData();
      alert('Payout rejected');
    } catch (error) {
      console.error('Failed to reject payout:', error);
      alert('Failed to reject payout');
    }
  };

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">
            <FaShieldAlt className="inline mr-2 text-turquoise" />
            Admin <span className="text-gradient">Dashboard</span>
          </h1>
          <p className="text-gray-400">Platform administration and moderation</p>
        </div>

        {/* Tabs */}
        <div className="flex gap-4 mb-8 overflow-x-auto">
          {['overview', 'users', 'content', 'payouts', 'logs'].map((tab) => (
            <button
              key={tab}
              onClick={() => setActiveTab(tab)}
              className={`px-6 py-3 rounded-lg font-semibold capitalize transition-all ${
                activeTab === tab
                  ? 'bg-turquoise text-black'
                  : 'bg-gray-800 text-gray-300 hover:bg-gray-700'
              }`}
            >
              {tab}
            </button>
          ))}
        </div>

        {/* Overview Tab */}
        {activeTab === 'overview' && stats && (
          <div className="space-y-8">
            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div className="card p-6">
                <div className="flex items-center justify-between mb-4">
                  <FaUsers className="text-3xl text-turquoise" />
                  <span className="text-sm text-gray-400">Users</span>
                </div>
                <h3 className="text-3xl font-bold">{stats.users.total}</h3>
                <p className="text-sm text-gray-400 mt-2">
                  {stats.users.active} active • {stats.users.banned} banned
                </p>
              </div>

              <div className="card p-6">
                <div className="flex items-center justify-between mb-4">
                  <FaBox className="text-3xl text-purple" />
                  <span className="text-sm text-gray-400">Products</span>
                </div>
                <h3 className="text-3xl font-bold">{stats.content.publishedProducts}</h3>
                <p className="text-sm text-gray-400 mt-2">
                  {stats.content.pendingProducts} pending approval
                </p>
              </div>

              <div className="card p-6">
                <div className="flex items-center justify-between mb-4">
                  <FaExclamationTriangle className="text-3xl text-yellow-500" />
                  <span className="text-sm text-gray-400">Pending</span>
                </div>
                <h3 className="text-3xl font-bold">
                  {stats.content.pendingProducts + stats.payouts.pending}
                </h3>
                <p className="text-sm text-gray-400 mt-2">Items need review</p>
              </div>

              <div className="card p-6">
                <div className="flex items-center justify-between mb-4">
                  <FaMoneyBillWave className="text-3xl text-green-500" />
                  <span className="text-sm text-gray-400">Payouts</span>
                </div>
                <h3 className="text-3xl font-bold">{stats.payouts.pending}</h3>
                <p className="text-sm text-gray-400 mt-2">Awaiting approval</p>
              </div>
            </div>

            {/* Recent Activity */}
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-4">Recent Admin Activity</h3>
              <div className="space-y-3">
                {auditLogs.length === 0 ? (
                  <p className="text-gray-400">No recent activity</p>
                ) : (
                  auditLogs.map((log, idx) => (
                    <div key={idx} className="flex items-center justify-between p-3 bg-gray-800 rounded-lg">
                      <div>
                        <p className="font-semibold">
                          {log.admin?.username || 'Admin'} - {log.action.replace(/_/g, ' ')}
                        </p>
                        <p className="text-sm text-gray-400">
                          {new Date(log.createdAt).toLocaleString()}
                        </p>
                      </div>
                      <span className="text-xs px-3 py-1 bg-turquoise/20 text-turquoise rounded-full">
                        {log.targetType}
                      </span>
                    </div>
                  ))
                )}
              </div>
            </div>
          </div>
        )}

        {/* Users Tab */}
        {activeTab === 'users' && (
          <div className="space-y-6">
            {/* Filters */}
            <div className="card p-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-2">Search</label>
                  <div className="relative">
                    <FaSearch className="absolute left-3 top-3 text-gray-400" />
                    <input
                      type="text"
                      placeholder="Username or email..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="input-field pl-10"
                    />
                  </div>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Role</label>
                  <select
                    value={filterRole}
                    onChange={(e) => setFilterRole(e.target.value)}
                    className="input-field"
                  >
                    <option value="">All Roles</option>
                    <option value="user">User</option>
                    <option value="creator">Creator</option>
                    <option value="admin">Admin</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Subscription</label>
                  <select
                    value={filterTier}
                    onChange={(e) => setFilterTier(e.target.value)}
                    className="input-field"
                  >
                    <option value="">All Tiers</option>
                    <option value="free">Free</option>
                    <option value="pro">Pro</option>
                    <option value="creator">Creator</option>
                  </select>
                </div>
              </div>
            </div>

            {/* Users Table */}
            <div className="card overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b border-gray-700">
                    <th className="text-left p-4">User</th>
                    <th className="text-left p-4">Email</th>
                    <th className="text-left p-4">Role</th>
                    <th className="text-left p-4">Tier</th>
                    <th className="text-left p-4">Status</th>
                    <th className="text-left p-4">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {users.map((user) => (
                    <tr key={user._id} className="border-b border-gray-800 hover:bg-gray-800/50">
                      <td className="p-4">
                        <div className="flex items-center gap-3">
                          {user.avatar && (
                            <img src={user.avatar} alt="" className="w-10 h-10 rounded-full" />
                          )}
                          <span className="font-semibold">{user.username}</span>
                        </div>
                      </td>
                      <td className="p-4 text-gray-400">{user.email}</td>
                      <td className="p-4">
                        <select
                          value={user.role}
                          onChange={(e) => handleChangeRole(user._id, e.target.value)}
                          className="bg-gray-700 px-3 py-1 rounded"
                        >
                          <option value="user">User</option>
                          <option value="creator">Creator</option>
                          <option value="admin">Admin</option>
                        </select>
                      </td>
                      <td className="p-4">
                        <span className="px-3 py-1 bg-purple/20 text-purple rounded-full text-sm">
                          {user.subscriptionTier}
                        </span>
                      </td>
                      <td className="p-4">
                        {user.isBanned ? (
                          <span className="px-3 py-1 bg-red-500/20 text-red-500 rounded-full text-sm flex items-center gap-1 w-fit">
                            <FaBan /> Banned
                          </span>
                        ) : (
                          <span className="px-3 py-1 bg-green-500/20 text-green-500 rounded-full text-sm">
                            Active
                          </span>
                        )}
                      </td>
                      <td className="p-4">
                        <div className="flex gap-2">
                          {user.isBanned ? (
                            <button
                              onClick={() => handleUnbanUser(user._id)}
                              className="btn-secondary py-1 px-3 text-sm"
                            >
                              Unban
                            </button>
                          ) : (
                            <button
                              onClick={() => handleBanUser(user._id)}
                              className="bg-red-500 hover:bg-red-600 text-white py-1 px-3 rounded text-sm"
                            >
                              Ban
                            </button>
                          )}
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Content Moderation Tab */}
        {activeTab === 'content' && (
          <div className="space-y-6">
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-4">Pending Products ({pendingContent.length})</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {pendingContent.length === 0 ? (
                  <p className="text-gray-400 col-span-2">No pending products</p>
                ) : (
                  pendingContent.map((product) => (
                    <div key={product._id} className="bg-gray-800 rounded-lg p-4">
                      <div className="flex justify-between items-start mb-3">
                        <div>
                          <h4 className="font-bold text-lg">{product.title}</h4>
                          <p className="text-sm text-gray-400">
                            by {product.creator?.username || 'Unknown'}
                          </p>
                        </div>
                        <span className="text-turquoise font-bold">${product.price}</span>
                      </div>
                      <p className="text-gray-400 text-sm mb-4 line-clamp-2">
                        {product.description}
                      </p>
                      <div className="flex gap-2">
                        <button
                          onClick={() => handleApproveProduct(product._id)}
                          className="btn-primary flex-1 flex items-center justify-center gap-2"
                        >
                          <FaCheck /> Approve
                        </button>
                        <button
                          onClick={() => handleRejectProduct(product._id)}
                          className="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded flex-1 flex items-center justify-center gap-2"
                        >
                          <FaTimes /> Reject
                        </button>
                      </div>
                    </div>
                  ))
                )}
              </div>
            </div>
          </div>
        )}

        {/* Payouts Tab */}
        {activeTab === 'payouts' && (
          <div className="card overflow-x-auto">
            <h3 className="text-xl font-bold p-6 pb-4">Pending Payouts ({pendingPayouts.length})</h3>
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-700">
                  <th className="text-left p-4">Creator</th>
                  <th className="text-left p-4">Amount</th>
                  <th className="text-left p-4">Requested</th>
                  <th className="text-left p-4">Actions</th>
                </tr>
              </thead>
              <tbody>
                {pendingPayouts.length === 0 ? (
                  <tr>
                    <td colSpan="4" className="p-4 text-center text-gray-400">
                      No pending payouts
                    </td>
                  </tr>
                ) : (
                  pendingPayouts.map((payout) => (
                    <tr key={payout._id} className="border-b border-gray-800 hover:bg-gray-800/50">
                      <td className="p-4">
                        <div>
                          <p className="font-semibold">{payout.creator?.username}</p>
                          <p className="text-sm text-gray-400">{payout.creator?.email}</p>
                        </div>
                      </td>
                      <td className="p-4">
                        <span className="text-green-500 font-bold text-lg">${payout.amount}</span>
                      </td>
                      <td className="p-4 text-gray-400">
                        {new Date(payout.createdAt).toLocaleDateString()}
                      </td>
                      <td className="p-4">
                        <div className="flex gap-2">
                          <button
                            onClick={() => handleApprovePayout(payout._id)}
                            className="btn-primary px-4 py-2 flex items-center gap-2"
                          >
                            <FaCheck /> Approve
                          </button>
                          <button
                            onClick={() => handleRejectPayout(payout._id)}
                            className="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded flex items-center gap-2"
                          >
                            <FaTimes /> Reject
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        )}

        {/* Audit Logs Tab */}
        {activeTab === 'logs' && (
          <div className="card p-6">
            <h3 className="text-xl font-bold mb-4">Audit Logs</h3>
            <div className="space-y-3">
              {auditLogs.map((log, idx) => (
                <div key={idx} className="p-4 bg-gray-800 rounded-lg">
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <p className="font-semibold">
                        {log.admin?.username || 'Admin'} - {log.action.replace(/_/g, ' ')}
                      </p>
                      <p className="text-sm text-gray-400">{log.reason || 'No reason provided'}</p>
                    </div>
                    <span className="text-xs px-3 py-1 bg-turquoise/20 text-turquoise rounded-full">
                      {log.targetType}
                    </span>
                  </div>
                  <p className="text-xs text-gray-500">
                    {new Date(log.createdAt).toLocaleString()} • IP: {log.ipAddress || 'N/A'}
                  </p>
                </div>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default AdminDashboard;
