import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import blogPosts from '../data/blogPosts';
import { 
  Calendar, 
  Clock, 
  User, 
  Tag, 
  ArrowRight,
  Heart,
  MessageCircle,
  Share2,
  BookOpen,
  Search,
  TrendingUp,
  SortAsc,
  Grid3x3,
  List,
  Bookmark,
  BookmarkCheck,
  Filter
} from 'lucide-react';

const Blog = () => {
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [searchQuery, setSearchQuery] = useState('');
  const [sortBy, setSortBy] = useState('latest'); // latest, popular, trending
  const [viewMode, setViewMode] = useState('grid'); // grid or list
  const [savedPosts, setSavedPosts] = useState([]);
  const [showSortMenu, setShowSortMenu] = useState(false);

  // Load saved posts from localStorage
  useEffect(() => {
    const saved = localStorage.getItem('savedBlogPosts');
    if (saved) setSavedPosts(JSON.parse(saved));
  }, []);

  // blogPosts imported from shared data file

  const categories = ['all', 'Philosophy', 'Productivity', 'Community', 'Tutorials', 'News', 'AI Insights', 'Security'];

  // Toggle save post
  const toggleSavePost = (postId) => {
    const newSaved = savedPosts.includes(postId)
      ? savedPosts.filter(id => id !== postId)
      : [...savedPosts, postId];
    setSavedPosts(newSaved);
    localStorage.setItem('savedBlogPosts', JSON.stringify(newSaved));
  };

  // Filter by category
  let filteredPosts = selectedCategory === 'all' 
    ? blogPosts 
    : blogPosts.filter(post => post.category === selectedCategory);

  // Search filter
  if (searchQuery) {
    filteredPosts = filteredPosts.filter(post => 
      post.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
      post.excerpt.toLowerCase().includes(searchQuery.toLowerCase()) ||
      post.author.toLowerCase().includes(searchQuery.toLowerCase())
    );
  }

  // Sort posts
  const sortedPosts = [...filteredPosts].sort((a, b) => {
    if (sortBy === 'popular') return b.likes - a.likes;
    if (sortBy === 'trending') return b.comments - a.comments;
    return b.id - a.id; // latest (by id)
  });

  return (
    <div className="min-h-screen bg-slate-900 text-gray-100 pt-20">
      {/* Header */}
      <div className="bg-gradient-to-r from-slate-900 to-slate-800 text-white py-16">
        <div className="max-w-7xl mx-auto px-6 sm:px-8 lg:px-12 text-center">
          <h1 className="text-5xl font-bold mb-4">Vibe Coding Blog</h1>
          <p className="text-xl text-blue-100 max-w-2xl mx-auto">
            Insights, tutorials, and stories from the vibe coding community
          </p>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-6 sm:px-8 lg:px-12 py-12">
        {/* Premium Search & Filters */}
        <div className="mb-8 bg-slate-800/50 backdrop-blur-sm rounded-xl p-6 border border-slate-700 shadow-xl">
          <div className="flex flex-col lg:flex-row gap-4 items-center justify-between">
            {/* Search Bar */}
            <div className="w-full lg:w-96 relative">
              <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search posts, authors, topics..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pl-12 pr-4 py-3 bg-slate-900 border border-slate-600 rounded-lg text-gray-100 placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-[#20B2AA] focus:border-transparent transition-all"
              />
              {searchQuery && (
                <button
                  onClick={() => setSearchQuery('')}
                  className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-white"
                >
                  ×
                </button>
              )}
            </div>

            {/* Controls */}
            <div className="flex items-center gap-3">
              {/* Sort Dropdown */}
              <div className="relative">
                <button
                  onClick={() => setShowSortMenu(!showSortMenu)}
                  className="flex items-center gap-2 px-4 py-2.5 bg-slate-900 border border-slate-600 rounded-lg text-gray-300 hover:border-[#20B2AA] transition-all"
                >
                  <SortAsc className="w-4 h-4" />
                  <span className="hidden sm:inline">
                    {sortBy === 'latest' && 'Latest'}
                    {sortBy === 'popular' && 'Most Liked'}
                    {sortBy === 'trending' && 'Trending'}
                  </span>
                </button>
                {showSortMenu && (
                  <div className="absolute right-0 mt-2 w-48 bg-slate-900 border border-slate-700 rounded-lg shadow-xl z-10">
                    <button
                      onClick={() => { setSortBy('latest'); setShowSortMenu(false); }}
                      className="w-full text-left px-4 py-3 hover:bg-slate-800 text-gray-300 transition-colors rounded-t-lg"
                    >
                      📅 Latest First
                    </button>
                    <button
                      onClick={() => { setSortBy('popular'); setShowSortMenu(false); }}
                      className="w-full text-left px-4 py-3 hover:bg-slate-800 text-gray-300 transition-colors"
                    >
                      ❤️ Most Liked
                    </button>
                    <button
                      onClick={() => { setSortBy('trending'); setShowSortMenu(false); }}
                      className="w-full text-left px-4 py-3 hover:bg-slate-800 text-gray-300 transition-colors rounded-b-lg"
                    >
                      🔥 Trending
                    </button>
                  </div>
                )}
              </div>

              {/* View Toggle */}
              <div className="flex bg-slate-900 border border-slate-600 rounded-lg p-1">
                <button
                  onClick={() => setViewMode('grid')}
                  className={`p-2 rounded transition-all ${
                    viewMode === 'grid' 
                      ? 'bg-[#20B2AA] text-black' 
                      : 'text-gray-400 hover:text-white'
                  }`}
                  title="Grid View"
                >
                  <Grid3x3 className="w-4 h-4" />
                </button>
                <button
                  onClick={() => setViewMode('list')}
                  className={`p-2 rounded transition-all ${
                    viewMode === 'list' 
                      ? 'bg-[#20B2AA] text-black' 
                      : 'text-gray-400 hover:text-white'
                  }`}
                  title="List View"
                >
                  <List className="w-4 h-4" />
                </button>
              </div>

              {/* Saved Filter */}
              <button
                onClick={() => setSelectedCategory(selectedCategory === 'saved' ? 'all' : 'saved')}
                className={`flex items-center gap-2 px-4 py-2.5 rounded-lg border transition-all ${
                  selectedCategory === 'saved'
                    ? 'bg-[#20B2AA] text-black border-[#20B2AA]'
                    : 'bg-slate-900 text-gray-300 border-slate-600 hover:border-[#20B2AA]'
                }`}
              >
                <Bookmark className="w-4 h-4" />
                <span className="hidden sm:inline">Saved ({savedPosts.length})</span>
              </button>
            </div>
          </div>

          {/* Results count */}
          <div className="mt-4 text-sm text-gray-400">
            {selectedCategory === 'saved' ? (
              <span>📚 Showing {savedPosts.length} saved post{savedPosts.length !== 1 ? 's' : ''}</span>
            ) : (
              <span>Found {sortedPosts.length} post{sortedPosts.length !== 1 ? 's' : ''}</span>
            )}
            {searchQuery && <span className="ml-2">matching "{searchQuery}"</span>}
          </div>
        </div>

        {/* Categories */}
        <div className="flex flex-wrap gap-3 mb-12 justify-center">
          {categories.map((category) => (
            <button
              key={category}
              onClick={() => setSelectedCategory(category)}
              className={`px-6 py-2 rounded-full font-medium transition-all ${
                selectedCategory === category
                  ? 'bg-[#20B2AA] text-black shadow-lg'
                  : 'bg-slate-800 text-gray-300 hover:bg-slate-700 border border-slate-700'
              }`}
            >
              {category.charAt(0).toUpperCase() + category.slice(1)}
            </button>
          ))}
        </div>

        {/* Featured Post */}
        {selectedCategory === 'all' && !searchQuery && blogPosts[0].featured && (
          <div className="mb-12">
            <div className="bg-slate-800 rounded-2xl shadow-xl overflow-hidden">
              <div className="grid md:grid-cols-2 gap-0">
                <div className="relative h-64 md:h-auto">
                  <img 
                    src={blogPosts[0].image} 
                    alt={blogPosts[0].title}
                    className="w-full h-full object-cover"
                  />
                  <div className="absolute top-4 left-4 bg-yellow-400 text-gray-900 px-4 py-2 rounded-full font-bold text-sm">
                    Featured
                  </div>
                </div>
                <div className="p-8 flex flex-col justify-center">
                  <div className="flex items-center gap-4 text-sm text-gray-300 mb-4">
                    <span className="bg-blue-700 text-white px-3 py-1 rounded-full font-medium">
                      {blogPosts[0].category}
                    </span>
                    <div className="flex items-center gap-1">
                      <Calendar className="w-4 h-4" />
                      <span>{blogPosts[0].date}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <Clock className="w-4 h-4" />
                      <span>{blogPosts[0].readTime}</span>
                    </div>
                  </div>
                  <h2 className="text-3xl font-bold text-white mb-4">
                    {blogPosts[0].title}
                  </h2>
                  <p className="text-gray-300 mb-6 leading-relaxed">
                    {blogPosts[0].excerpt}
                  </p>
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-bold">
                        {blogPosts[0].author.charAt(0)}
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-100">{blogPosts[0].author}</p>
                        <p className="text-xs text-gray-400">Author</p>
                      </div>
                    </div>
                    <Link
                      to={`/blog/${blogPosts[0].slug}`}
                      className="inline-flex items-center gap-2 bg-[#20B2AA] text-black px-6 py-3 rounded-lg font-semibold hover:opacity-95 transition-colors"
                    >
                      <span>Read More</span>
                      <ArrowRight className="w-4 h-4" />
                    </Link>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Blog Grid/List */}
        <div className={viewMode === 'grid' ? 'grid md:grid-cols-2 lg:grid-cols-3 gap-8' : 'space-y-6'}>
          {(selectedCategory === 'saved' 
            ? sortedPosts.filter(post => savedPosts.includes(post.id))
            : sortedPosts.slice(selectedCategory === 'all' && !searchQuery ? 1 : 0)
          ).map((post) => (
            <article key={post.id} className={`bg-slate-800 rounded-xl shadow-lg overflow-hidden hover:shadow-xl transition-all ${
              viewMode === 'list' ? 'flex flex-row' : ''
            }`}>
              <div className={`relative overflow-hidden ${
                viewMode === 'list' ? 'w-64 flex-shrink-0' : 'h-48'
              }`}>
                <img 
                  src={post.image} 
                  alt={post.title}
                  className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                />
                <div className="absolute top-4 right-4 bg-white text-gray-900 px-3 py-1 rounded-full font-medium text-xs shadow-md">
                  {post.category}
                </div>
                {/* Save Button Overlay */}
                <button
                  onClick={(e) => {
                    e.preventDefault();
                    toggleSavePost(post.id);
                  }}
                  className="absolute top-4 left-4 bg-slate-900/80 backdrop-blur-sm p-2 rounded-full hover:bg-slate-900 transition-all"
                  title={savedPosts.includes(post.id) ? 'Remove from saved' : 'Save for later'}
                >
                  {savedPosts.includes(post.id) ? (
                    <BookmarkCheck className="w-4 h-4 text-[#20B2AA]" />
                  ) : (
                    <Bookmark className="w-4 h-4 text-white" />
                  )}
                </button>
              </div>
              
              <div className="p-6 flex-1 flex flex-col">
                <div className="flex items-center gap-3 text-xs text-gray-300 mb-3">
                  <div className="flex items-center gap-1">
                    <Calendar className="w-3 h-3" />
                    <span>{post.date}</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <Clock className="w-3 h-3" />
                    <span>{post.readTime}</span>
                  </div>
                </div>
                
                <h3 className={`font-bold text-white mb-3 hover:text-[#20B2AA] transition-colors ${
                  viewMode === 'list' ? 'text-2xl line-clamp-2' : 'text-xl line-clamp-2'
                }`}>
                  <Link to={`/blog/${post.slug}`}>
                    {post.title}
                  </Link>
                </h3>
                
                <p className={`text-gray-300 text-sm mb-4 flex-1 ${
                  viewMode === 'list' ? 'line-clamp-2' : 'line-clamp-3'
                }`}>
                  {post.excerpt}
                </p>
                
                <div className="flex items-center justify-between pt-4 border-t border-gray-100">
                  <div className="flex items-center gap-2">
                    <div className="w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white text-xs font-bold">
                      {post.author.charAt(0)}
                    </div>
                    <span className="text-sm font-medium text-gray-700">{post.author}</span>
                  </div>
                  
                  <div className="flex items-center gap-4 text-gray-500 text-sm">
                    <div className="flex items-center gap-1">
                      <Heart className="w-4 h-4" />
                      <span>{post.likes}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <MessageCircle className="w-4 h-4" />
                      <span>{post.comments}</span>
                    </div>
                  </div>
                </div>
              </div>
            </article>
          ))}
        </div>

        {/* Newsletter CTA */}
          <div className="mt-16 bg-gradient-to-r from-[#20B2AA] to-[#FFD700] rounded-2xl p-8 md:p-12 text-white text-center">
          <BookOpen className="w-12 h-12 mx-auto mb-4" />
          <h2 className="text-3xl font-bold mb-4">Stay in the Loop</h2>
          <p className="text-blue-100 mb-6 max-w-2xl mx-auto">
            Get the latest vibe coding insights, tutorials, and community updates delivered to your inbox every week.
          </p>
          <form className="max-w-md mx-auto flex gap-3">
            <input
              type="email"
              placeholder="Enter your email"
              className="flex-1 px-4 py-3 rounded-lg text-gray-900 focus:outline-none focus:ring-2 focus:ring-blue-300"
            />
            <button
              type="submit"
              className="bg-white text-blue-600 px-6 py-3 rounded-lg font-semibold hover:bg-blue-50 transition-colors"
            >
              Subscribe
            </button>
          </form>
        </div>
      </div>
    </div>
  );
};

export default Blog;
