import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import api from '../services/api';
import {
  FaEye,
  FaUsers,
  FaCode,
  FaTrophy,
  FaArrowLeft,
  FaCalendar,
  FaDollarSign,
} from 'react-icons/fa';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
} from 'recharts';

const ChallengeSponsorDashboard = () => {
  const { challengeId } = useParams();
  const navigate = useNavigate();
  const [analytics, setAnalytics] = useState(null);
  const [challenge, setChallenge] = useState(null);
  const [payouts, setPayouts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showWinnerModal, setShowWinnerModal] = useState(false);
  const [selectedWinners, setSelectedWinners] = useState([
    { place: 1, userId: '' },
    { place: 2, userId: '' },
    { place: 3, userId: '' },
  ]);

  useEffect(() => {
    fetchData();
  }, [challengeId]);

  const fetchData = async () => {
    try {
      setLoading(true);
      const [analyticsRes, challengeRes, payoutsRes] = await Promise.all([
        api.get(`/challenge-sponsors/analytics/${challengeId}`),
        api.get(`/challenges/${challengeId}`),
        api.get(`/challenge-sponsors/payouts/${challengeId}`),
      ]);

      setAnalytics(analyticsRes.data.analytics);
      setChallenge(challengeRes.data.challenge);
      setPayouts(payoutsRes.data.payouts);
    } catch (error) {
      console.error('Failed to fetch data:', error);
      alert('Failed to load dashboard');
      navigate('/challenges');
    } finally {
      setLoading(false);
    }
  };

  const handleAnnounceWinners = async () => {
    try {
      const winners = selectedWinners.filter(w => w.userId);
      if (winners.length === 0) {
        alert('Please select at least one winner');
        return;
      }

      await api.post(`/challenge-sponsors/announce-winners/${challengeId}`, { winners });
      alert('Winners announced successfully!');
      setShowWinnerModal(false);
      fetchData();
    } catch (error) {
      console.error('Failed to announce winners:', error);
      alert(error.response?.data?.message || 'Failed to announce winners');
    }
  };

  const handleProcessPayout = async (payoutId) => {
    try {
      await api.post(`/challenge-sponsors/payout/${payoutId}`, {
        paymentMethod: 'stripe',
        paymentDetails: {},
      });
      alert('Payout processed successfully!');
      fetchData();
    } catch (error) {
      console.error('Failed to process payout:', error);
      alert('Failed to process payout');
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <button
            onClick={() => navigate('/challenges')}
            className="flex items-center gap-2 text-gray-400 hover:text-turquoise mb-4"
          >
            <FaArrowLeft />
            Back to Challenges
          </button>
          
          <div className="flex justify-between items-start">
            <div>
              <h1 className="text-4xl font-bold mb-2">{challenge?.title}</h1>
              <p className="text-gray-400">Sponsorship Dashboard</p>
            </div>
            
            {challenge?.status === 'completed' && !challenge?.winners?.length && (
              <button
                onClick={() => setShowWinnerModal(true)}
                className="btn-primary"
              >
                Announce Winners
              </button>
            )}
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaEye className="text-3xl text-turquoise" />
              <span className="text-sm text-gray-400">Views</span>
            </div>
            <div className="text-3xl font-bold">{analytics?.views || 0}</div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaUsers className="text-3xl text-purple" />
              <span className="text-sm text-gray-400">Participants</span>
            </div>
            <div className="text-3xl font-bold">{analytics?.participants || 0}</div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaCode className="text-3xl text-green-500" />
              <span className="text-sm text-gray-400">Submissions</span>
            </div>
            <div className="text-3xl font-bold">{analytics?.submissions || 0}</div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaDollarSign className="text-3xl text-yellow-500" />
              <span className="text-sm text-gray-400">Prize Pool</span>
            </div>
            <div className="text-3xl font-bold">${analytics?.prizePool || 0}</div>
          </div>
        </div>

        {/* Challenge Details */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
          {/* Sponsorship Info */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Sponsorship Details</h2>
            <div className="space-y-4">
              <div className="flex justify-between">
                <span className="text-gray-400">Tier:</span>
                <span className="font-semibold capitalize">{analytics?.tier}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">Amount Paid:</span>
                <span className="font-bold text-turquoise">
                  ${analytics?.sponsorshipAmount?.toFixed(2)}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">Status:</span>
                <span className="font-semibold capitalize">{analytics?.status}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">Start Date:</span>
                <span>{new Date(analytics?.startDate).toLocaleDateString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">End Date:</span>
                <span>{new Date(analytics?.endDate).toLocaleDateString()}</span>
              </div>
            </div>

            {analytics?.features && (
              <div className="mt-6 pt-6 border-t border-gray-700">
                <h3 className="font-semibold mb-3">Included Features:</h3>
                <ul className="space-y-2 text-sm text-gray-400">
                  {analytics.features.isFeatured && <li>✅ Featured placement</li>}
                  {analytics.features.isPinned && <li>✅ Top pinned position</li>}
                  {analytics.features.bannerImage && <li>✅ Custom banner image</li>}
                  {analytics.features.socialMediaPromotion && <li>✅ Social media promotion</li>}
                  {analytics.features.newsletterInclusion && <li>✅ Newsletter inclusion</li>}
                </ul>
              </div>
            )}
          </div>

          {/* Prize Distribution */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Prize Distribution</h2>
            {analytics?.prizeDistribution?.map((prize, idx) => (
              <div key={idx} className="flex justify-between items-center mb-4 p-4 bg-gray-800 rounded-lg">
                <div className="flex items-center gap-3">
                  <FaTrophy
                    className={`text-2xl ${
                      prize.place === 1
                        ? 'text-yellow-500'
                        : prize.place === 2
                        ? 'text-gray-400'
                        : 'text-orange-500'
                    }`}
                  />
                  <span className="font-semibold">{prize.description}</span>
                </div>
                <span className="text-xl font-bold text-turquoise">
                  ${prize.amount.toFixed(0)}
                </span>
              </div>
            ))}
          </div>
        </div>

        {/* Winners & Payouts */}
        {analytics?.winners?.length > 0 && (
          <div className="card p-6 mb-8">
            <h2 className="text-2xl font-bold mb-6">Winners & Payouts</h2>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b border-gray-700">
                    <th className="text-left py-3 px-4">Place</th>
                    <th className="text-left py-3 px-4">Winner</th>
                    <th className="text-right py-3 px-4">Prize</th>
                    <th className="text-center py-3 px-4">Status</th>
                    <th className="text-center py-3 px-4">Action</th>
                  </tr>
                </thead>
                <tbody>
                  {payouts.map((payout) => (
                    <tr key={payout._id} className="border-b border-gray-800">
                      <td className="py-4 px-4">
                        <FaTrophy
                          className={`text-xl ${
                            payout.place === 1
                              ? 'text-yellow-500'
                              : payout.place === 2
                              ? 'text-gray-400'
                              : 'text-orange-500'
                          }`}
                        />
                      </td>
                      <td className="py-4 px-4">
                        <div className="flex items-center gap-3">
                          <img
                            src={payout.winner.avatar || '/default-avatar.png'}
                            alt={payout.winner.username}
                            className="w-10 h-10 rounded-full"
                          />
                          <div>
                            <div className="font-medium">{payout.winner.username}</div>
                            <div className="text-sm text-gray-400">{payout.winner.email}</div>
                          </div>
                        </div>
                      </td>
                      <td className="text-right py-4 px-4 font-bold text-turquoise">
                        ${payout.amount.toFixed(2)}
                      </td>
                      <td className="text-center py-4 px-4">
                        <span
                          className={`px-3 py-1 rounded-full text-sm ${
                            payout.status === 'completed'
                              ? 'bg-green-500/20 text-green-500'
                              : payout.status === 'processing'
                              ? 'bg-yellow-500/20 text-yellow-500'
                              : 'bg-gray-500/20 text-gray-500'
                          }`}
                        >
                          {payout.status}
                        </span>
                      </td>
                      <td className="text-center py-4 px-4">
                        {payout.status === 'pending' && (
                          <button
                            onClick={() => handleProcessPayout(payout._id)}
                            className="btn-primary text-sm"
                          >
                            Process Payout
                          </button>
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Submissions */}
        <div className="card p-6">
          <h2 className="text-2xl font-bold mb-6">Recent Submissions</h2>
          <div className="space-y-4">
            {challenge?.submissions?.slice(0, 10).map((submission, idx) => (
              <div key={idx} className="flex items-center justify-between p-4 bg-gray-800 rounded-lg">
                <div className="flex items-center gap-3">
                  <img
                    src={submission.user.avatar || '/default-avatar.png'}
                    alt={submission.user.username}
                    className="w-10 h-10 rounded-full"
                  />
                  <div>
                    <div className="font-medium">{submission.user.username}</div>
                    <div className="text-sm text-gray-400">
                      {new Date(submission.submittedAt).toLocaleDateString()}
                    </div>
                  </div>
                </div>
                <div className="flex gap-2">
                  {submission.repoUrl && (
                    <a
                      href={submission.repoUrl}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="btn-secondary text-sm"
                    >
                      View Code
                    </a>
                  )}
                  {submission.liveUrl && (
                    <a
                      href={submission.liveUrl}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="btn-primary text-sm"
                    >
                      View Live
                    </a>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Winner Selection Modal */}
      {showWinnerModal && (
        <div className="fixed inset-0 bg-black/80 flex items-center justify-center z-50 p-4">
          <div className="card max-w-2xl w-full p-8 max-h-[90vh] overflow-y-auto">
            <h2 className="text-2xl font-bold mb-6">Select Winners</h2>
            
            {[1, 2, 3].map((place) => (
              <div key={place} className="mb-6">
                <label className="block text-sm font-medium mb-2">
                  {place === 1 ? '1st' : place === 2 ? '2nd' : '3rd'} Place Winner
                </label>
                <select
                  value={selectedWinners.find(w => w.place === place)?.userId || ''}
                  onChange={(e) => {
                    const newWinners = [...selectedWinners];
                    const winnerIdx = newWinners.findIndex(w => w.place === place);
                    newWinners[winnerIdx].userId = e.target.value;
                    setSelectedWinners(newWinners);
                  }}
                  className="input-field"
                >
                  <option value="">Select winner...</option>
                  {challenge?.submissions?.map((sub) => (
                    <option key={sub.user._id} value={sub.user._id}>
                      {sub.user.username}
                    </option>
                  ))}
                </select>
              </div>
            ))}

            <div className="flex gap-4">
              <button onClick={handleAnnounceWinners} className="btn-primary flex-1">
                Announce Winners
              </button>
              <button onClick={() => setShowWinnerModal(false)} className="btn-secondary flex-1">
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ChallengeSponsorDashboard;
