import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';
import { FaRocket, FaStar, FaCrown, FaCheck, FaTrophy } from 'react-icons/fa';

const ChallengeSponsorPricing = ({ challengeId }) => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [showCompanyForm, setShowCompanyForm] = useState(false);
  const [selectedTier, setSelectedTier] = useState(null);
  const [companyDetails, setCompanyDetails] = useState({
    name: '',
    logo: '',
    website: '',
    description: '',
  });

  const tiers = [
    {
      name: 'Silver',
      tier: 'silver',
      price: 499.99,
      prizePool: 500,
      duration: 30,
      icon: <FaTrophy className="text-4xl text-gray-400" />,
      features: [
        '30 days sponsorship',
        '$500 prize pool',
        'Standard placement',
        'Newsletter inclusion',
        'Basic branding',
      ],
      color: 'gray-400',
    },
    {
      name: 'Gold',
      tier: 'gold',
      price: 999.99,
      prizePool: 1500,
      duration: 45,
      icon: <FaStar className="text-4xl text-yellow-500" />,
      popular: true,
      features: [
        '45 days sponsorship',
        '$1,500 prize pool',
        'Featured placement',
        'Custom banner image',
        'Brand color highlight',
        'Social media promotion',
        'Newsletter inclusion',
      ],
      color: 'yellow-500',
    },
    {
      name: 'Platinum',
      tier: 'platinum',
      price: 2499.99,
      prizePool: 5000,
      duration: 60,
      icon: <FaCrown className="text-4xl text-purple" />,
      features: [
        '60 days sponsorship',
        '$5,000 prize pool',
        'Top pinned position',
        'Custom banner image',
        'Brand color highlight',
        'Social media promotion',
        'Newsletter inclusion',
        'Dedicated account manager',
        'Custom challenge branding',
      ],
      color: 'purple',
    },
  ];

  const handleSelectTier = (tier) => {
    setSelectedTier(tier);
    setShowCompanyForm(true);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!companyDetails.name) {
      alert('Company name is required');
      return;
    }

    try {
      setLoading(true);
      const response = await api.post('/challenge-sponsors/checkout', {
        challengeId,
        tier: selectedTier,
        companyDetails,
      });

      // Redirect to Stripe checkout
      window.location.href = response.data.url;
    } catch (error) {
      console.error('Checkout error:', error);
      alert(error.response?.data?.message || 'Failed to start checkout');
      setLoading(false);
    }
  };

  if (showCompanyForm) {
    return (
      <div className="min-h-screen px-4 py-20">
        <div className="max-w-2xl mx-auto">
          <button
            onClick={() => setShowCompanyForm(false)}
            className="text-gray-400 hover:text-turquoise mb-6"
          >
            ← Back to pricing
          </button>

          <div className="card p-8">
            <h2 className="text-3xl font-bold mb-6">Company Details</h2>
            <p className="text-gray-400 mb-8">
              Please provide your company information for challenge branding
            </p>

            <form onSubmit={handleSubmit} className="space-y-6">
              <div>
                <label className="block text-sm font-medium mb-2">
                  Company Name *
                </label>
                <input
                  type="text"
                  value={companyDetails.name}
                  onChange={(e) => setCompanyDetails({ ...companyDetails, name: e.target.value })}
                  className="input-field"
                  placeholder="Your Company Inc."
                  required
                />
              </div>

              <div>
                <label className="block text-sm font-medium mb-2">
                  Company Logo URL
                </label>
                <input
                  type="url"
                  value={companyDetails.logo}
                  onChange={(e) => setCompanyDetails({ ...companyDetails, logo: e.target.value })}
                  className="input-field"
                  placeholder="https://example.com/logo.png"
                />
              </div>

              <div>
                <label className="block text-sm font-medium mb-2">
                  Company Website
                </label>
                <input
                  type="url"
                  value={companyDetails.website}
                  onChange={(e) => setCompanyDetails({ ...companyDetails, website: e.target.value })}
                  className="input-field"
                  placeholder="https://yourcompany.com"
                />
              </div>

              <div>
                <label className="block text-sm font-medium mb-2">
                  Company Description
                </label>
                <textarea
                  value={companyDetails.description}
                  onChange={(e) => setCompanyDetails({ ...companyDetails, description: e.target.value })}
                  className="input-field"
                  rows="4"
                  placeholder="Brief description of your company..."
                />
              </div>

              <div className="flex gap-4">
                <button
                  type="submit"
                  disabled={loading}
                  className="btn-primary flex-1 disabled:opacity-50"
                >
                  {loading ? 'Processing...' : 'Continue to Payment'}
                </button>
                <button
                  type="button"
                  onClick={() => setShowCompanyForm(false)}
                  className="btn-secondary flex-1"
                >
                  Cancel
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-5xl font-bold mb-4">
            Sponsor a <span className="text-gradient">Challenge</span>
          </h1>
          <p className="text-xl text-gray-400 max-w-2xl mx-auto">
            Connect with talented developers and showcase your brand to the community
          </p>
        </div>

        {/* Pricing Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12">
          {tiers.map((tierInfo) => (
            <div
              key={tierInfo.tier}
              className={`card relative ${
                tierInfo.popular ? 'ring-2 ring-turquoise' : ''
              }`}
            >
              {tierInfo.popular && (
                <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                  <span className="bg-gradient-to-r from-turquoise to-purple px-4 py-1 rounded-full text-sm font-semibold">
                    Most Popular
                  </span>
                </div>
              )}

              <div className="p-8">
                <div className="flex justify-center mb-4">{tierInfo.icon}</div>
                
                <h3 className="text-2xl font-bold text-center mb-2">{tierInfo.name}</h3>
                
                <div className="text-center mb-2">
                  <span className="text-4xl font-bold">${tierInfo.price}</span>
                </div>
                
                <div className="text-center mb-6">
                  <span className="text-2xl font-bold text-turquoise">
                    ${tierInfo.prizePool}
                  </span>
                  <span className="text-gray-400 ml-2">prize pool</span>
                </div>

                <ul className="space-y-3 mb-8">
                  {tierInfo.features.map((feature, idx) => (
                    <li key={idx} className="flex items-start gap-2">
                      <FaCheck className="text-turquoise mt-1 flex-shrink-0" />
                      <span className="text-gray-300">{feature}</span>
                    </li>
                  ))}
                </ul>

                <button
                  onClick={() => handleSelectTier(tierInfo.tier)}
                  disabled={loading}
                  className={`w-full py-3 rounded-lg font-semibold transition-all disabled:opacity-50 ${
                    tierInfo.popular
                      ? 'btn-primary'
                      : 'bg-gray-700 hover:bg-gray-600 text-white'
                  }`}
                >
                  Select Plan
                </button>
              </div>
            </div>
          ))}
        </div>

        {/* Benefits */}
        <div className="card p-8 mb-8">
          <h2 className="text-3xl font-bold mb-6 text-center">Why Sponsor a Challenge?</h2>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center">
              <div className="text-4xl mb-4">🎯</div>
              <h3 className="text-xl font-bold mb-2">Brand Awareness</h3>
              <p className="text-gray-400">
                Showcase your brand to thousands of talented developers
              </p>
            </div>
            <div className="text-center">
              <div className="text-4xl mb-4">💼</div>
              <h3 className="text-xl font-bold mb-2">Talent Pipeline</h3>
              <p className="text-gray-400">
                Discover and recruit top developers through their submissions
              </p>
            </div>
            <div className="text-center">
              <div className="text-4xl mb-4">🚀</div>
              <h3 className="text-xl font-bold mb-2">Community Impact</h3>
              <p className="text-gray-400">
                Support the developer community and give back
              </p>
            </div>
          </div>
        </div>

        {/* Prize Distribution */}
        <div className="card p-8">
          <h2 className="text-3xl font-bold mb-6 text-center">Prize Distribution</h2>
          <p className="text-gray-400 text-center mb-8">
            Prizes are automatically distributed to winners
          </p>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {tiers.map((tierInfo) => (
              <div key={tierInfo.tier} className="bg-gray-800 rounded-lg p-6">
                <h3 className="text-xl font-bold mb-4 text-center">{tierInfo.name}</h3>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-gray-400">1st Place:</span>
                    <span className="font-bold text-turquoise">
                      ${(tierInfo.prizePool * 0.5).toFixed(0)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-400">2nd Place:</span>
                    <span className="font-bold text-yellow-500">
                      ${(tierInfo.prizePool * 0.3).toFixed(0)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-400">3rd Place:</span>
                    <span className="font-bold text-orange-500">
                      ${(tierInfo.prizePool * 0.2).toFixed(0)}
                    </span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default ChallengeSponsorPricing;
