import { useState, useEffect } from 'react';
import { 
  FaTrophy, FaClock, FaUsers, FaDollarSign, FaTimes, FaCode, FaExternalLinkAlt,
  FaGithub, FaFileAlt, FaComments, FaPlay, FaCheckCircle, FaStar, FaAward,
  FaCalendarAlt, FaLink, FaDownload, FaSearch, FaFilter, FaTimesCircle
} from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

const Challenges = () => {
  const [challenges, setChallenges] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState('all');
  const [selectedChallenge, setSelectedChallenge] = useState(null);
  const [showFilters, setShowFilters] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedDifficulties, setSelectedDifficulties] = useState([]);
  const [selectedTags, setSelectedTags] = useState([]);
  const [prizeRange, setPrizeRange] = useState([0, 10000]);
  const [timeFilter, setTimeFilter] = useState('all');
  const [sponsoredOnly, setSponsoredOnly] = useState(false);
  const [liveParticipants, setLiveParticipants] = useState({});
  const [savedFilters, setSavedFilters] = useState([]);
  const [currentFilterName, setCurrentFilterName] = useState('');
  const [showSaveFilter, setShowSaveFilter] = useState(false);
  const [compareMode, setCompareMode] = useState(false);
  const [selectedForComparison, setSelectedForComparison] = useState([]);
  const [showCreateSponsored, setShowCreateSponsored] = useState(false);
  const { isAuthenticated, user } = useAuth();

  useEffect(() => {
    fetchChallenges();
    
    // Load saved filters from localStorage
    const saved = localStorage.getItem('challengeFilters');
    if (saved) setSavedFilters(JSON.parse(saved));
    
    // Real-time participant tracking simulation (in production, use WebSocket)
    const interval = setInterval(() => {
      challenges.forEach(challenge => {
        if (challenge.status === 'active') {
          setLiveParticipants(prev => ({
            ...prev,
            [challenge._id]: (prev[challenge._id] || challenge.participants || 0) + Math.floor(Math.random() * 3)
          }));
        }
      });
    }, 15000); // Update every 15 seconds
    
    return () => clearInterval(interval);
  }, [filter, challenges]);

  const fetchChallenges = async () => {
    try {
      const params = filter !== 'all' ? `?status=${filter}` : '';
      const response = await api.get(`/challenges${params}`);
      setChallenges(response.data.challenges || []);
    } catch (error) {
      console.error('Error fetching challenges:', error);
    } finally {
      setLoading(false);
    }
  };

  const filters = ['All', 'Active', 'Upcoming', 'Completed'];
  const difficulties = ['Easy', 'Medium', 'Hard'];
  const techTags = ['React', 'Node.js', 'Python', 'AI/ML', 'Blockchain', 'Mobile', 'DevOps', 'UI/UX'];
  const timeFilters = [
    { value: 'all', label: 'All Time' },
    { value: 'today', label: 'Ending Today' },
    { value: 'week', label: 'This Week' },
    { value: 'month', label: 'This Month' }
  ];

  // Filter challenges based on all criteria
  const getFilteredChallenges = () => {
    return challenges.filter(challenge => {
      // Search query
      if (searchQuery && !challenge.title.toLowerCase().includes(searchQuery.toLowerCase()) &&
          !challenge.description.toLowerCase().includes(searchQuery.toLowerCase())) {
        return false;
      }

      // Difficulty filter
      if (selectedDifficulties.length > 0 && !selectedDifficulties.includes(challenge.difficulty)) {
        return false;
      }

      // Sponsored filter
      if (sponsoredOnly && !challenge.isSponsored) {
        return false;
      }

      // Time filter
      if (timeFilter !== 'all') {
        const now = new Date();
        const endDate = new Date(challenge.endDate);
        const diff = endDate - now;
        const daysRemaining = diff / (1000 * 60 * 60 * 24);

        if (timeFilter === 'today' && daysRemaining > 1) return false;
        if (timeFilter === 'week' && daysRemaining > 7) return false;
        if (timeFilter === 'month' && daysRemaining > 30) return false;
      }

      return true;
    });
  };

  const filteredChallenges = getFilteredChallenges();

  // Count active filters
  const activeFilterCount = 
    selectedDifficulties.length + 
    selectedTags.length + 
    (searchQuery ? 1 : 0) + 
    (sponsoredOnly ? 1 : 0) + 
    (timeFilter !== 'all' ? 1 : 0);

  const clearAllFilters = () => {
    setSearchQuery('');
    setSelectedDifficulties([]);
    setSelectedTags([]);
    setPrizeRange([0, 10000]);
    setTimeFilter('all');
    setSponsoredOnly(false);
  };

  // Save current filter preset
  const saveFilterPreset = () => {
    if (!currentFilterName.trim()) return;
    const newFilter = {
      name: currentFilterName,
      filters: {
        searchQuery,
        selectedDifficulties,
        selectedTags,
        prizeRange,
        timeFilter,
        sponsoredOnly
      }
    };
    const updated = [...savedFilters, newFilter];
    setSavedFilters(updated);
    localStorage.setItem('challengeFilters', JSON.stringify(updated));
    setCurrentFilterName('');
    setShowSaveFilter(false);
  };

  // Load filter preset
  const loadFilterPreset = (preset) => {
    setSearchQuery(preset.filters.searchQuery);
    setSelectedDifficulties(preset.filters.selectedDifficulties);
    setSelectedTags(preset.filters.selectedTags);
    setPrizeRange(preset.filters.prizeRange);
    setTimeFilter(preset.filters.timeFilter);
    setSponsoredOnly(preset.filters.sponsoredOnly);
  };

  // Delete filter preset
  const deleteFilterPreset = (index) => {
    const updated = savedFilters.filter((_, i) => i !== index);
    setSavedFilters(updated);
    localStorage.setItem('challengeFilters', JSON.stringify(updated));
  };

  // Toggle challenge for comparison
  const toggleComparison = (challenge) => {
    if (selectedForComparison.find(c => c._id === challenge._id)) {
      setSelectedForComparison(selectedForComparison.filter(c => c._id !== challenge._id));
    } else if (selectedForComparison.length < 3) {
      setSelectedForComparison([...selectedForComparison, challenge]);
    }
  };

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-8">
          <h1 className="text-4xl md:text-5xl font-bold mb-4">
            Coding <span className="text-gradient">Challenges</span>
          </h1>
          <p className="text-xl text-gray-400 mb-6">
            Compete, learn, and win amazing prizes
          </p>
          
          {/* CTA Buttons */}
          <div className="flex justify-center gap-4">
            <button className="btn-primary flex items-center gap-2">
              <FaTrophy /> Browse Challenges
            </button>
            <button 
              onClick={() => setShowCreateSponsored(true)}
              className="px-6 py-3 bg-gold hover:bg-gold/80 text-vibe-black font-semibold rounded-lg transition-all flex items-center gap-2"
            >
              <FaStar /> Sponsor a Challenge
            </button>
          </div>
        </div>

        {/* Featured Sponsored Challenges Carousel */}
        {challenges.filter(c => c.isSponsored && c.status === 'active').length > 0 && (
          <div className="mb-12 relative">
            <div className="bg-gradient-to-r from-gold/20 via-purple/10 to-gold/20 border-2 border-gold/40 rounded-2xl p-8 shadow-2xl">
              <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 bg-gold text-black px-6 py-2 rounded-full font-bold text-sm shadow-lg">
                ⭐ FEATURED SPONSORED CHALLENGES ⭐
              </div>
              <div className="mt-4 grid md:grid-cols-2 gap-6">
                {challenges.filter(c => c.isSponsored && c.status === 'active').slice(0, 2).map(challenge => (
                  <div key={challenge._id} className="bg-gray-900/50 backdrop-blur-sm rounded-xl p-6 border border-gold/30 hover:border-gold/60 transition-all cursor-pointer"
                       onClick={() => setSelectedChallenge(challenge)}>
                    <div className="flex items-start gap-4 mb-4">
                      {challenge.sponsorLogo && (
                        <img src={challenge.sponsorLogo} alt={challenge.sponsorCompany} className="w-16 h-16 rounded-lg" />
                      )}
                      <div className="flex-1">
                        <h3 className="font-bold text-xl mb-1">{challenge.title}</h3>
                        <p className="text-sm text-gray-400">Sponsored by {challenge.sponsorCompany}</p>
                      </div>
                      <div className="text-right">
                        <div className="text-gold font-bold text-xl">{challenge.prize}</div>
                        <div className="text-xs text-gray-400">Prize Pool</div>
                      </div>
                    </div>
                    <p className="text-gray-300 text-sm mb-4 line-clamp-2">{challenge.description}</p>
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3 text-sm">
                        <span className="flex items-center gap-1 text-turquoise">
                          <FaUsers /> {liveParticipants[challenge._id] || challenge.participants || 0}
                          <span className="text-xs animate-pulse">🔴 LIVE</span>
                        </span>
                        <span className={`px-2 py-1 rounded text-xs ${
                          challenge.difficulty === 'Easy' ? 'bg-green-500/20 text-green-400' :
                          challenge.difficulty === 'Medium' ? 'bg-yellow-500/20 text-yellow-400' :
                          'bg-red-500/20 text-red-400'
                        }`}>
                          {challenge.difficulty}
                        </span>
                      </div>
                      <button className="btn-primary text-sm">View Details →</button>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Sponsor Showcase Banner */}
        <div className="mb-12 bg-gradient-to-r from-gold/10 via-transparent to-gold/10 border border-gold/30 rounded-xl p-6">
          <div className="flex flex-col md:flex-row items-center justify-between gap-6">
            <div className="flex-1">
              <h3 className="text-2xl font-bold mb-2">Become a Challenge Sponsor</h3>
              <p className="text-gray-400 mb-4">
                Connect with top developers, build your brand, and discover talented engineers for your team.
              </p>
              <ul className="space-y-2 text-sm text-gray-300">
                <li className="flex items-center gap-2">
                  <FaCheckCircle className="text-turquoise" /> Custom branded challenges with your company logo
                </li>
                <li className="flex items-center gap-2">
                  <FaCheckCircle className="text-turquoise" /> Direct access to challenge participants for hiring
                </li>
                <li className="flex items-center gap-2">
                  <FaCheckCircle className="text-turquoise" /> Analytics dashboard tracking engagement metrics
                </li>
              </ul>
            </div>
            <div className="flex flex-col gap-3">
              <button 
                onClick={() => setShowCreateSponsored(true)}
                className="btn-primary whitespace-nowrap"
              >
                Create Sponsored Challenge
              </button>
              <div className="text-center text-sm text-gray-400">Starting at $2,500</div>
            </div>
          </div>
        </div>

        {/* Leaderboard & Stats Section */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-12">
          {/* Hall of Fame */}
          <div className="card bg-gradient-to-br from-gold/10 to-transparent border-gold/30">
            <div className="flex items-center gap-3 mb-4">
              <FaTrophy className="text-gold text-3xl" />
              <div>
                <h3 className="font-bold text-lg">Hall of Fame</h3>
                <p className="text-xs text-gray-400">Top performers</p>
              </div>
            </div>
            <div className="space-y-2">
              {[
                { name: 'CodeMaster99', wins: 15, prize: '$45,000' },
                { name: 'DevQueen', wins: 12, prize: '$38,500' },
                { name: 'AlgoWiz', wins: 10, prize: '$32,000' }
              ].map((winner, idx) => (
                <div key={idx} className="flex items-center justify-between p-2 bg-gray-800/50 rounded">
                  <div className="flex items-center gap-2">
                    <span className="text-2xl">
                      {idx === 0 ? '🥇' : idx === 1 ? '🥈' : '🥉'}
                    </span>
                    <div>
                      <div className="font-semibold text-sm">{winner.name}</div>
                      <div className="text-xs text-gray-400">{winner.wins} wins</div>
                    </div>
                  </div>
                  <div className="text-gold font-bold text-sm">{winner.prize}</div>
                </div>
              ))}
            </div>
          </div>

          {/* Active Challenges Stats */}
          <div className="card bg-gradient-to-br from-turquoise/10 to-transparent border-turquoise/30">
            <div className="flex items-center gap-3 mb-4">
              <FaCode className="text-turquoise text-3xl" />
              <div>
                <h3 className="font-bold text-lg">Active Now</h3>
                <p className="text-xs text-gray-400">Join the competition</p>
              </div>
            </div>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="text-gray-400">Live Challenges</span>
                <span className="text-2xl font-bold text-turquoise">
                  {challenges.filter(c => c.status === 'active').length}
                </span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-gray-400">Total Participants</span>
                <span className="text-2xl font-bold text-turquoise">
                  {challenges.reduce((sum, c) => sum + (c.participants || 0), 0)}
                </span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-gray-400">Prize Pool</span>
                <span className="text-2xl font-bold text-gold">$150K+</span>
              </div>
            </div>
          </div>

          {/* Recent Winners */}
          <div className="card bg-gradient-to-br from-green-500/10 to-transparent border-green-500/30">
            <div className="flex items-center gap-3 mb-4">
              <FaAward className="text-green-400 text-3xl" />
              <div>
                <h3 className="font-bold text-lg">Recent Winners</h3>
                <p className="text-xs text-gray-400">Latest champions</p>
              </div>
            </div>
            <div className="space-y-2">
              {[
                { challenge: 'React Dashboard', winner: 'DevNinja', prize: '$5,000' },
                { challenge: 'AI Chatbot', winner: 'MLPro', prize: '$3,500' },
                { challenge: 'Mobile App', winner: 'AppDev42', prize: '$2,500' }
              ].map((recent, idx) => (
                <div key={idx} className="p-2 bg-gray-800/50 rounded">
                  <div className="flex items-center justify-between mb-1">
                    <span className="font-semibold text-sm">{recent.challenge}</span>
                    <span className="text-green-400 text-xs font-bold">{recent.prize}</span>
                  </div>
                  <div className="text-xs text-gray-400">Winner: {recent.winner}</div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-8">
          {/* Status Filter Tabs */}
          <div className="flex justify-center gap-4 mb-6">
            {filters.map((f) => (
              <button
                key={f}
                onClick={() => setFilter(f.toLowerCase())}
                className={`px-6 py-2 rounded-lg transition-colors ${
                  filter === f.toLowerCase()
                    ? 'bg-turquoise text-white'
                    : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
                }`}
              >
                {f}
              </button>
            ))}
          </div>

          {/* Search & Advanced Filters */}
          <div className="max-w-4xl mx-auto space-y-4">
            {/* Search Bar */}
            <div className="relative">
              <FaSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400" />
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search challenges by title or description..."
                className="w-full pl-12 pr-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
              />
            </div>

            {/* Filter Toggle Button & Actions */}
            <div className="flex items-center justify-between flex-wrap gap-3">
              <div className="flex items-center gap-3">
                <button
                  onClick={() => setShowFilters(!showFilters)}
                  className="flex items-center gap-2 px-4 py-2 bg-gray-800 hover:bg-gray-700 rounded-lg transition-colors"
                >
                  <FaFilter />
                  Advanced Filters
                  {activeFilterCount > 0 && (
                    <span className="bg-turquoise text-white text-xs px-2 py-0.5 rounded-full">
                      {activeFilterCount}
                    </span>
                  )}
                </button>

                {/* Comparison Mode Toggle */}
                <button
                  onClick={() => {
                    setCompareMode(!compareMode);
                    if (compareMode) setSelectedForComparison([]);
                  }}
                  className={`flex items-center gap-2 px-4 py-2 rounded-lg transition-all ${
                    compareMode 
                      ? 'bg-turquoise text-black font-semibold' 
                      : 'bg-gray-800 hover:bg-gray-700'
                  }`}
                >
                  <FaCode />
                  Compare {compareMode && `(${selectedForComparison.length}/3)`}
                </button>
              </div>

              <div className="flex items-center gap-3">
                {activeFilterCount > 0 && (
                  <>
                    <button
                      onClick={() => setShowSaveFilter(true)}
                      className="flex items-center gap-2 text-sm px-3 py-2 bg-turquoise/20 text-turquoise hover:bg-turquoise/30 rounded-lg transition-colors"
                    >
                      <FaStar /> Save Filter
                    </button>
                    <button
                      onClick={clearAllFilters}
                      className="flex items-center gap-2 text-sm text-gray-400 hover:text-white"
                    >
                      <FaTimesCircle /> Clear all
                    </button>
                  </>
                )}
              </div>
            </div>

            {/* Saved Filter Presets */}
            {savedFilters.length > 0 && (
              <div className="flex items-center gap-2 flex-wrap">
                <span className="text-sm text-gray-400">Quick Filters:</span>
                {savedFilters.map((preset, idx) => (
                  <div key={idx} className="flex items-center gap-1 bg-gray-800 rounded-lg overflow-hidden">
                    <button
                      onClick={() => loadFilterPreset(preset)}
                      className="px-3 py-1.5 text-sm hover:bg-gray-700 transition-colors"
                    >
                      {preset.name}
                    </button>
                    <button
                      onClick={() => deleteFilterPreset(idx)}
                      className="px-2 py-1.5 text-xs text-gray-400 hover:text-red-400 hover:bg-gray-700 transition-colors"
                    >
                      ✕
                    </button>
                  </div>
                ))}
              </div>
            )}

            {/* Advanced Filters Panel */}
            {showFilters && (
              <div className="bg-gray-800 border border-gray-700 rounded-lg p-6 space-y-6">
                {/* Difficulty */}
                <div>
                  <h3 className="text-sm font-semibold mb-3">Difficulty Level</h3>
                  <div className="flex flex-wrap gap-2">
                    {difficulties.map(diff => (
                      <button
                        key={diff}
                        onClick={() => {
                          if (selectedDifficulties.includes(diff)) {
                            setSelectedDifficulties(selectedDifficulties.filter(d => d !== diff));
                          } else {
                            setSelectedDifficulties([...selectedDifficulties, diff]);
                          }
                        }}
                        className={`px-4 py-2 rounded-lg transition-colors ${
                          selectedDifficulties.includes(diff)
                            ? diff === 'Easy' ? 'bg-green-500 text-white' :
                              diff === 'Medium' ? 'bg-yellow-500 text-white' :
                              'bg-red-500 text-white'
                            : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                        }`}
                      >
                        {diff}
                      </button>
                    ))}
                  </div>
                </div>

                {/* Tech Tags */}
                <div>
                  <h3 className="text-sm font-semibold mb-3">Technologies</h3>
                  <div className="flex flex-wrap gap-2">
                    {techTags.map(tag => (
                      <button
                        key={tag}
                        onClick={() => {
                          if (selectedTags.includes(tag)) {
                            setSelectedTags(selectedTags.filter(t => t !== tag));
                          } else {
                            setSelectedTags([...selectedTags, tag]);
                          }
                        }}
                        className={`px-3 py-1.5 rounded-lg text-sm transition-colors ${
                          selectedTags.includes(tag)
                            ? 'bg-turquoise text-white'
                            : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                        }`}
                      >
                        {tag}
                      </button>
                    ))}
                  </div>
                </div>

                {/* Time Remaining */}
                <div>
                  <h3 className="text-sm font-semibold mb-3">Time Remaining</h3>
                  <div className="flex flex-wrap gap-2">
                    {timeFilters.map(tf => (
                      <button
                        key={tf.value}
                        onClick={() => setTimeFilter(tf.value)}
                        className={`px-4 py-2 rounded-lg transition-colors ${
                          timeFilter === tf.value
                            ? 'bg-turquoise text-white'
                            : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                        }`}
                      >
                        {tf.label}
                      </button>
                    ))}
                  </div>
                </div>

                {/* Sponsored Toggle */}
                <div>
                  <label className="flex items-center gap-3 cursor-pointer">
                    <input
                      type="checkbox"
                      checked={sponsoredOnly}
                      onChange={(e) => setSponsoredOnly(e.target.checked)}
                      className="w-5 h-5 rounded border-gray-600 bg-gray-700 text-turquoise focus:ring-turquoise"
                    />
                    <span className="text-sm">Show sponsored challenges only</span>
                  </label>
                </div>
              </div>
            )}
          </div>
        </div>

        {/* Results Count */}
        <div className="mb-6 text-center text-gray-400">
          Showing <span className="text-white font-semibold">{filteredChallenges.length}</span> of {challenges.length} challenges
        </div>

        {/* Challenges Grid */}
        {loading ? (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise mx-auto"></div>
          </div>
        ) : filteredChallenges.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-400 text-lg">No challenges match your filters.</p>
            {activeFilterCount > 0 && (
              <button onClick={clearAllFilters} className="btn-primary mt-4">
                Clear Filters
              </button>
            )}
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredChallenges.map((challenge) => (
              <div key={challenge._id} className={`card hover:scale-105 transition-transform relative ${
                selectedForComparison.find(c => c._id === challenge._id) ? 'ring-2 ring-turquoise' : ''
              }`}>
                {/* Comparison Checkbox */}
                {compareMode && (
                  <div className="absolute top-4 left-4 z-10">
                    <input
                      type="checkbox"
                      checked={!!selectedForComparison.find(c => c._id === challenge._id)}
                      onChange={() => toggleComparison(challenge)}
                      disabled={selectedForComparison.length >= 3 && !selectedForComparison.find(c => c._id === challenge._id)}
                      className="w-5 h-5 rounded border-gray-600 bg-gray-700 text-turquoise focus:ring-turquoise cursor-pointer"
                    />
                  </div>
                )}
                
                {/* Challenge Header */}
                <div className="flex items-start justify-between mb-4">
                  <FaTrophy className="text-gold text-3xl" />
                  <div className="flex flex-col items-end gap-1">
                    {challenge.isSponsored && (
                      <span className="text-xs bg-gold/20 text-gold px-2 py-1 rounded">
                        SPONSORED
                      </span>
                    )}
                    {challenge.status === 'active' && liveParticipants[challenge._id] && (
                      <span className="text-xs bg-green-500/20 text-green-400 px-2 py-1 rounded flex items-center gap-1">
                        🔴 {liveParticipants[challenge._id]} live
                      </span>
                    )}
                  </div>
                </div>

                <h3 className="text-xl font-bold mb-2">{challenge.title}</h3>
                <p className="text-gray-400 mb-4 line-clamp-3">{challenge.description}</p>

                {/* Timeline Progress */}
                {(() => {
                  const now = new Date();
                  const start = new Date(challenge.startDate);
                  const end = new Date(challenge.endDate);
                  const total = end - start;
                  const elapsed = now - start;
                  const progress = Math.min(Math.max((elapsed / total) * 100, 0), 100);
                  const daysRemaining = Math.ceil((end - now) / (1000 * 60 * 60 * 24));

                  return (
                    <div className="mb-4">
                      <div className="flex items-center justify-between text-xs text-gray-400 mb-2">
                        <span>{challenge.status === 'upcoming' ? 'Starts soon' : daysRemaining > 0 ? `${daysRemaining} days left` : 'Ended'}</span>
                        <span>{Math.round(progress)}%</span>
                      </div>
                      <div className="w-full bg-gray-700 rounded-full h-2 overflow-hidden">
                        <div 
                          className={`h-full rounded-full transition-all ${
                            daysRemaining <= 1 ? 'bg-red-500' :
                            daysRemaining <= 3 ? 'bg-yellow-500' :
                            'bg-turquoise'
                          }`}
                          style={{ width: `${progress}%` }}
                        />
                      </div>
                      {daysRemaining <= 3 && daysRemaining > 0 && (
                        <div className="mt-2 text-xs text-red-400 flex items-center gap-1 animate-pulse">
                          <FaClock /> Ending soon!
                        </div>
                      )}
                    </div>
                  );
                })()}

                {/* Challenge Info */}
                <div className="space-y-2 mb-4">
                  <div className="flex items-center gap-2 text-sm text-gray-400">
                    <FaClock className="text-turquoise" />
                    <span>Ends: {new Date(challenge.endDate).toLocaleDateString()}</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm text-gray-400">
                    <FaUsers className="text-turquoise" />
                    <span>{liveParticipants[challenge._id] || challenge.participants || 0} participants</span>
                    {challenge.status === 'active' && (
                      <span className="text-xs text-green-400 animate-pulse">●</span>
                    )}
                  </div>
                  {challenge.prize && (
                    <div className="flex items-center gap-2 text-sm text-gold font-semibold">
                      <FaDollarSign />
                      <span>{challenge.prize} in prizes</span>
                    </div>
                  )}
                </div>

                {/* Difficulty */}
                <div className="mb-4">
                  <span
                    className={`text-xs px-3 py-1 rounded-full ${
                      challenge.difficulty === 'Easy'
                        ? 'bg-green-500/20 text-green-400'
                        : challenge.difficulty === 'Medium'
                        ? 'bg-yellow-500/20 text-yellow-400'
                        : 'bg-red-500/20 text-red-400'
                    }`}
                  >
                    {challenge.difficulty}
                  </span>
                </div>

                {/* CTA */}
                <button 
                  onClick={() => setSelectedChallenge(challenge)}
                  className="w-full btn-primary"
                >
                  View Challenge
                </button>
              </div>
            ))}
          </div>
        )}

        {/* Challenge Detail Modal */}
        {selectedChallenge && (
          <ChallengeDetailModal
            challenge={selectedChallenge}
            onClose={() => setSelectedChallenge(null)}
            isAuthenticated={isAuthenticated}
            currentUser={user}
          />
        )}

        {/* Save Filter Modal */}
        {showSaveFilter && (
          <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={() => setShowSaveFilter(false)}>
            <div className="bg-gray-900 rounded-xl p-6 max-w-md w-full border border-gray-700" onClick={e => e.stopPropagation()}>
              <h3 className="text-xl font-bold mb-4">Save Filter Preset</h3>
              <input
                type="text"
                value={currentFilterName}
                onChange={(e) => setCurrentFilterName(e.target.value)}
                placeholder="Enter preset name (e.g., 'Easy React Challenges')"
                className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 mb-4 focus:outline-none focus:border-turquoise"
                autoFocus
              />
              <div className="flex gap-3">
                <button onClick={saveFilterPreset} className="btn-primary flex-1">Save</button>
                <button onClick={() => setShowSaveFilter(false)} className="btn-secondary flex-1">Cancel</button>
              </div>
            </div>
          </div>
        )}

        {/* Challenge Comparison Modal */}
        {compareMode && selectedForComparison.length >= 2 && (
          <div className="fixed bottom-0 left-0 right-0 bg-gray-900 border-t-2 border-turquoise p-6 z-40 shadow-2xl">
            <div className="max-w-7xl mx-auto">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-xl font-bold">Challenge Comparison ({selectedForComparison.length})</h3>
                <button onClick={() => { setCompareMode(false); setSelectedForComparison([]); }} className="text-gray-400 hover:text-white">
                  <FaTimes size={20} />
                </button>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {selectedForComparison.map(challenge => (
                  <div key={challenge._id} className="bg-gray-800 rounded-lg p-4 border border-gray-700">
                    <div className="flex items-start justify-between mb-3">
                      <h4 className="font-bold text-lg">{challenge.title}</h4>
                      <button onClick={() => toggleComparison(challenge)} className="text-red-400 hover:text-red-300">
                        <FaTimes />
                      </button>
                    </div>
                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between">
                        <span className="text-gray-400">Difficulty:</span>
                        <span className={challenge.difficulty === 'Easy' ? 'text-green-400' : challenge.difficulty === 'Medium' ? 'text-yellow-400' : 'text-red-400'}>
                          {challenge.difficulty}
                        </span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-400">Prize:</span>
                        <span className="text-gold font-bold">{challenge.prize}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-400">Participants:</span>
                        <span className="text-turquoise">{liveParticipants[challenge._id] || challenge.participants || 0}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-400">Ends:</span>
                        <span className="text-gray-300">{new Date(challenge.endDate).toLocaleDateString()}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-400">Status:</span>
                        <span className={challenge.status === 'active' ? 'text-green-400' : 'text-gray-400'}>
                          {challenge.status.toUpperCase()}
                        </span>
                      </div>
                    </div>
                    <button onClick={() => setSelectedChallenge(challenge)} className="w-full btn-primary text-sm mt-4">
                      View Details
                    </button>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Create Sponsored Challenge Modal */}
        {showCreateSponsored && (
          <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4 overflow-y-auto" onClick={() => setShowCreateSponsored(false)}>
            <div className="bg-gray-900 rounded-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto border-2 border-gold/40" onClick={e => e.stopPropagation()}>
              <div className="sticky top-0 bg-gray-900 border-b border-gold/30 p-6 z-10">
                <div className="flex items-start justify-between">
                  <div>
                    <h2 className="text-3xl font-bold mb-2 flex items-center gap-3">
                      <FaStar className="text-gold" />
                      Create Sponsored Challenge
                    </h2>
                    <p className="text-gray-400">Connect with top developers and build your brand</p>
                  </div>
                  <button onClick={() => setShowCreateSponsored(false)} className="text-gray-400 hover:text-white">
                    <FaTimes size={24} />
                  </button>
                </div>
              </div>
              
              <div className="p-6 space-y-6">
                {/* Sponsor Benefits */}
                <div className="grid md:grid-cols-3 gap-4">
                  <div className="bg-turquoise/10 border border-turquoise/30 p-4 rounded-lg">
                    <FaUsers className="text-turquoise text-3xl mb-2" />
                    <h3 className="font-bold mb-1">Reach Thousands</h3>
                    <p className="text-sm text-gray-400">Avg. 5,000+ developers per challenge</p>
                  </div>
                  <div className="bg-gold/10 border border-gold/30 p-4 rounded-lg">
                    <FaTrophy className="text-gold text-3xl mb-2" />
                    <h3 className="font-bold mb-1">Top Talent</h3>
                    <p className="text-sm text-gray-400">Direct access to challenge winners</p>
                  </div>
                  <div className="bg-purple/10 border border-purple/30 p-4 rounded-lg">
                    <FaAward className="text-purple text-3xl mb-2" />
                    <h3 className="font-bold mb-1">Brand Visibility</h3>
                    <p className="text-sm text-gray-400">Premium placement & analytics</p>
                  </div>
                </div>

                {/* Pricing Tiers */}
                <div>
                  <h3 className="text-xl font-bold mb-4">Choose Your Package</h3>
                  <div className="grid md:grid-cols-3 gap-4">
                    {[
                      { name: 'Starter', price: '$2,500', features: ['1 month duration', '1,000-2,000 participants', 'Basic analytics', 'Logo placement'] },
                      { name: 'Professional', price: '$5,000', features: ['2 month duration', '3,000-5,000 participants', 'Advanced analytics', 'Featured placement', 'Hiring dashboard'], highlight: true },
                      { name: 'Enterprise', price: '$10,000+', features: ['Custom duration', '10,000+ participants', 'Full analytics suite', 'Premium placement', 'Dedicated support', 'Custom integration'] }
                    ].map((tier, idx) => (
                      <div key={idx} className={`p-6 rounded-xl border-2 ${
                        tier.highlight 
                          ? 'bg-gradient-to-br from-gold/20 to-purple/20 border-gold' 
                          : 'bg-gray-800 border-gray-700'
                      }`}>
                        {tier.highlight && (
                          <div className="bg-gold text-black text-xs font-bold px-3 py-1 rounded-full inline-block mb-3">
                            MOST POPULAR
                          </div>
                        )}
                        <h4 className="text-2xl font-bold mb-2">{tier.name}</h4>
                        <div className="text-3xl font-bold text-gold mb-4">{tier.price}</div>
                        <ul className="space-y-2 mb-6">
                          {tier.features.map((feature, i) => (
                            <li key={i} className="flex items-start gap-2 text-sm">
                              <FaCheckCircle className="text-green-400 mt-0.5 flex-shrink-0" />
                              <span>{feature}</span>
                            </li>
                          ))}
                        </ul>
                        <button className={tier.highlight ? 'btn-primary w-full' : 'btn-secondary w-full'}>
                          Get Started
                        </button>
                      </div>
                    ))}
                  </div>
                </div>

                {/* Contact Form */}
                <div className="bg-gray-800 p-6 rounded-lg">
                  <h3 className="text-xl font-bold mb-4">Request Sponsored Challenge</h3>
                  <form className="space-y-4" onSubmit={async (e) => {
                    e.preventDefault();
                    const formData = new FormData(e.target);
                    const data = {
                      packageTier: 'professional',
                      companyName: formData.get('companyName'),
                      companyEmail: formData.get('companyEmail'),
                      challengeTitle: formData.get('challengeTitle'),
                      challengeGoal: formData.get('challengeGoal')
                    };
                    
                    if (!isAuthenticated) {
                      alert('Please sign in to create a sponsored challenge');
                      return;
                    }
                    
                    try {
                      const response = await api.post('/stripe/create-sponsored-challenge-checkout', data);
                      if (response.data.url) {
                        window.location.href = response.data.url;
                      }
                    } catch (error) {
                      alert(error.response?.data?.message || 'Failed to create checkout session');
                    }
                  }}>
                    <div className="grid md:grid-cols-2 gap-4">
                      <input 
                        type="text" 
                        name="companyName"
                        placeholder="Company Name *" 
                        required
                        className="px-4 py-3 bg-gray-900 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise" 
                      />
                      <input 
                        type="email" 
                        name="companyEmail"
                        placeholder="Work Email *" 
                        required
                        className="px-4 py-3 bg-gray-900 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise" 
                      />
                    </div>
                    <input 
                      type="text" 
                      name="challengeTitle"
                      placeholder="Challenge Title / Idea" 
                      className="w-full px-4 py-3 bg-gray-900 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise" 
                    />
                    <textarea 
                      name="challengeGoal"
                      placeholder="Challenge Goal / Additional Details" 
                      rows="3" 
                      className="w-full px-4 py-3 bg-gray-900 border border-gray-700 rounded-lg text-gray-300 resize-none focus:outline-none focus:border-turquoise" 
                    />
                    <button type="submit" className="btn-primary w-full flex items-center justify-center gap-2">
                      <FaCheckCircle /> Continue to Payment
                    </button>
                  </form>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

// Challenge Detail Modal Component
const ChallengeDetailModal = ({ challenge, onClose, isAuthenticated, currentUser }) => {
  const [activeTab, setActiveTab] = useState('overview');
  const [submissionForm, setSubmissionForm] = useState({
    repoUrl: '',
    liveUrl: '',
    description: ''
  });
  const [hasSubmitted, setHasSubmitted] = useState(false);

  const handleSubmit = async (e) => {
    e.preventDefault();
    if (!isAuthenticated) {
      alert('Please sign in to submit');
      return;
    }
    try {
      await api.post(`/challenges/${challenge._id}/submit`, submissionForm);
      alert('Submission successful!');
      setHasSubmitted(true);
      setSubmissionForm({ repoUrl: '', liveUrl: '', description: '' });
    } catch (error) {
      alert(error.response?.data?.message || 'Submission failed');
    }
  };

  const timeRemaining = () => {
    const now = new Date();
    const end = new Date(challenge.endDate);
    const diff = end - now;
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    return { days, hours, isActive: diff > 0 };
  };

  const time = timeRemaining();

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4 overflow-y-auto" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-5xl w-full max-h-[90vh] overflow-y-auto border border-gray-700" onClick={e => e.stopPropagation()}>
        {/* Header */}
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 z-10">
          <div className="flex items-start justify-between mb-4">
            <div className="flex-1">
              <div className="flex items-center gap-3 mb-3">
                <FaTrophy className="text-gold text-3xl" />
                <div>
                  <h2 className="text-3xl font-bold">{challenge.title}</h2>
                  <p className="text-gray-400 text-sm">
                    {challenge.participants || 0} participants • {challenge.views || 0} views
                  </p>
                </div>
              </div>
              
              {/* Status Badges */}
              <div className="flex flex-wrap gap-2">
                <span className={`text-xs px-3 py-1 rounded-full ${
                  challenge.difficulty === 'Easy' ? 'bg-green-500/20 text-green-400' :
                  challenge.difficulty === 'Medium' ? 'bg-yellow-500/20 text-yellow-400' :
                  'bg-red-500/20 text-red-400'
                }`}>
                  {challenge.difficulty}
                </span>
                {challenge.isSponsored && (
                  <span className="text-xs bg-gold/20 text-gold px-3 py-1 rounded-full">
                    SPONSORED
                  </span>
                )}
                <span className={`text-xs px-3 py-1 rounded-full ${
                  challenge.status === 'active' ? 'bg-green-500/20 text-green-400' :
                  challenge.status === 'upcoming' ? 'bg-blue-500/20 text-blue-400' :
                  'bg-gray-500/20 text-gray-400'
                }`}>
                  {challenge.status.toUpperCase()}
                </span>
              </div>
            </div>
            <button onClick={onClose} className="text-gray-400 hover:text-white p-2">
              <FaTimes size={24} />
            </button>
          </div>

          {/* Time & Prize Info */}
          <div className="grid grid-cols-3 gap-4 mt-4">
            <div className="bg-gray-800 p-3 rounded-lg">
              <div className="flex items-center gap-2 text-turquoise mb-1">
                <FaClock />
                <span className="text-xs font-semibold">Time Remaining</span>
              </div>
              <div className="text-xl font-bold">
                {time.isActive ? `${time.days}d ${time.hours}h` : 'Ended'}
              </div>
            </div>
            {challenge.prize && (
              <div className="bg-gray-800 p-3 rounded-lg">
                <div className="flex items-center gap-2 text-gold mb-1">
                  <FaDollarSign />
                  <span className="text-xs font-semibold">Prize Pool</span>
                </div>
                <div className="text-xl font-bold">{challenge.prize}</div>
              </div>
            )}
            <div className="bg-gray-800 p-3 rounded-lg">
              <div className="flex items-center gap-2 text-purple-400 mb-1">
                <FaCalendarAlt />
                <span className="text-xs font-semibold">Deadline</span>
              </div>
              <div className="text-sm font-bold">
                {new Date(challenge.endDate).toLocaleDateString()}
              </div>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-800 px-6">
          <div className="flex gap-4">
            {['overview', 'submit', 'leaderboard', 'submissions', 'discussion'].map(tab => (
              <button
                key={tab}
                onClick={() => setActiveTab(tab)}
                className={`py-3 px-4 border-b-2 transition-colors ${
                  activeTab === tab
                    ? 'border-turquoise text-turquoise'
                    : 'border-transparent text-gray-400 hover:text-white'
                }`}
              >
                {tab.charAt(0).toUpperCase() + tab.slice(1)}
              </button>
            ))}
          </div>
        </div>

        {/* Content */}
        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              {/* Challenge Timeline */}
              <div>
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                  <FaCalendarAlt className="text-turquoise" />
                  Challenge Timeline
                </h3>
                <div className="bg-gray-800 p-6 rounded-lg">
                  <div className="relative">
                    {/* Timeline Line */}
                    <div className="absolute left-6 top-0 bottom-0 w-0.5 bg-gray-700"></div>
                    
                    {/* Phase Milestones */}
                    <div className="space-y-6">
                      {[
                        { 
                          phase: 'Registration Opens', 
                          date: challenge.startDate, 
                          status: new Date() > new Date(challenge.startDate) ? 'completed' : 'upcoming',
                          icon: '📝'
                        },
                        { 
                          phase: 'Submission Deadline', 
                          date: challenge.endDate, 
                          status: challenge.status === 'completed' ? 'completed' : 
                                 new Date() > new Date(challenge.endDate) ? 'completed' : 
                                 new Date() > new Date(challenge.startDate) ? 'active' : 'upcoming',
                          icon: '🚀'
                        },
                        { 
                          phase: 'Judging Period', 
                          date: new Date(new Date(challenge.endDate).getTime() + 7 * 24 * 60 * 60 * 1000), 
                          status: challenge.status === 'completed' ? 'completed' : 'upcoming',
                          icon: '⚖️'
                        },
                        { 
                          phase: 'Winners Announced', 
                          date: new Date(new Date(challenge.endDate).getTime() + 14 * 24 * 60 * 60 * 1000), 
                          status: challenge.status === 'completed' && challenge.winners ? 'completed' : 'upcoming',
                          icon: '🏆'
                        }
                      ].map((milestone, idx) => (
                        <div key={idx} className="relative flex items-start gap-4 pl-0">
                          <div className={`relative z-10 w-12 h-12 rounded-full flex items-center justify-center text-2xl ${
                            milestone.status === 'completed' ? 'bg-green-500/20 border-2 border-green-500' :
                            milestone.status === 'active' ? 'bg-turquoise/20 border-2 border-turquoise animate-pulse' :
                            'bg-gray-700 border-2 border-gray-600'
                          }`}>
                            {milestone.status === 'completed' ? '✓' : milestone.icon}
                          </div>
                          <div className="flex-1">
                            <div className={`font-semibold ${
                              milestone.status === 'completed' ? 'text-green-400' :
                              milestone.status === 'active' ? 'text-turquoise' :
                              'text-gray-400'
                            }`}>
                              {milestone.phase}
                            </div>
                            <div className="text-sm text-gray-400">
                              {new Date(milestone.date).toLocaleDateString('en-US', { 
                                month: 'long', 
                                day: 'numeric', 
                                year: 'numeric' 
                              })}
                            </div>
                            {milestone.status === 'active' && (
                              <div className="text-xs text-turquoise mt-1">← You are here</div>
                            )}
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>

              {/* Description */}
              <div>
                <h3 className="text-xl font-semibold mb-3">Description</h3>
                <p className="text-gray-300 leading-relaxed">{challenge.description}</p>
              </div>

              {/* Requirements */}
              <div>
                <h3 className="text-xl font-semibold mb-3 flex items-center gap-2">
                  <FaFileAlt className="text-turquoise" />
                  Requirements
                </h3>
                <div className="bg-gray-800 p-4 rounded-lg">
                  <p className="text-gray-300 whitespace-pre-line">{challenge.requirements}</p>
                </div>
              </div>

              {/* Prize Distribution */}
              {challenge.prizeDistribution && challenge.prizeDistribution.length > 0 && (
                <div>
                  <h3 className="text-xl font-semibold mb-3 flex items-center gap-2">
                    <FaAward className="text-gold" />
                    Prize Distribution
                  </h3>
                  <div className="space-y-2">
                    {challenge.prizeDistribution.map((prize, idx) => (
                      <div key={idx} className="bg-gray-800 p-3 rounded-lg flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          <span className="text-2xl">
                            {prize.place === 1 ? '🥇' : prize.place === 2 ? '🥈' : '🥉'}
                          </span>
                          <div>
                            <div className="font-semibold">{prize.place === 1 ? '1st' : prize.place === 2 ? '2nd' : '3rd'} Place</div>
                            <div className="text-sm text-gray-400">{prize.description}</div>
                          </div>
                        </div>
                        <div className="text-xl font-bold text-gold">${prize.amount}</div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Sponsor Info */}
              {challenge.isSponsored && challenge.sponsorCompany && (
                <div>
                  <h3 className="text-xl font-semibold mb-3">Sponsored By</h3>
                  <div className="bg-gradient-to-r from-gold/10 to-transparent p-6 rounded-lg border border-gold/30">
                    <div className="flex items-start gap-4 mb-6">
                      {challenge.sponsorLogo && (
                        <img src={challenge.sponsorLogo} alt={challenge.sponsorCompany} className="w-20 h-20 rounded-lg" />
                      )}
                      <div className="flex-1">
                        <h4 className="font-bold text-2xl mb-2">{challenge.sponsorCompany}</h4>
                        <p className="text-gray-300 mb-3">{challenge.sponsorDescription}</p>
                        <div className="flex gap-3">
                          {challenge.sponsorWebsite && (
                            <a 
                              href={challenge.sponsorWebsite} 
                              target="_blank" 
                              rel="noopener noreferrer"
                              className="px-4 py-2 bg-gray-800 hover:bg-gray-700 rounded-lg text-sm flex items-center gap-2 transition-colors"
                            >
                              Visit Website <FaExternalLinkAlt size={12} />
                            </a>
                          )}
                        </div>
                      </div>
                    </div>

                    {/* Hiring Opportunities */}
                    <div className="bg-gray-800 p-4 rounded-lg">
                      <div className="flex items-start justify-between mb-3">
                        <div>
                          <h4 className="font-semibold text-lg mb-1 flex items-center gap-2">
                            <FaUsers className="text-turquoise" />
                            Hiring Opportunity
                          </h4>
                          <p className="text-sm text-gray-400">
                            Top performers may be invited to interview with {challenge.sponsorCompany}
                          </p>
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-3">
                        <div className="bg-gray-900 p-3 rounded">
                          <div className="text-xs text-gray-400">Open Positions</div>
                          <div className="text-xl font-bold text-turquoise">12+</div>
                        </div>
                        <div className="bg-gray-900 p-3 rounded">
                          <div className="text-xs text-gray-400">Avg. Salary</div>
                          <div className="text-xl font-bold text-gold">$120K+</div>
                        </div>
                      </div>
                      <button className="w-full mt-4 btn-primary">
                        View Open Positions
                      </button>
                    </div>
                  </div>
                </div>
              )}

              {/* Resources */}
              <div>
                <h3 className="text-xl font-semibold mb-3 flex items-center gap-2">
                  <FaDownload className="text-turquoise" />
                  Resources & Starter Templates
                </h3>
                <div className="grid grid-cols-2 gap-3">
                  <button className="btn-secondary flex items-center justify-center gap-2">
                    <FaGithub /> Starter Code
                  </button>
                  <button className="btn-secondary flex items-center justify-center gap-2">
                    <FaFileAlt /> Documentation
                  </button>
                </div>
              </div>
            </div>
          )}

          {activeTab === 'submit' && (
            <div className="space-y-4">
              {!isAuthenticated ? (
                <div className="text-center py-12 bg-gray-800 rounded-lg">
                  <FaCheckCircle className="text-gray-600 text-5xl mx-auto mb-4" />
                  <p className="text-gray-400 mb-4">Please sign in to submit your solution</p>
                  <button className="btn-primary">Sign In</button>
                </div>
              ) : hasSubmitted ? (
                <div className="text-center py-12 bg-green-900/20 border border-green-500/30 rounded-lg">
                  <FaCheckCircle className="text-green-400 text-5xl mx-auto mb-4" />
                  <h3 className="text-2xl font-bold mb-2">Submission Received!</h3>
                  <p className="text-gray-300">Your solution has been submitted successfully.</p>
                </div>
              ) : (
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="bg-blue-500/10 border border-blue-500/30 rounded-lg p-4">
                    <p className="text-sm text-blue-300">
                      <FaCheckCircle className="inline mr-2" />
                      Make sure your submission meets all requirements before submitting.
                    </p>
                  </div>

                  <div>
                    <label className="block text-sm font-semibold mb-2 flex items-center gap-2">
                      <FaGithub className="text-turquoise" />
                      GitHub Repository URL *
                    </label>
                    <input
                      type="url"
                      required
                      value={submissionForm.repoUrl}
                      onChange={(e) => setSubmissionForm({ ...submissionForm, repoUrl: e.target.value })}
                      placeholder="https://github.com/username/repo"
                      className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-semibold mb-2 flex items-center gap-2">
                      <FaExternalLinkAlt className="text-turquoise" />
                      Live Demo URL
                    </label>
                    <input
                      type="url"
                      value={submissionForm.liveUrl}
                      onChange={(e) => setSubmissionForm({ ...submissionForm, liveUrl: e.target.value })}
                      placeholder="https://your-demo.com"
                      className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-semibold mb-2">Description</label>
                    <textarea
                      value={submissionForm.description}
                      onChange={(e) => setSubmissionForm({ ...submissionForm, description: e.target.value })}
                      placeholder="Brief description of your solution..."
                      rows="4"
                      className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 resize-none focus:outline-none focus:border-turquoise"
                    />
                  </div>

                  <button type="submit" className="w-full btn-primary flex items-center justify-center gap-2">
                    <FaCheckCircle /> Submit Solution
                  </button>
                </form>
              )}
            </div>
          )}

          {activeTab === 'leaderboard' && (
            <div className="space-y-6">
              {/* Judging Criteria */}
              <div className="bg-blue-500/10 border border-blue-500/30 rounded-lg p-4">
                <h3 className="font-semibold mb-3 flex items-center gap-2">
                  <FaCheckCircle className="text-blue-400" />
                  Judging Criteria
                </h3>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                  <div>
                    <div className="text-gray-400">Code Quality</div>
                    <div className="font-bold text-white">30%</div>
                  </div>
                  <div>
                    <div className="text-gray-400">Functionality</div>
                    <div className="font-bold text-white">30%</div>
                  </div>
                  <div>
                    <div className="text-gray-400">Creativity</div>
                    <div className="font-bold text-white">25%</div>
                  </div>
                  <div>
                    <div className="text-gray-400">Community Vote</div>
                    <div className="font-bold text-white">15%</div>
                  </div>
                </div>
              </div>

              {/* Top 10 Leaderboard */}
              <div>
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <FaTrophy className="text-gold" />
                  Current Rankings
                </h3>
                <div className="space-y-2">
                  {(challenge.submissions?.slice(0, 10) || [
                    { user: { username: 'Loading...' }, score: 0, votes: 0 }
                  ]).map((submission, idx) => (
                    <div 
                      key={idx} 
                      className={`p-4 rounded-lg flex items-center justify-between ${
                        idx === 0 ? 'bg-gradient-to-r from-gold/20 to-transparent border border-gold/30' :
                        idx === 1 ? 'bg-gradient-to-r from-gray-400/10 to-transparent border border-gray-400/20' :
                        idx === 2 ? 'bg-gradient-to-r from-orange-600/10 to-transparent border border-orange-600/20' :
                        'bg-gray-800'
                      }`}
                    >
                      <div className="flex items-center gap-4">
                        <div className="text-3xl font-bold w-12 text-center">
                          {idx === 0 ? '🥇' : idx === 1 ? '🥈' : idx === 2 ? '🥉' : `#${idx + 1}`}
                        </div>
                        <div className="flex items-center gap-3">
                          <div className="w-12 h-12 bg-gradient-to-br from-turquoise to-gold rounded-full flex items-center justify-center font-bold text-lg">
                            {submission.user?.username?.[0]?.toUpperCase() || 'U'}
                          </div>
                          <div>
                            <div className="font-bold">{submission.user?.username || 'Anonymous'}</div>
                            <div className="flex items-center gap-3 text-xs text-gray-400">
                              <span className="flex items-center gap-1">
                                <FaStar className="text-gold" /> {submission.votes || Math.floor(Math.random() * 50)} votes
                              </span>
                            </div>
                          </div>
                        </div>
                      </div>
                      <div className="text-right">
                        <div className="text-2xl font-bold text-turquoise">
                          {submission.score || (95 - idx * 5)}
                        </div>
                        <div className="text-xs text-gray-400">points</div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              {/* Achievement Badges */}
              {challenge.status === 'completed' && challenge.winners && (
                <div>
                  <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                    <FaAward className="text-gold" />
                    Special Awards
                  </h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    {[
                      { badge: '💡', title: 'Most Innovative', winner: 'CodeGenius' },
                      { badge: '⚡', title: 'Best Performance', winner: 'SpeedDev' },
                      { badge: '🎨', title: 'Best Design', winner: 'UIArtist' }
                    ].map((award, idx) => (
                      <div key={idx} className="bg-gray-800 p-4 rounded-lg text-center">
                        <div className="text-4xl mb-2">{award.badge}</div>
                        <div className="font-semibold text-sm mb-1">{award.title}</div>
                        <div className="text-xs text-turquoise">{award.winner}</div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          )}

          {activeTab === 'submissions' && (
            <div className="space-y-4">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold">
                  All Submissions ({challenge.submissions?.length || 0})
                </h3>
                <select className="px-3 py-2 bg-gray-800 border border-gray-700 rounded-lg text-sm">
                  <option>Most Recent</option>
                  <option>Most Voted</option>
                  <option>Top Rated</option>
                </select>
              </div>

              {challenge.submissions && challenge.submissions.length > 0 ? (
                <div className="space-y-3">
                  {challenge.submissions.map((submission, idx) => (
                    <div key={idx} className="bg-gray-800 p-4 rounded-lg hover:bg-gray-750 transition-colors">
                      <div className="flex items-start justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <div className="w-10 h-10 bg-gradient-to-br from-turquoise to-gold rounded-full flex items-center justify-center font-bold">
                            {submission.user?.username?.[0]?.toUpperCase() || 'U'}
                          </div>
                          <div>
                            <div className="font-semibold">{submission.user?.username || 'Anonymous'}</div>
                            <div className="text-xs text-gray-400">
                              {new Date(submission.createdAt).toLocaleDateString()}
                            </div>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <button className="text-gray-400 hover:text-turquoise">
                            <FaStar />
                          </button>
                          <span className="text-sm text-gray-400">{submission.votes || 0}</span>
                        </div>
                      </div>
                      
                      {submission.description && (
                        <p className="text-gray-300 text-sm mb-3">{submission.description}</p>
                      )}

                      <div className="flex gap-2">
                        {submission.repoUrl && (
                          <a 
                            href={submission.repoUrl} 
                            target="_blank" 
                            rel="noopener noreferrer"
                            className="flex items-center gap-2 px-3 py-1.5 bg-gray-900 hover:bg-gray-700 rounded text-sm transition-colors"
                          >
                            <FaGithub /> Code
                          </a>
                        )}
                        {submission.liveUrl && (
                          <a 
                            href={submission.liveUrl} 
                            target="_blank" 
                            rel="noopener noreferrer"
                            className="flex items-center gap-2 px-3 py-1.5 bg-gray-900 hover:bg-gray-700 rounded text-sm transition-colors"
                          >
                            <FaPlay /> Live Demo
                          </a>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-12 bg-gray-800 rounded-lg">
                  <FaCode className="text-gray-600 text-5xl mx-auto mb-4" />
                  <p className="text-gray-400">No submissions yet. Be the first!</p>
                </div>
              )}
            </div>
          )}

          {activeTab === 'discussion' && (
            <div className="space-y-4">
              <div className="bg-gray-800 p-4 rounded-lg">
                <textarea 
                  placeholder="Ask a question or share your thoughts..."
                  className="w-full bg-gray-900 border border-gray-700 rounded-lg p-3 text-gray-300 resize-none focus:outline-none focus:border-turquoise"
                  rows="3"
                />
                <div className="flex justify-end mt-2">
                  <button className="btn-primary text-sm">Post Comment</button>
                </div>
              </div>
              
              <div className="text-center py-12 bg-gray-800 rounded-lg">
                <FaComments className="text-gray-600 text-5xl mx-auto mb-4" />
                <p className="text-gray-400">No comments yet. Start the discussion!</p>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Challenges;
