import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

const Checkout = () => {
  const navigate = useNavigate();
  const { user } = useAuth();
  const [cart, setCart] = useState(null);
  const [loading, setLoading] = useState(true);
  const [processing, setProcessing] = useState(false);

  useEffect(() => {
    fetchCart();
  }, []);

  const fetchCart = async () => {
    try {
      const response = await api.get('/cart');
      setCart(response.data.cart);
    } catch (error) {
      console.error('Failed to fetch cart:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleCheckout = async () => {
    if (!cart || cart.items.length === 0) {
      return;
    }

    try {
      setProcessing(true);
      const productIds = cart.items.map(item => item.product._id);
      
      const response = await api.post('/stripe/create-product-checkout', {
        cartItems: productIds,
      });

      // Redirect to Stripe checkout
      window.location.href = response.data.url;
    } catch (error) {
      console.error('Checkout error:', error);
      alert(error.response?.data?.message || 'Checkout failed');
      setProcessing(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  if (!cart || cart.items.length === 0) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold mb-4">Your cart is empty</h2>
          <button onClick={() => navigate('/marketplace')} className="btn-primary">
            Browse Marketplace
          </button>
        </div>
      </div>
    );
  }

  const total = cart.items.reduce((sum, item) => sum + (item.product?.price || 0), 0);

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-3xl mx-auto">
        <h1 className="text-4xl font-bold mb-8 text-center">
          <span className="text-gradient">Checkout</span>
        </h1>

        <div className="card p-8">
          {/* Order Summary */}
          <h2 className="text-2xl font-bold mb-6">Order Summary</h2>

          <div className="space-y-4 mb-6">
            {cart.items.map(item => (
              <div key={item._id} className="flex justify-between items-center pb-4 border-b border-gray-700">
                <div>
                  <h3 className="font-medium">{item.product?.title}</h3>
                  <p className="text-sm text-gray-400">by {item.product?.creator?.username}</p>
                </div>
                <span className="font-bold text-turquoise">${item.product?.price.toFixed(2)}</span>
              </div>
            ))}
          </div>

          {/* Total */}
          <div className="flex justify-between items-center text-xl font-bold mb-8 pt-4 border-t-2 border-gray-700">
            <span>Total</span>
            <span className="text-turquoise">${total.toFixed(2)}</span>
          </div>

          {/* Checkout Button */}
          <button
            onClick={handleCheckout}
            disabled={processing}
            className="w-full btn-primary text-lg py-4 disabled:opacity-50"
          >
            {processing ? 'Processing...' : 'Proceed to Payment'}
          </button>

          <p className="text-xs text-gray-500 text-center mt-4">
            Secure checkout powered by Stripe. You will be redirected to complete your purchase.
          </p>

          <button
            onClick={() => navigate('/cart')}
            className="w-full btn-secondary mt-4"
          >
            Back to Cart
          </button>
        </div>
      </div>
    </div>
  );
};

export default Checkout;
