import React, { useEffect, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { FaCheckCircle, FaDownload } from 'react-icons/fa';
import api from '../services/api';

const CheckoutSuccess = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [loading, setLoading] = useState(true);
  const [purchases, setPurchases] = useState([]);

  useEffect(() => {
    const sessionId = searchParams.get('session_id');
    if (sessionId) {
      // Clear cart and fetch purchases
      clearCartAndFetchPurchases();
    } else {
      setLoading(false);
    }
  }, [searchParams]);

  const clearCartAndFetchPurchases = async () => {
    try {
      // Clear cart
      await api.delete('/cart');
      
      // Fetch recent purchases
      const response = await api.get('/products/user/purchases');
      setPurchases(response.data.purchases.slice(0, 5)); // Last 5 purchases
    } catch (error) {
      console.error('Error:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-2xl mx-auto text-center">
        {/* Success Icon */}
        <div className="mb-8">
          <FaCheckCircle className="text-6xl text-green-500 mx-auto mb-4" />
          <h1 className="text-4xl font-bold mb-4">
            Purchase <span className="text-gradient">Complete!</span>
          </h1>
          <p className="text-xl text-gray-400">
            Thank you for your purchase. Your products are ready to download.
          </p>
        </div>

        {/* Purchased Items */}
        {purchases.length > 0 && (
          <div className="card p-8 mb-8">
            <h2 className="text-2xl font-bold mb-6">Your Purchases</h2>
            <div className="space-y-4">
              {purchases.map(purchase => (
                <div
                  key={purchase._id}
                  className="flex items-center justify-between p-4 bg-gray-800 rounded-lg"
                >
                  <div className="text-left">
                    <h3 className="font-medium">{purchase.product?.title}</h3>
                    <p className="text-sm text-gray-400">
                      by {purchase.product?.creator?.username}
                    </p>
                  </div>
                  <button
                    onClick={async () => {
                      try {
                        const response = await api.get(`/products/${purchase.product._id}/download`);
                        window.open(response.data.downloadUrl, '_blank');
                      } catch (error) {
                        console.error('Download failed:', error);
                        alert('Download failed. Please try again from your purchases page.');
                      }
                    }}
                    className="btn-primary flex items-center gap-2"
                  >
                    <FaDownload />
                    Download
                  </button>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Action Buttons */}
        <div className="flex flex-col sm:flex-row gap-4 justify-center">
          <button
            onClick={() => navigate('/dashboard')}
            className="btn-primary"
          >
            Go to Dashboard
          </button>
          <button
            onClick={() => navigate('/marketplace')}
            className="btn-secondary"
          >
            Continue Shopping
          </button>
        </div>

        {/* Receipt Info */}
        <p className="text-sm text-gray-500 mt-8">
          A receipt has been sent to your email. You can access your purchases anytime from your dashboard.
        </p>
      </div>
    </div>
  );
};

export default CheckoutSuccess;
