import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { FaHeart, FaComment, FaShare, FaPlus, FaExternalLinkAlt, FaGithub, FaCloudSun, FaUsers, FaTasks, FaFire, FaStar, FaEye, FaPlay, FaCode, FaCheckCircle, FaTh, FaList, FaFilter, FaBookmark, FaChartLine, FaCrown, FaSearch, FaTimes, FaClock, FaTrophy } from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import MessageButton from '../components/MessageButton';
import api from '../services/api';

const Community = () => {
  const [apps, setApps] = useState([]);
  const [filteredApps, setFilteredApps] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState('all');
  const [sortBy, setSortBy] = useState('recent');
  const [viewMode, setViewMode] = useState('grid');
  const [showFilters, setShowFilters] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedApp, setSelectedApp] = useState(null);
  const [showComments, setShowComments] = useState(false);
  const [collection, setCollection] = useState('all');
  const { isAuthenticated, user } = useAuth();

  useEffect(() => {
    fetchApps();
  }, []);

  useEffect(() => {
    applyFiltersAndSort();
  }, [apps, filter, sortBy, searchQuery, collection]);

  const fetchApps = async () => {
    try {
      const response = await api.get('/apps');
      const appsData = response.data.apps || [];
      // Simulate view counts and trending scores for demo
      const enhancedApps = appsData.map(app => ({
        ...app,
        views: Math.floor(Math.random() * 5000) + 100,
        trending: Math.random() > 0.5,
        verified: app.creator?.subscriptionTier === 'creator',
        hasVideo: Math.random() > 0.7,
        codeQuality: Math.floor(Math.random() * 40) + 60,
      }));
      setApps(enhancedApps);
    } catch (error) {
      console.error('Error fetching apps:', error);
    } finally {
      setLoading(false);
    }
  };

  const applyFiltersAndSort = () => {
    let filtered = [...apps];

    // Collection filter
    if (collection === 'trending') {
      filtered = filtered.filter(app => app.trending);
    } else if (collection === 'verified') {
      filtered = filtered.filter(app => app.verified);
    } else if (collection === 'popular') {
      filtered = filtered.sort((a, b) => (b.likes?.length || 0) - (a.likes?.length || 0)).slice(0, 12);
    }

    // Tech stack filter
    if (filter !== 'all') {
      filtered = filtered.filter(app => 
        app.techStack?.some(tech => tech.toLowerCase().includes(filter))
      );
    }

    // Search filter
    if (searchQuery) {
      const query = searchQuery.toLowerCase();
      filtered = filtered.filter(app =>
        app.title.toLowerCase().includes(query) ||
        app.description.toLowerCase().includes(query) ||
        app.techStack?.some(tech => tech.toLowerCase().includes(query))
      );
    }

    // Sort
    switch (sortBy) {
      case 'popular':
        filtered.sort((a, b) => (b.likes?.length || 0) - (a.likes?.length || 0));
        break;
      case 'recent':
        filtered.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        break;
      case 'views':
        filtered.sort((a, b) => (b.views || 0) - (a.views || 0));
        break;
      case 'trending':
        filtered.sort((a, b) => {
          if (a.trending && !b.trending) return -1;
          if (!a.trending && b.trending) return 1;
          return (b.views || 0) - (a.views || 0);
        });
        break;
      default:
        break;
    }

    setFilteredApps(filtered);
  };

  const handleLike = async (appId) => {
    if (!isAuthenticated) {
      alert('Please sign in to like projects');
      return;
    }
    try {
      await api.post(`/apps/${appId}/like`);
      fetchApps();
    } catch (error) {
      console.error('Error liking app:', error);
    }
  };

  const handleBookmark = async (appId) => {
    if (!isAuthenticated) {
      alert('Please sign in to bookmark projects');
      return;
    }
    try {
      await api.post(`/apps/${appId}/bookmark`);
      fetchApps();
    } catch (error) {
      console.error('Error bookmarking app:', error);
    }
  };

  const trackView = async (appId) => {
    try {
      await api.post(`/apps/${appId}/view`);
    } catch (error) {
      // Silent fail for analytics
    }
  };

  const openComments = (app) => {
    setSelectedApp(app);
    setShowComments(true);
    trackView(app._id);
  };

  const collections = [
    { id: 'all', label: 'All Projects', icon: <FaTh /> },
    { id: 'trending', label: 'Trending', icon: <FaFire /> },
    { id: 'verified', label: 'Verified', icon: <FaCheckCircle /> },
    { id: 'popular', label: 'Most Popular', icon: <FaTrophy /> },
  ];

  const techStacks = ['All', 'React', 'Vue', 'Angular', 'Node.js', 'Python', 'Django', 'Next.js'];

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center mb-6">
          <div>
            <h1 className="text-4xl font-bold mb-2">
              Community <span className="text-gradient">Hub</span>
            </h1>
            <p className="text-gray-400">Discover {filteredApps.length} amazing projects from the community</p>
          </div>
          {isAuthenticated && (
            <Link to="/community/new" className="btn-primary mt-4 md:mt-0 flex items-center gap-2">
              <FaPlus /> Share Your App
            </Link>
          )}
        </div>

        {/* Collections Bar */}
        <div className="mb-6 flex gap-3 overflow-x-auto pb-2">
          {collections.map((col) => (
            <button
              key={col.id}
              onClick={() => setCollection(col.id)}
              className={`flex items-center gap-2 px-4 py-2 rounded-lg whitespace-nowrap transition-all ${
                collection === col.id
                  ? 'bg-gradient-to-r from-turquoise to-gold text-black font-semibold'
                  : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
              }`}
            >
              {col.icon}
              <span>{col.label}</span>
              {col.id === 'trending' && collection !== 'trending' && (
                <FaFire className="text-orange-500 animate-pulse" />
              )}
            </button>
          ))}
        </div>

        {/* Search & Controls */}
        <div className="mb-6 flex flex-col md:flex-row gap-4">
          {/* Search Bar */}
          <div className="flex-1 relative">
            <FaSearch className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500" />
            <input
              type="text"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              placeholder="Search projects, tech stacks, creators..."
              className="w-full pl-10 pr-10 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 placeholder-gray-500 focus:outline-none focus:border-turquoise transition-colors"
            />
            {searchQuery && (
              <button
                onClick={() => setSearchQuery('')}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-300"
              >
                <FaTimes />
              </button>
            )}
          </div>

          {/* View Toggle */}
          <div className="flex gap-2">
            <button
              onClick={() => setViewMode('grid')}
              className={`p-3 rounded-lg transition-colors ${
                viewMode === 'grid' ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
              }`}
              title="Grid View"
            >
              <FaTh />
            </button>
            <button
              onClick={() => setViewMode('list')}
              className={`p-3 rounded-lg transition-colors ${
                viewMode === 'list' ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
              }`}
              title="List View"
            >
              <FaList />
            </button>
            <button
              onClick={() => setShowFilters(!showFilters)}
              className={`p-3 rounded-lg transition-colors ${
                showFilters ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
              }`}
              title="Advanced Filters"
            >
              <FaFilter />
            </button>
          </div>

          {/* Sort Dropdown */}
          <select
            value={sortBy}
            onChange={(e) => setSortBy(e.target.value)}
            className="px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise transition-colors"
          >
            <option value="recent">Most Recent</option>
            <option value="popular">Most Popular</option>
            <option value="trending">Trending</option>
            <option value="views">Most Viewed</option>
          </select>
        </div>

        {/* Advanced Filters */}
        {showFilters && (
          <div className="mb-6 p-4 bg-gray-800 rounded-lg border border-gray-700">
            <h3 className="text-sm font-semibold text-gray-400 mb-3">Filter by Technology</h3>
            <div className="flex flex-wrap gap-2">
              {techStacks.map((tech) => (
                <button
                  key={tech}
                  onClick={() => setFilter(tech.toLowerCase())}
                  className={`px-4 py-2 rounded-lg text-sm transition-colors ${
                    filter === tech.toLowerCase()
                      ? 'bg-turquoise text-black font-semibold'
                      : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                  }`}
                >
                  {tech}
                </button>
              ))}
            </div>
          </div>
        )}

        {/* Apps Grid/List */}
        {loading ? (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise mx-auto"></div>
            <p className="text-gray-400 mt-4">Loading projects...</p>
          </div>
        ) : filteredApps.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-400 text-lg">
              {searchQuery || filter !== 'all' ? 'No projects match your filters. Try adjusting your search.' : 'No projects yet. Be the first to share!'}
            </p>
            {(searchQuery || filter !== 'all') && (
              <button
                onClick={() => {
                  setSearchQuery('');
                  setFilter('all');
                }}
                className="mt-4 text-turquoise hover:underline"
              >
                Clear filters
              </button>
            )}
          </div>
        ) : (
          <div className={viewMode === 'grid' ? 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6' : 'space-y-4'}>
            {filteredApps.map((app) => (
              <div 
                key={app._id} 
                className={`card group ${viewMode === 'list' ? 'flex flex-row gap-4' : ''}`}
              >
                {/* App Image */}
                <div className={`relative rounded-lg overflow-hidden bg-gray-900 flex items-center justify-center ${
                  viewMode === 'list' ? 'w-48 h-32 flex-shrink-0' : 'h-48 mb-4'
                }`}>
                  {/* Badges Overlay */}
                  <div className="absolute top-2 left-2 flex gap-2 z-10">
                    {app.trending && (
                      <span className="flex items-center gap-1 px-2 py-1 bg-orange-500 text-white text-xs rounded-full">
                        <FaFire /> Trending
                      </span>
                    )}
                    {app.verified && (
                      <span className="flex items-center gap-1 px-2 py-1 bg-turquoise text-black text-xs rounded-full font-semibold">
                        <FaCheckCircle /> Verified
                      </span>
                    )}
                  </div>

                  {app.hasVideo && (
                    <button className="absolute inset-0 flex items-center justify-center bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity z-10">
                      <div className="w-16 h-16 rounded-full bg-turquoise/90 flex items-center justify-center">
                        <FaPlay className="text-black text-xl ml-1" />
                      </div>
                    </button>
                  )}

                  <img
                    src={app.image}
                    alt={app.title}
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300 absolute inset-0"
                    style={{ zIndex: 1 }}
                    onError={e => {
                      e.target.onerror = null;
                      e.target.style.display = 'none';
                      const fallback = e.target.nextSibling;
                      if (fallback) fallback.style.display = 'flex';
                    }}
                  />
                  {/* Icon fallback */}
                  <span
                    style={{ display: 'none', zIndex: 2 }}
                    className="absolute inset-0 flex flex-col items-center justify-center text-5xl text-turquoise bg-gray-900/80"
                  >
                    {app.title.toLowerCase().includes('weather') && <FaCloudSun />}
                    {app.title.toLowerCase().includes('collab') && <FaUsers />}
                    {app.title.toLowerCase().includes('task') && <FaTasks />}
                    {!['weather','collab','task'].some(word => app.title.toLowerCase().includes(word)) && <FaGithub />}
                    <span className="text-xs mt-2 text-gray-300">No Image</span>
                  </span>
                </div>

                {/* App Info */}
                <div className="flex-1">
                  <div className="flex items-start justify-between mb-2">
                    <h3 className="text-xl font-bold flex-1">{app.title}</h3>
                    {app.codeQuality && (
                      <div className="ml-2">
                        <div className={`text-xs font-semibold px-2 py-1 rounded ${
                          app.codeQuality >= 80 ? 'bg-green-500/20 text-green-400' :
                          app.codeQuality >= 60 ? 'bg-yellow-500/20 text-yellow-400' :
                          'bg-red-500/20 text-red-400'
                        }`}>
                          {app.codeQuality}% Quality
                        </div>
                      </div>
                    )}
                  </div>

                  <p className="text-gray-400 mb-4 line-clamp-2">{app.description}</p>

                  {/* Tech Stack */}
                  <div className="flex flex-wrap gap-2 mb-4">
                    {app.techStack?.slice(0, viewMode === 'list' ? 8 : 4).map((tech, index) => (
                      <span
                        key={index}
                        className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded"
                      >
                        {tech}
                      </span>
                    ))}
                    {app.techStack?.length > (viewMode === 'list' ? 8 : 4) && (
                      <span className="text-xs text-gray-500 px-2 py-1">
                        +{app.techStack.length - (viewMode === 'list' ? 8 : 4)} more
                      </span>
                    )}
                  </div>

                  {/* Links */}
                  <div className="flex gap-3 mb-4">
                    {app.liveUrl && (
                      <a
                        href={app.liveUrl}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="flex items-center gap-1 text-sm text-turquoise hover:underline"
                      >
                        <FaExternalLinkAlt /> Live Demo
                      </a>
                    )}
                    {app.repoUrl && (
                      <a
                        href={app.repoUrl}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="flex items-center gap-1 text-sm text-turquoise hover:underline"
                      >
                        <FaGithub /> Code
                      </a>
                    )}
                    {app.hasVideo && (
                      <button className="flex items-center gap-1 text-sm text-gold hover:underline">
                        <FaPlay /> Video Tour
                      </button>
                    )}
                  </div>

                  {/* Stats Row */}
                  <div className="flex items-center gap-4 text-sm text-gray-500 mb-4">
                    <span className="flex items-center gap-1">
                      <FaEye /> {app.views?.toLocaleString() || 0}
                    </span>
                    <span className="flex items-center gap-1">
                      <FaClock /> {new Date(app.createdAt).toLocaleDateString()}
                    </span>
                  </div>

                  {/* Actions */}
                  <div className="flex items-center justify-between pt-4 border-t border-gray-800">
                    <button
                      onClick={() => handleLike(app._id)}
                      className={`flex items-center gap-2 transition-colors ${
                        app.isLiked ? 'text-red-500' : 'text-gray-400 hover:text-turquoise'
                      }`}
                    >
                      <FaHeart /> {app.likes?.length || 0}
                    </button>
                    <button 
                      onClick={() => openComments(app)}
                      className="flex items-center gap-2 text-gray-400 hover:text-turquoise transition-colors"
                    >
                      <FaComment /> {app.comments?.length || 0}
                    </button>
                    {isAuthenticated && (
                      <button 
                        onClick={() => handleBookmark(app._id)}
                        className={`flex items-center gap-2 transition-colors ${
                          app.isBookmarked ? 'text-gold' : 'text-gray-400 hover:text-gold'
                        }`}
                      >
                        <FaBookmark />
                      </button>
                    )}
                    <button className="flex items-center gap-2 text-gray-400 hover:text-turquoise transition-colors">
                      <FaShare />
                    </button>
                  </div>

                  {/* Creator */}
                  <div className="mt-4 pt-4 border-t border-gray-800 flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <p className="text-sm text-gray-400">
                        by <span className="text-turquoise">{app.creator?.username || 'Anonymous'}</span>
                      </p>
                      {app.creator?.subscriptionTier === 'creator' && (
                        <FaCrown className="text-gold text-xs" title="Creator Tier" />
                      )}
                    </div>
                    {app.creator?._id && (
                      <MessageButton 
                        userId={app.creator._id} 
                        username={app.creator.username}
                        size="small"
                      />
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}

        {/* Comments Modal */}
        {showComments && selectedApp && (
          <CommentsModal 
            app={selectedApp}
            onClose={() => {
              setShowComments(false);
              setSelectedApp(null);
            }}
            onUpdate={fetchApps}
          />
        )}
      </div>
    </div>
  );
};

// Comments Modal Component
const CommentsModal = ({ app, onClose, onUpdate }) => {
  const [comments, setComments] = useState([]);
  const [newComment, setNewComment] = useState('');
  const [replyingTo, setReplyingTo] = useState(null);
  const { user, isAuthenticated } = useAuth();

  useEffect(() => {
    fetchComments();
  }, [app._id]);

  const fetchComments = async () => {
    try {
      const response = await api.get(`/apps/${app._id}/comments`);
      setComments(response.data.comments || []);
    } catch (error) {
      console.error('Error fetching comments:', error);
      // Mock comments for demo
      setComments([
        {
          _id: '1',
          text: 'This looks amazing! Great work on the UI.',
          user: { username: 'designfan', avatar: '' },
          createdAt: new Date().toISOString(),
          replies: []
        }
      ]);
    }
  };

  const handleSubmitComment = async (e) => {
    e.preventDefault();
    if (!newComment.trim() || !isAuthenticated) return;

    try {
      await api.post(`/apps/${app._id}/comments`, {
        text: newComment,
        parentId: replyingTo
      });
      setNewComment('');
      setReplyingTo(null);
      fetchComments();
      onUpdate();
    } catch (error) {
      console.error('Error posting comment:', error);
    }
  };

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-2xl w-full max-h-[80vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
        {/* Modal Header */}
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between">
          <h3 className="text-2xl font-bold">{app.title} - Comments</h3>
          <button onClick={onClose} className="text-gray-400 hover:text-white">
            <FaTimes size={20} />
          </button>
        </div>

        {/* Comment Form */}
        {isAuthenticated ? (
          <form onSubmit={handleSubmitComment} className="p-6 border-b border-gray-800">
            {replyingTo && (
              <div className="mb-2 flex items-center justify-between text-sm text-gray-400">
                <span>Replying to comment...</span>
                <button type="button" onClick={() => setReplyingTo(null)} className="text-turquoise hover:underline">
                  Cancel
                </button>
              </div>
            )}
            <textarea
              value={newComment}
              onChange={(e) => setNewComment(e.target.value)}
              placeholder="Share your thoughts..."
              className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 placeholder-gray-500 focus:outline-none focus:border-turquoise resize-none"
              rows="3"
            />
            <div className="mt-3 flex justify-end">
              <button type="submit" className="btn-primary">
                Post Comment
              </button>
            </div>
          </form>
        ) : (
          <div className="p-6 border-b border-gray-800 text-center">
            <p className="text-gray-400">Please sign in to comment</p>
          </div>
        )}

        {/* Comments List */}
        <div className="p-6 space-y-4">
          {comments.length === 0 ? (
            <p className="text-gray-500 text-center py-8">No comments yet. Be the first to share your thoughts!</p>
          ) : (
            comments.map(comment => (
              <div key={comment._id} className="bg-gray-800 rounded-lg p-4">
                <div className="flex items-start gap-3">
                  <div className="w-10 h-10 rounded-full bg-gradient-to-br from-turquoise to-gold flex items-center justify-center text-black font-bold flex-shrink-0">
                    {comment.user?.username?.[0]?.toUpperCase() || 'U'}
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-1">
                      <span className="font-semibold text-turquoise">{comment.user?.username || 'Anonymous'}</span>
                      <span className="text-xs text-gray-500">
                        {new Date(comment.createdAt).toLocaleDateString()}
                      </span>
                    </div>
                    <p className="text-gray-300">{comment.text}</p>
                    {isAuthenticated && (
                      <button
                        onClick={() => setReplyingTo(comment._id)}
                        className="mt-2 text-sm text-gray-400 hover:text-turquoise"
                      >
                        Reply
                      </button>
                    )}
                  </div>
                </div>
              </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default Community;
