import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';
import {
  FaDollarSign,
  FaShoppingCart,
  FaBox,
  FaChartLine,
  FaDownload,
  FaClock,
  FaPlus,
  FaMoneyBillWave,
} from 'react-icons/fa';
import {
  LineChart,
  Line,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
} from 'recharts';

const CreatorDashboard = () => {
  const navigate = useNavigate();
  const { user } = useAuth();
  const [analytics, setAnalytics] = useState(null);
  const [topProducts, setTopProducts] = useState([]);
  const [recentSales, setRecentSales] = useState([]);
  const [payouts, setPayouts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showPayoutModal, setShowPayoutModal] = useState(false);
  const [payoutAmount, setPayoutAmount] = useState('');
  const [payoutNote, setPayoutNote] = useState('');

  useEffect(() => {
    // Check if user is creator
    if (user && user.subscriptionTier !== 'creator') {
      navigate('/pricing');
      return;
    }
    fetchData();
  }, [user, navigate]);

  const fetchData = async () => {
    try {
      setLoading(true);
      const [analyticsRes, topProductsRes, salesRes, payoutsRes] = await Promise.all([
        api.get('/creator/analytics'),
        api.get('/creator/top-products?limit=5'),
        api.get('/creator/sales?limit=10'),
        api.get('/creator/payouts'),
      ]);

      setAnalytics(analyticsRes.data);
      setTopProducts(topProductsRes.data.products);
      setRecentSales(salesRes.data.purchases);
      setPayouts(payoutsRes.data.payouts);
    } catch (error) {
      console.error('Failed to fetch creator data:', error);
    } finally {
      setLoading(false);
    }
  };

  const handlePayoutRequest = async () => {
    try {
      await api.post('/creator/payout', {
        amount: parseFloat(payoutAmount),
        method: 'stripe',
        note: payoutNote,
      });

      alert('Payout request submitted successfully!');
      setShowPayoutModal(false);
      setPayoutAmount('');
      setPayoutNote('');
      fetchData();
    } catch (error) {
      console.error('Payout request failed:', error);
      alert(error.response?.data?.message || 'Failed to request payout');
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  const COLORS = ['#14b8a6', '#8b5cf6', '#f59e0b', '#10b981', '#ef4444'];

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="flex justify-between items-center mb-8">
          <div>
            <h1 className="text-4xl font-bold mb-2">
              Creator <span className="text-gradient">Dashboard</span>
            </h1>
            <p className="text-gray-400">Track your sales, earnings, and performance</p>
          </div>
          <button
            onClick={() => navigate('/marketplace/create')}
            className="btn-primary flex items-center gap-2"
          >
            <FaPlus />
            Create Product
          </button>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaDollarSign className="text-3xl text-turquoise" />
              <span className="text-sm text-gray-400">Total Earnings</span>
            </div>
            <div className="text-3xl font-bold mb-1">
              ${analytics?.totalEarnings?.toFixed(2) || '0.00'}
            </div>
            <div className="text-sm text-gray-400">
              Last 30 days: ${analytics?.recentRevenue?.toFixed(2) || '0.00'}
            </div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaMoneyBillWave className="text-3xl text-green-500" />
              <span className="text-sm text-gray-400">Available Balance</span>
            </div>
            <div className="text-3xl font-bold mb-1">
              ${analytics?.availableBalance?.toFixed(2) || '0.00'}
            </div>
            <button
              onClick={() => setShowPayoutModal(true)}
              disabled={!analytics?.availableBalance || analytics.availableBalance < 10}
              className="btn-secondary text-sm mt-2 disabled:opacity-50"
            >
              Request Payout
            </button>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaShoppingCart className="text-3xl text-purple" />
              <span className="text-sm text-gray-400">Total Sales</span>
            </div>
            <div className="text-3xl font-bold mb-1">{analytics?.totalSales || 0}</div>
            <div className="text-sm text-gray-400">
              Last 30 days: {analytics?.recentSales || 0}
            </div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaBox className="text-3xl text-yellow-500" />
              <span className="text-sm text-gray-400">Total Products</span>
            </div>
            <div className="text-3xl font-bold mb-1">{analytics?.totalProducts || 0}</div>
            <button
              onClick={() => navigate('/creator/products')}
              className="text-sm text-turquoise hover:underline mt-2"
            >
              Manage Products →
            </button>
          </div>
        </div>

        {/* Charts Row */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
          {/* Sales Trend */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6 flex items-center gap-2">
              <FaChartLine className="text-turquoise" />
              Sales Trend
            </h2>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={analytics?.salesTrend || []}>
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="month" stroke="#9ca3af" />
                <YAxis stroke="#9ca3af" />
                <Tooltip
                  contentStyle={{ backgroundColor: '#1f2937', border: 'none' }}
                  labelStyle={{ color: '#fff' }}
                />
                <Legend />
                <Line
                  type="monotone"
                  dataKey="revenue"
                  stroke="#14b8a6"
                  strokeWidth={2}
                  name="Revenue ($)"
                />
                <Line
                  type="monotone"
                  dataKey="sales"
                  stroke="#8b5cf6"
                  strokeWidth={2}
                  name="Sales"
                />
              </LineChart>
            </ResponsiveContainer>
          </div>

          {/* Sales by Product */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Sales by Product</h2>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={analytics?.salesByProduct || []}
                  dataKey="revenue"
                  nameKey="product"
                  cx="50%"
                  cy="50%"
                  outerRadius={100}
                  label
                >
                  {(analytics?.salesByProduct || []).map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip
                  contentStyle={{ backgroundColor: '#1f2937', border: 'none' }}
                  formatter={(value) => `$${value.toFixed(2)}`}
                />
              </PieChart>
            </ResponsiveContainer>
          </div>
        </div>

        {/* Top Products */}
        <div className="card p-6 mb-8">
          <h2 className="text-2xl font-bold mb-6">Top Performing Products</h2>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-700">
                  <th className="text-left py-3 px-4">Product</th>
                  <th className="text-right py-3 px-4">Sales</th>
                  <th className="text-right py-3 px-4">Revenue</th>
                  <th className="text-right py-3 px-4">Downloads</th>
                  <th className="text-right py-3 px-4">Rating</th>
                </tr>
              </thead>
              <tbody>
                {topProducts.map((item, idx) => (
                  <tr key={idx} className="border-b border-gray-800 hover:bg-gray-800/50">
                    <td className="py-4 px-4">
                      <div className="flex items-center gap-3">
                        {item.product.previewImages?.[0] ? (
                          <img
                            src={item.product.previewImages[0]}
                            alt={item.product.title}
                            className="w-12 h-12 rounded object-cover"
                          />
                        ) : (
                          <div className="w-12 h-12 rounded bg-gradient-to-br from-turquoise/20 to-purple/20 flex items-center justify-center">
                            📦
                          </div>
                        )}
                        <div>
                          <div className="font-medium">{item.product.title}</div>
                          <div className="text-sm text-gray-400">
                            ${item.product.price.toFixed(2)}
                          </div>
                        </div>
                      </div>
                    </td>
                    <td className="text-right py-4 px-4 font-medium">{item.sales}</td>
                    <td className="text-right py-4 px-4 text-turquoise font-bold">
                      ${item.revenue.toFixed(2)}
                    </td>
                    <td className="text-right py-4 px-4 text-gray-400">
                      <FaDownload className="inline mr-1" />
                      {item.product.downloads}
                    </td>
                    <td className="text-right py-4 px-4">
                      ⭐ {item.averageRating.toFixed(1)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>

        {/* Recent Sales & Payouts */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Sales */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Recent Sales</h2>
            <div className="space-y-4">
              {recentSales.map((sale) => (
                <div
                  key={sale._id}
                  className="flex items-center justify-between p-4 bg-gray-800 rounded-lg"
                >
                  <div className="flex items-center gap-3">
                    <img
                      src={sale.buyer?.avatar || '/default-avatar.png'}
                      alt={sale.buyer?.username}
                      className="w-10 h-10 rounded-full"
                    />
                    <div>
                      <div className="font-medium">{sale.product?.title}</div>
                      <div className="text-sm text-gray-400">
                        by {sale.buyer?.username}
                      </div>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="font-bold text-turquoise">
                      ${sale.sellerEarnings.toFixed(2)}
                    </div>
                    <div className="text-xs text-gray-400">
                      {new Date(sale.createdAt).toLocaleDateString()}
                    </div>
                  </div>
                </div>
              ))}
              {recentSales.length === 0 && (
                <p className="text-gray-400 text-center py-8">No sales yet</p>
              )}
            </div>
          </div>

          {/* Payout History */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Payout History</h2>
            <div className="space-y-4">
              {payouts.slice(0, 5).map((payout) => (
                <div
                  key={payout._id}
                  className="flex items-center justify-between p-4 bg-gray-800 rounded-lg"
                >
                  <div>
                    <div className="font-medium">${payout.amount.toFixed(2)}</div>
                    <div className="text-sm text-gray-400">
                      {new Date(payout.requestedAt).toLocaleDateString()}
                    </div>
                  </div>
                  <div className="text-right">
                    <span
                      className={`px-3 py-1 rounded-full text-sm ${
                        payout.status === 'completed'
                          ? 'bg-green-500/20 text-green-500'
                          : payout.status === 'processing'
                          ? 'bg-yellow-500/20 text-yellow-500'
                          : payout.status === 'failed'
                          ? 'bg-red-500/20 text-red-500'
                          : 'bg-gray-500/20 text-gray-500'
                      }`}
                    >
                      {payout.status}
                    </span>
                  </div>
                </div>
              ))}
              {payouts.length === 0 && (
                <p className="text-gray-400 text-center py-8">No payouts yet</p>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Payout Modal */}
      {showPayoutModal && (
        <div className="fixed inset-0 bg-black/80 flex items-center justify-center z-50 p-4">
          <div className="card max-w-md w-full p-8">
            <h2 className="text-2xl font-bold mb-6">Request Payout</h2>

            <div className="mb-4">
              <div className="text-sm text-gray-400 mb-2">Available Balance</div>
              <div className="text-3xl font-bold text-turquoise mb-4">
                ${analytics?.availableBalance?.toFixed(2)}
              </div>
            </div>

            <div className="mb-4">
              <label className="block text-sm font-medium mb-2">Amount ($)</label>
              <input
                type="number"
                min="10"
                max={analytics?.availableBalance}
                step="0.01"
                value={payoutAmount}
                onChange={(e) => setPayoutAmount(e.target.value)}
                className="input-field"
                placeholder="Minimum $10"
              />
            </div>

            <div className="mb-6">
              <label className="block text-sm font-medium mb-2">Note (Optional)</label>
              <textarea
                value={payoutNote}
                onChange={(e) => setPayoutNote(e.target.value)}
                className="input-field"
                rows="3"
                placeholder="Any additional notes..."
              />
            </div>

            <div className="flex gap-4">
              <button
                onClick={handlePayoutRequest}
                disabled={!payoutAmount || parseFloat(payoutAmount) < 10}
                className="btn-primary flex-1 disabled:opacity-50"
              >
                Submit Request
              </button>
              <button onClick={() => setShowPayoutModal(false)} className="btn-secondary flex-1">
                Cancel
              </button>
            </div>

            <p className="text-xs text-gray-500 text-center mt-4">
              Payouts are typically processed within 3-5 business days
            </p>
          </div>
        </div>
      )}
    </div>
  );
};

export default CreatorDashboard;
