import React, { useState, useEffect } from 'react';
import { useAuth } from '../context/AuthContext';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';
import {
  FaPalette,
  FaUser,
  FaGlobe,
  FaGithub,
  FaTwitter,
  FaLinkedin,
  FaSave,
  FaPlus,
  FaTimes,
} from 'react-icons/fa';

const CustomProfile = () => {
  const { user, updateUser } = useAuth();
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [profile, setProfile] = useState({
    theme: 'default',
    accentColor: '#14b8a6',
    bannerImage: '',
    tagline: '',
    portfolio: [],
    skills: [],
    socialLinks: {
      github: '',
      twitter: '',
      linkedin: '',
      website: '',
    },
  });
  const [newSkill, setNewSkill] = useState('');
  const [newPortfolio, setNewPortfolio] = useState({
    title: '',
    description: '',
    url: '',
    image: '',
    tags: [],
  });
  const [showPortfolioForm, setShowPortfolioForm] = useState(false);

  useEffect(() => {
    // Check if user has Pro or Creator tier
    if (user && user.subscriptionTier === 'free') {
      navigate('/pricing');
      return;
    }
    
    loadProfile();
  }, [user, navigate]);

  const loadProfile = async () => {
    try {
      const response = await api.get('/users/profile');
      if (response.data.customProfile) {
        setProfile({
          ...profile,
          ...response.data.customProfile,
          portfolio: response.data.portfolio || [],
          skills: response.data.skills || [],
          socialLinks: response.data.socialLinks || profile.socialLinks,
        });
      }
    } catch (error) {
      console.error('Failed to load profile:', error);
    }
  };

  const handleSave = async () => {
    try {
      setLoading(true);
      await api.put('/users/profile', { customProfile: profile, ...profile });
      alert('Profile updated successfully!');
      updateUser();
    } catch (error) {
      console.error('Failed to save profile:', error);
      alert('Failed to save profile');
    } finally {
      setLoading(false);
    }
  };

  const addSkill = () => {
    if (newSkill && !profile.skills.includes(newSkill)) {
      setProfile({ ...profile, skills: [...profile.skills, newSkill] });
      setNewSkill('');
    }
  };

  const removeSkill = (skill) => {
    setProfile({
      ...profile,
      skills: profile.skills.filter((s) => s !== skill),
    });
  };

  const addPortfolio = () => {
    if (newPortfolio.title && newPortfolio.url) {
      setProfile({
        ...profile,
        portfolio: [...profile.portfolio, newPortfolio],
      });
      setNewPortfolio({
        title: '',
        description: '',
        url: '',
        image: '',
        tags: [],
      });
      setShowPortfolioForm(false);
    }
  };

  const removePortfolio = (index) => {
    setProfile({
      ...profile,
      portfolio: profile.portfolio.filter((_, i) => i !== index),
    });
  };

  const themes = [
    { value: 'default', label: 'Default', bg: 'from-gray-900 to-gray-800' },
    { value: 'dark', label: 'Dark', bg: 'from-black to-gray-900' },
    { value: 'ocean', label: 'Ocean', bg: 'from-blue-900 to-cyan-900' },
    { value: 'sunset', label: 'Sunset', bg: 'from-orange-900 to-pink-900' },
    { value: 'forest', label: 'Forest', bg: 'from-green-900 to-emerald-900' },
  ];

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-4xl mx-auto">
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">
            Custom <span className="text-gradient">Profile</span>
          </h1>
          <p className="text-gray-400">Personalize your profile and showcase your work</p>
        </div>

        {/* Theme Selection */}
        <div className="card p-6 mb-6">
          <h2 className="text-2xl font-bold mb-4 flex items-center gap-2">
            <FaPalette className="text-turquoise" />
            Profile Theme
          </h2>
          
          <div className="grid grid-cols-2 md:grid-cols-5 gap-4 mb-4">
            {themes.map((theme) => (
              <button
                key={theme.value}
                onClick={() => setProfile({ ...profile, theme: theme.value })}
                className={`p-4 rounded-lg bg-gradient-to-br ${theme.bg} border-2 transition-all ${
                  profile.theme === theme.value ? 'border-turquoise' : 'border-transparent'
                }`}
              >
                <div className="text-center text-sm font-medium">{theme.label}</div>
              </button>
            ))}
          </div>

          <div className="mt-4">
            <label className="block text-sm font-medium mb-2">Accent Color</label>
            <div className="flex gap-4 items-center">
              <input
                type="color"
                value={profile.accentColor}
                onChange={(e) => setProfile({ ...profile, accentColor: e.target.value })}
                className="w-20 h-10 rounded cursor-pointer"
              />
              <span className="text-gray-400">{profile.accentColor}</span>
            </div>
          </div>
        </div>

        {/* Banner & Tagline */}
        <div className="card p-6 mb-6">
          <h2 className="text-2xl font-bold mb-4">Profile Header</h2>
          
          <div className="mb-4">
            <label className="block text-sm font-medium mb-2">Banner Image URL</label>
            <input
              type="url"
              value={profile.bannerImage}
              onChange={(e) => setProfile({ ...profile, bannerImage: e.target.value })}
              className="input-field"
              placeholder="https://example.com/banner.jpg"
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2">Tagline</label>
            <input
              type="text"
              value={profile.tagline}
              onChange={(e) => setProfile({ ...profile, tagline: e.target.value })}
              className="input-field"
              placeholder="Full-stack developer | Open source enthusiast"
              maxLength={100}
            />
            <p className="text-xs text-gray-500 mt-1">{profile.tagline.length}/100</p>
          </div>
        </div>

        {/* Skills */}
        <div className="card p-6 mb-6">
          <h2 className="text-2xl font-bold mb-4">Skills</h2>
          
          <div className="flex gap-2 mb-4">
            <input
              type="text"
              value={newSkill}
              onChange={(e) => setNewSkill(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && addSkill()}
              className="input-field flex-1"
              placeholder="Add a skill (e.g., React, Node.js)"
            />
            <button onClick={addSkill} className="btn-primary">
              <FaPlus />
            </button>
          </div>

          <div className="flex flex-wrap gap-2">
            {profile.skills.map((skill, idx) => (
              <span
                key={idx}
                className="bg-turquoise/20 text-turquoise px-3 py-1 rounded-full text-sm flex items-center gap-2"
              >
                {skill}
                <button onClick={() => removeSkill(skill)} className="hover:text-red-500">
                  <FaTimes size={12} />
                </button>
              </span>
            ))}
          </div>
        </div>

        {/* Social Links */}
        <div className="card p-6 mb-6">
          <h2 className="text-2xl font-bold mb-4">Social Links</h2>
          
          <div className="space-y-4">
            <div className="flex gap-3 items-center">
              <FaGithub className="text-2xl" />
              <input
                type="url"
                value={profile.socialLinks.github}
                onChange={(e) =>
                  setProfile({
                    ...profile,
                    socialLinks: { ...profile.socialLinks, github: e.target.value },
                  })
                }
                className="input-field flex-1"
                placeholder="https://github.com/username"
              />
            </div>

            <div className="flex gap-3 items-center">
              <FaTwitter className="text-2xl hover:text-black transition-colors cursor-pointer" />
              <input
                type="url"
                value={profile.socialLinks.twitter}
                onChange={(e) =>
                  setProfile({
                    ...profile,
                    socialLinks: { ...profile.socialLinks, twitter: e.target.value },
                  })
                }
                className="input-field flex-1"
                placeholder="https://twitter.com/username"
              />
            </div>

            <div className="flex gap-3 items-center">
              <FaLinkedin className="text-2xl hover:text-[#0077B5] transition-colors cursor-pointer" />
              <input
                type="url"
                value={profile.socialLinks.linkedin}
                onChange={(e) =>
                  setProfile({
                    ...profile,
                    socialLinks: { ...profile.socialLinks, linkedin: e.target.value },
                  })
                }
                className="input-field flex-1"
                placeholder="https://linkedin.com/in/username"
              />
            </div>

            <div className="flex gap-3 items-center">
              <FaGlobe className="text-2xl" />
              <input
                type="url"
                value={profile.socialLinks.website}
                onChange={(e) =>
                  setProfile({
                    ...profile,
                    socialLinks: { ...profile.socialLinks, website: e.target.value },
                  })
                }
                className="input-field flex-1"
                placeholder="https://yourwebsite.com"
              />
            </div>
          </div>
        </div>

        {/* Portfolio */}
        <div className="card p-6 mb-6">
          <div className="flex justify-between items-center mb-4">
            <h2 className="text-2xl font-bold">Portfolio</h2>
            <button
              onClick={() => setShowPortfolioForm(!showPortfolioForm)}
              className="btn-primary flex items-center gap-2"
            >
              <FaPlus /> Add Project
            </button>
          </div>

          {showPortfolioForm && (
            <div className="mb-6 p-4 bg-gray-800 rounded-lg">
              <input
                type="text"
                value={newPortfolio.title}
                onChange={(e) => setNewPortfolio({ ...newPortfolio, title: e.target.value })}
                className="input-field mb-3"
                placeholder="Project Title"
              />
              <textarea
                value={newPortfolio.description}
                onChange={(e) => setNewPortfolio({ ...newPortfolio, description: e.target.value })}
                className="input-field mb-3"
                rows="3"
                placeholder="Project Description"
              />
              <input
                type="url"
                value={newPortfolio.url}
                onChange={(e) => setNewPortfolio({ ...newPortfolio, url: e.target.value })}
                className="input-field mb-3"
                placeholder="Project URL"
              />
              <input
                type="url"
                value={newPortfolio.image}
                onChange={(e) => setNewPortfolio({ ...newPortfolio, image: e.target.value })}
                className="input-field mb-3"
                placeholder="Project Image URL"
              />
              <div className="flex gap-2">
                <button onClick={addPortfolio} className="btn-primary">
                  Save Project
                </button>
                <button
                  onClick={() => setShowPortfolioForm(false)}
                  className="btn-secondary"
                >
                  Cancel
                </button>
              </div>
            </div>
          )}

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {profile.portfolio.map((project, idx) => (
              <div key={idx} className="bg-gray-800 rounded-lg p-4 relative">
                <button
                  onClick={() => removePortfolio(idx)}
                  className="absolute top-2 right-2 text-red-500 hover:text-red-400"
                >
                  <FaTimes />
                </button>
                {project.image && (
                  <img
                    src={project.image}
                    alt={project.title}
                    className="w-full h-32 object-cover rounded mb-3"
                  />
                )}
                <h3 className="font-bold mb-1">{project.title}</h3>
                <p className="text-sm text-gray-400 mb-2">{project.description}</p>
                <a
                  href={project.url}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-turquoise text-sm hover:underline"
                >
                  View Project →
                </a>
              </div>
            ))}
          </div>
        </div>

        {/* Save Button */}
        <div className="flex justify-end">
          <button
            onClick={handleSave}
            disabled={loading}
            className="btn-primary flex items-center gap-2 disabled:opacity-50"
          >
            <FaSave />
            {loading ? 'Saving...' : 'Save Profile'}
          </button>
        </div>
      </div>
    </div>
  );
};

export default CustomProfile;
