import { useState, useEffect } from 'react';
import { useAuth } from '../context/AuthContext';
import { FaCode, FaTrophy, FaBriefcase, FaChartLine } from 'react-icons/fa';
import api from '../services/api';

const Dashboard = () => {
  const { user } = useAuth();
  const [stats, setStats] = useState({
    myApps: 0,
    myChallenges: 0,
    totalLikes: 0,
    rank: 0,
  });
  const [myApps, setMyApps] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchDashboardData();
  }, []);

  const fetchDashboardData = async () => {
    try {
      const response = await api.get('/users/dashboard');
      setStats(response.data.stats || {});
      setMyApps(response.data.apps || []);
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
    } finally {
      setLoading(false);
    }
  };

  const statCards = [
    { icon: <FaCode size={30} />, label: 'My Projects', value: stats.myApps, color: 'turquoise' },
    { icon: <FaTrophy size={30} />, label: 'Challenges Completed', value: stats.myChallenges, color: 'gold' },
    { icon: <FaChartLine size={30} />, label: 'Total Likes', value: stats.totalLikes, color: 'turquoise' },
    { icon: <FaBriefcase size={30} />, label: 'Community Rank', value: `#${stats.rank}`, color: 'gold' },
  ];

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">
            Welcome back, <span className="text-gradient">{user?.username}</span>!
          </h1>
          <p className="text-gray-400">Here's what's happening with your profile</p>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
          {statCards.map((stat, index) => (
            <div key={index} className="card">
              <div className={`text-${stat.color} mb-3`}>{stat.icon}</div>
              <div className="text-3xl font-bold mb-1">{stat.value}</div>
              <div className="text-gray-400 text-sm">{stat.label}</div>
            </div>
          ))}
        </div>

        {/* Recent Activity */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* My Projects */}
          <div>
            <h2 className="text-2xl font-bold mb-4">My Projects</h2>
            {loading ? (
              <div className="card text-center">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-turquoise mx-auto"></div>
              </div>
            ) : myApps.length === 0 ? (
              <div className="card text-center">
                <p className="text-gray-400">You haven't shared any projects yet.</p>
                <button className="btn-primary mt-4">Share Your First Project</button>
              </div>
            ) : (
              <div className="space-y-4">
                {myApps.slice(0, 5).map((app) => (
                  <div key={app._id} className="card">
                    <h3 className="font-bold mb-1">{app.title}</h3>
                    <p className="text-sm text-gray-400 mb-2 line-clamp-1">{app.description}</p>
                    <div className="flex items-center gap-4 text-sm text-gray-400">
                      <span>{app.likes?.length || 0} likes</span>
                      <span>{app.comments?.length || 0} comments</span>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Profile Settings */}
          <div>
            <h2 className="text-2xl font-bold mb-4">Profile</h2>
            <div className="card">
              <div className="flex items-center gap-4 mb-6">
                <div className="w-20 h-20 rounded-full bg-gradient-to-br from-turquoise to-gold flex items-center justify-center text-2xl font-bold">
                  {user?.username?.charAt(0).toUpperCase()}
                </div>
                <div>
                  <h3 className="text-xl font-bold">{user?.username}</h3>
                  <p className="text-gray-400">{user?.email}</p>
                </div>
              </div>
              
              <div className="space-y-3">
                <div>
                  <label className="text-sm text-gray-400">Member since</label>
                  <p className="font-semibold">
                    {user?.createdAt ? new Date(user.createdAt).toLocaleDateString() : 'N/A'}
                  </p>
                </div>
                <div>
                  <label className="text-sm text-gray-400">Bio</label>
                  <p className="text-gray-300">{user?.bio || 'No bio yet'}</p>
                </div>
              </div>
              
              <button className="btn-outline w-full mt-6">Edit Profile</button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Dashboard;
