import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import api from '../services/api';
import {
  FaEye,
  FaMousePointer,
  FaUserCheck,
  FaChartLine,
  FaArrowLeft,
  FaCalendar,
  FaStar,
} from 'react-icons/fa';
import {
  LineChart,
  Line,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
} from 'recharts';

const JobAnalytics = () => {
  const { jobId } = useParams();
  const navigate = useNavigate();
  const [analytics, setAnalytics] = useState(null);
  const [job, setJob] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchAnalytics();
  }, [jobId]);

  const fetchAnalytics = async () => {
    try {
      setLoading(true);
      const [analyticsRes, jobRes] = await Promise.all([
        api.get(`/job-payments/analytics/${jobId}`),
        api.get(`/jobs/${jobId}`),
      ]);

      setAnalytics(analyticsRes.data.analytics);
      setJob(jobRes.data);
    } catch (error) {
      console.error('Failed to fetch analytics:', error);
      alert('Failed to load analytics');
      navigate('/jobs');
    } finally {
      setLoading(false);
    }
  };

  const handleBoost = () => {
    navigate(`/jobs/${jobId}/pricing`);
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <button
            onClick={() => navigate('/jobs')}
            className="flex items-center gap-2 text-gray-400 hover:text-turquoise mb-4"
          >
            <FaArrowLeft />
            Back to Jobs
          </button>
          
          <div className="flex justify-between items-start">
            <div>
              <h1 className="text-4xl font-bold mb-2">{job?.title}</h1>
              <p className="text-gray-400">{job?.company}</p>
            </div>
            
            <div className="flex gap-4">
              {analytics.tier === 'free' ? (
                <button onClick={handleBoost} className="btn-primary">
                  Boost This Job
                </button>
              ) : (
                <div className="text-right">
                  <div className="flex items-center gap-2 text-turquoise font-semibold">
                    <FaStar />
                    {analytics.tier.charAt(0).toUpperCase() + analytics.tier.slice(1)} Tier
                  </div>
                  <div className="text-sm text-gray-400">
                    {analytics.daysRemaining} days remaining
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaEye className="text-3xl text-turquoise" />
              <span className="text-sm text-gray-400">Views</span>
            </div>
            <div className="text-3xl font-bold">{analytics.views}</div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaMousePointer className="text-3xl text-purple" />
              <span className="text-sm text-gray-400">Clicks</span>
            </div>
            <div className="text-3xl font-bold">{analytics.clicks}</div>
            <div className="text-sm text-gray-400 mt-1">
              {analytics.clickThroughRate}% CTR
            </div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaUserCheck className="text-3xl text-green-500" />
              <span className="text-sm text-gray-400">Applications</span>
            </div>
            <div className="text-3xl font-bold">{analytics.applications}</div>
            <div className="text-sm text-gray-400 mt-1">
              {analytics.conversionRate}% conversion
            </div>
          </div>

          <div className="card p-6">
            <div className="flex items-center justify-between mb-4">
              <FaCalendar className="text-3xl text-yellow-500" />
              <span className="text-sm text-gray-400">Time Left</span>
            </div>
            <div className="text-3xl font-bold">{analytics.daysRemaining}</div>
            <div className="text-sm text-gray-400 mt-1">days</div>
          </div>
        </div>

        {/* Performance Chart */}
        <div className="card p-6 mb-8">
          <h2 className="text-2xl font-bold mb-6 flex items-center gap-2">
            <FaChartLine className="text-turquoise" />
            Performance Overview
          </h2>
          
          <ResponsiveContainer width="100%" height={300}>
            <BarChart
              data={[
                { name: 'Views', value: analytics.views },
                { name: 'Clicks', value: analytics.clicks },
                { name: 'Applications', value: analytics.applications },
              ]}
            >
              <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
              <XAxis dataKey="name" stroke="#9ca3af" />
              <YAxis stroke="#9ca3af" />
              <Tooltip
                contentStyle={{ backgroundColor: '#1f2937', border: 'none' }}
                labelStyle={{ color: '#fff' }}
              />
              <Bar dataKey="value" fill="#14b8a6" />
            </BarChart>
          </ResponsiveContainer>
        </div>

        {/* Listing Details */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Current Plan */}
          <div className="card p-6">
            <h2 className="text-2xl font-bold mb-6">Current Plan</h2>
            <div className="space-y-4">
              <div className="flex justify-between">
                <span className="text-gray-400">Tier:</span>
                <span className="font-semibold capitalize">{analytics.tier}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">Featured:</span>
                <span>{analytics.isFeatured ? '✅ Yes' : '❌ No'}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-400">Pinned:</span>
                <span>{analytics.isPinned ? '✅ Yes' : '❌ No'}</span>
              </div>
              {analytics.boostedUntil && (
                <div className="flex justify-between">
                  <span className="text-gray-400">Active Until:</span>
                  <span>{new Date(analytics.boostedUntil).toLocaleDateString()}</span>
                </div>
              )}
              {analytics.amountPaid > 0 && (
                <div className="flex justify-between">
                  <span className="text-gray-400">Amount Paid:</span>
                  <span className="font-bold text-turquoise">
                    ${analytics.amountPaid.toFixed(2)}
                  </span>
                </div>
              )}
            </div>

            {analytics.tier === 'free' && (
              <button onClick={handleBoost} className="btn-primary w-full mt-6">
                Upgrade to Boost Performance
              </button>
            )}
          </div>

          {/* Payment Details */}
          {analytics.payment && (
            <div className="card p-6">
              <h2 className="text-2xl font-bold mb-6">Payment Details</h2>
              <div className="space-y-4">
                <div className="flex justify-between">
                  <span className="text-gray-400">Plan:</span>
                  <span className="font-semibold capitalize">{analytics.payment.tier}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400">Amount:</span>
                  <span className="font-bold">${analytics.payment.amount.toFixed(2)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400">Duration:</span>
                  <span>{analytics.payment.features.duration} days</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400">Start Date:</span>
                  <span>{new Date(analytics.payment.startDate).toLocaleDateString()}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400">End Date:</span>
                  <span>{new Date(analytics.payment.endDate).toLocaleDateString()}</span>
                </div>
              </div>

              <div className="mt-6 pt-6 border-t border-gray-700">
                <h3 className="font-semibold mb-3">Included Features:</h3>
                <ul className="space-y-2 text-sm text-gray-400">
                  {analytics.payment.features.isFeatured && (
                    <li>✅ Featured placement</li>
                  )}
                  {analytics.payment.features.isPinned && (
                    <li>✅ Top pinned position</li>
                  )}
                  {analytics.payment.features.socialMediaPromotion && (
                    <li>✅ Social media promotion</li>
                  )}
                  {analytics.payment.features.newsletterInclusion && (
                    <li>✅ Newsletter inclusion</li>
                  )}
                </ul>
              </div>
            </div>
          )}
        </div>

        {/* Tips */}
        <div className="card p-6 mt-8 bg-gradient-to-r from-turquoise/10 to-purple/10">
          <h3 className="text-xl font-bold mb-4">💡 Tips to Improve Performance</h3>
          <ul className="space-y-2 text-gray-300">
            <li>• Include specific salary ranges to increase click-through rate</li>
            <li>• Add detailed benefits and perks to attract more applicants</li>
            <li>• Use clear, descriptive job titles that candidates search for</li>
            <li>• Upgrade to Premium or Enterprise for better visibility</li>
            <li>• Update your listing regularly to boost engagement</li>
          </ul>
        </div>
      </div>
    </div>
  );
};

export default JobAnalytics;
