import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';
import { FaRocket, FaStar, FaCrown, FaCheck } from 'react-icons/fa';

const JobPostPricing = ({ jobId }) => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);

  const tiers = [
    {
      name: 'Basic',
      tier: 'basic',
      price: 49.99,
      duration: 30,
      icon: <FaRocket className="text-4xl text-turquoise" />,
      features: [
        '30 days listing',
        'Standard placement',
        'Basic analytics',
        'Email support',
      ],
      color: 'turquoise',
    },
    {
      name: 'Premium',
      tier: 'premium',
      price: 99.99,
      duration: 60,
      icon: <FaStar className="text-4xl text-yellow-500" />,
      popular: true,
      features: [
        '60 days listing',
        'Featured placement',
        'Highlighted listing',
        'Advanced analytics',
        'Social media promotion',
        'Newsletter inclusion',
        'Priority support',
      ],
      color: 'yellow-500',
    },
    {
      name: 'Enterprise',
      tier: 'enterprise',
      price: 199.99,
      duration: 90,
      icon: <FaCrown className="text-4xl text-purple" />,
      features: [
        '90 days listing',
        'Top pinned position',
        'Custom highlight color',
        'Premium analytics',
        'Social media promotion',
        'Newsletter inclusion',
        'Dedicated account manager',
        'Custom branding options',
      ],
      color: 'purple',
    },
  ];

  const handleSelectTier = async (tier) => {
    try {
      setLoading(true);
      const response = await api.post('/job-payments/checkout', {
        jobId,
        tier,
      });

      // Redirect to Stripe checkout
      window.location.href = response.data.url;
    } catch (error) {
      console.error('Checkout error:', error);
      alert(error.response?.data?.message || 'Failed to start checkout');
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-5xl font-bold mb-4">
            Boost Your <span className="text-gradient">Job Listing</span>
          </h1>
          <p className="text-xl text-gray-400 max-w-2xl mx-auto">
            Choose a plan to make your job posting stand out and reach more qualified candidates
          </p>
        </div>

        {/* Pricing Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12">
          {tiers.map((tierInfo) => (
            <div
              key={tierInfo.tier}
              className={`card relative ${
                tierInfo.popular ? 'ring-2 ring-turquoise' : ''
              }`}
            >
              {tierInfo.popular && (
                <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                  <span className="bg-gradient-to-r from-turquoise to-purple px-4 py-1 rounded-full text-sm font-semibold">
                    Most Popular
                  </span>
                </div>
              )}

              <div className="p-8">
                <div className="flex justify-center mb-4">{tierInfo.icon}</div>
                
                <h3 className="text-2xl font-bold text-center mb-2">{tierInfo.name}</h3>
                
                <div className="text-center mb-6">
                  <span className="text-4xl font-bold">${tierInfo.price}</span>
                  <span className="text-gray-400 ml-2">/ {tierInfo.duration} days</span>
                </div>

                <ul className="space-y-3 mb-8">
                  {tierInfo.features.map((feature, idx) => (
                    <li key={idx} className="flex items-start gap-2">
                      <FaCheck className="text-turquoise mt-1 flex-shrink-0" />
                      <span className="text-gray-300">{feature}</span>
                    </li>
                  ))}
                </ul>

                <button
                  onClick={() => handleSelectTier(tierInfo.tier)}
                  disabled={loading}
                  className={`w-full py-3 rounded-lg font-semibold transition-all disabled:opacity-50 ${
                    tierInfo.popular
                      ? 'btn-primary'
                      : 'bg-gray-700 hover:bg-gray-600 text-white'
                  }`}
                >
                  {loading ? 'Processing...' : 'Select Plan'}
                </button>
              </div>
            </div>
          ))}
        </div>

        {/* Comparison Table */}
        <div className="card p-8">
          <h2 className="text-3xl font-bold mb-6 text-center">Feature Comparison</h2>
          
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-700">
                  <th className="text-left py-4 px-4">Feature</th>
                  <th className="text-center py-4 px-4">Basic</th>
                  <th className="text-center py-4 px-4">Premium</th>
                  <th className="text-center py-4 px-4">Enterprise</th>
                </tr>
              </thead>
              <tbody>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Listing Duration</td>
                  <td className="text-center py-4 px-4">30 days</td>
                  <td className="text-center py-4 px-4">60 days</td>
                  <td className="text-center py-4 px-4">90 days</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Featured Placement</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">✅</td>
                  <td className="text-center py-4 px-4">✅</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Top Pinned Position</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">✅</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Highlight Color</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">Standard</td>
                  <td className="text-center py-4 px-4">Custom</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Analytics</td>
                  <td className="text-center py-4 px-4">Basic</td>
                  <td className="text-center py-4 px-4">Advanced</td>
                  <td className="text-center py-4 px-4">Premium</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Social Media Promotion</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">✅</td>
                  <td className="text-center py-4 px-4">✅</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Newsletter Inclusion</td>
                  <td className="text-center py-4 px-4">❌</td>
                  <td className="text-center py-4 px-4">✅</td>
                  <td className="text-center py-4 px-4">✅</td>
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4">Support Level</td>
                  <td className="text-center py-4 px-4">Email</td>
                  <td className="text-center py-4 px-4">Priority</td>
                  <td className="text-center py-4 px-4">Dedicated Manager</td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>

        {/* Benefits */}
        <div className="mt-12 text-center">
          <h2 className="text-3xl font-bold mb-6">Why Promote Your Job?</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="card p-6">
              <div className="text-4xl mb-4">👀</div>
              <h3 className="text-xl font-bold mb-2">More Visibility</h3>
              <p className="text-gray-400">
                Stand out from hundreds of listings and get seen by top talent
              </p>
            </div>
            <div className="card p-6">
              <div className="text-4xl mb-4">⚡</div>
              <h3 className="text-xl font-bold mb-2">Faster Hiring</h3>
              <p className="text-gray-400">
                Fill positions 3x faster with featured placement
              </p>
            </div>
            <div className="card p-6">
              <div className="text-4xl mb-4">📊</div>
              <h3 className="text-xl font-bold mb-2">Better Insights</h3>
              <p className="text-gray-400">
                Track views, clicks, and applications in real-time
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default JobPostPricing;
