import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  FaBriefcase, FaMapMarkerAlt, FaDollarSign, FaClock, FaStar, FaThumbtack, FaChartBar,
  FaSearch, FaFilter, FaTimesCircle, FaBookmark, FaRegBookmark, FaBuilding, FaUsers,
  FaTimes, FaExternalLinkAlt, FaCheckCircle, FaGlobeAmericas,
  FaLinkedin, FaTwitter, FaFacebook, FaUserTie, FaLightbulb, FaRocket, FaAward,
  FaUpload, FaFile, FaFilePdf, FaCalendarAlt, FaHistory,
  FaPaperPlane, FaCopy, FaSpinner
} from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

const Jobs = () => {
  const navigate = useNavigate();
  const { user } = useAuth();
  const DEFAULT_SALARY_RANGE = [0, 300000];
  const savedSearchStorageKey = `jobSavedSearches${user?._id ? `:${user._id}` : ''}`;
  const [jobs, setJobs] = useState([]);
  const [loading, setLoading] = useState(true);
  const [loadError, setLoadError] = useState('');
  const [filters, setFilters] = useState({
    location: 'all',
    type: 'all',
    level: 'all',
  });
  const [searchQuery, setSearchQuery] = useState('');
  const [showFilters, setShowFilters] = useState(false);
  const [selectedTechStack, setSelectedTechStack] = useState([]);
  const [salaryRange, setSalaryRange] = useState(DEFAULT_SALARY_RANGE);
  const [selectedCompanySize, setSelectedCompanySize] = useState([]);
  const [selectedBenefits, setSelectedBenefits] = useState([]);
  const [datePosted, setDatePosted] = useState('all');
  const [savedJobs, setSavedJobs] = useState([]);
  const [savedSearches, setSavedSearches] = useState([]);
  const [selectedSavedSearchId, setSelectedSavedSearchId] = useState('');
  const [sortBy, setSortBy] = useState('relevance');
  const [selectedJob, setSelectedJob] = useState(null);
  const [activeTab, setActiveTab] = useState('overview');
  const [showApplyModal, setShowApplyModal] = useState(false);
  const [applicationData, setApplicationData] = useState({
    resume: null,
    coverLetter: '',
    email: user?.email || '',
    phone: '',
    linkedIn: '',
    portfolio: ''
  });
  const [applications, setApplications] = useState([]);
  const [showApplicationHistory, setShowApplicationHistory] = useState(false);
  const [selectedTemplate, setSelectedTemplate] = useState('professional');
  const [uploading, setUploading] = useState(false);
  
  // Employer portal state
  const [showPostJobModal, setShowPostJobModal] = useState(false);
  const [showEmployerDashboard, setShowEmployerDashboard] = useState(false);
  const [myPostedJobs, setMyPostedJobs] = useState([]);
  const [jobFormData, setJobFormData] = useState({
    title: '',
    company: '',
    companyLogo: '',
    description: '',
    location: '',
    type: 'fulltime',
    level: 'mid',
    salary: '',
    techStack: [],
    applyUrl: '',
    companySize: '',
    benefits: [],
    isRemote: false,
  });
  const [selectedPostedJob, setSelectedPostedJob] = useState(null);
  const [jobApplications, setJobApplications] = useState([]);

  const formatCurrency = (value) => {
    try {
      return new Intl.NumberFormat(undefined, { style: 'currency', currency: 'USD', maximumFractionDigits: 0 }).format(value);
    } catch {
      return `$${value.toLocaleString()}`;
    }
  };

  const parseSalaryToAnnualRange = (salary) => {
    if (!salary || typeof salary !== 'string') return null;

    const normalized = salary.toLowerCase();
    const isHourly = normalized.includes('/hour') || normalized.includes('per hour') || normalized.includes('hour');
    const numbers = salary.match(/\d[\d,]*/g);
    if (!numbers || numbers.length === 0) return null;

    const values = numbers
      .map(n => parseInt(n.replace(/,/g, ''), 10))
      .filter(n => Number.isFinite(n) && n > 0);

    if (values.length === 0) return null;

    let min = Math.min(...values);
    let max = Math.max(...values);

    if (isHourly) {
      // Rough conversion: 40h/week * 52 weeks
      min *= 2080;
      max *= 2080;
    }

    return { min, max, isHourly };
  };

  const getJobMatchScore = (job) => {
    const userSkills = (user?.skills || []).map(s => String(s).toLowerCase());
    const jobSkills = (job?.techStack || []).map(s => String(s).toLowerCase());
    if (userSkills.length === 0 || jobSkills.length === 0) return 0;
    const overlap = jobSkills.filter(s => userSkills.includes(s)).length;
    return Math.round((overlap / Math.max(1, jobSkills.length)) * 100);
  };

  const getJobShareUrl = (job) => {
    // Prefer external apply URL if present
    if (job?.applyUrl) return job.applyUrl;
    return window.location.href;
  };

  const getCompanyInitials = (company) => {
    const cleaned = String(company || '').trim();
    if (!cleaned) return 'CO';

    const parts = cleaned.split(/\s+/).filter(Boolean);
    const a = parts[0]?.[0] || '';
    const b = parts[1]?.[0] || '';
    const initials = `${a}${b}`.trim();
    if (initials) return initials.toUpperCase().slice(0, 2);
    return cleaned.slice(0, 2).toUpperCase();
  };

  const COMPANY_FALLBACK_COLORS = [
    '#20B2AA', // turquoise
    '#9370DB', // purple
    '#FF6347', // tomato
    '#FFD700', // gold
    '#00CED1', // dark turquoise
    '#FF69B4', // hot pink
    '#8A2BE2', // blue violet
    '#32CD32', // lime green
    '#FF8C00', // dark orange
    '#1E90FF', // dodger blue
    '#DC143C', // crimson
    '#2F4F4F', // dark slate
  ];

  const hashStringToInt = (value) => {
    const str = String(value || '');
    let hash = 5381;
    for (let i = 0; i < str.length; i += 1) {
      hash = ((hash << 5) + hash) ^ str.charCodeAt(i);
    }
    return hash >>> 0;
  };

  const getCompanyFallbackColor = (company) => {
    const idx = hashStringToInt(company) % COMPANY_FALLBACK_COLORS.length;
    return COMPANY_FALLBACK_COLORS[idx];
  };

  const getReadableTextColor = (background) => {
    const raw = String(background || '').trim();
    const hex = raw.startsWith('#') ? raw.slice(1) : raw;
    const expanded = hex.length === 3 ? hex.split('').map((c) => c + c).join('') : hex;
    if (!/^[0-9a-fA-F]{6}$/.test(expanded)) return '#FFFFFF';

    const r = parseInt(expanded.slice(0, 2), 16);
    const g = parseInt(expanded.slice(2, 4), 16);
    const b = parseInt(expanded.slice(4, 6), 16);
    const luminance = (0.2126 * r + 0.7152 * g + 0.0722 * b) / 255;
    return luminance > 0.62 ? '#0B0F14' : '#FFFFFF';
  };

  const companyLogoFallbackDataUri = (company, backgroundColor) => {
    const bg = backgroundColor || getCompanyFallbackColor(company);
    const fg = getReadableTextColor(bg);
    const initials = getCompanyInitials(company);
    const svg = `
      <svg xmlns="http://www.w3.org/2000/svg" width="128" height="128" viewBox="0 0 128 128">
        <rect width="128" height="128" rx="18" ry="18" fill="${bg}" />
        <text x="50%" y="54%" dominant-baseline="middle" text-anchor="middle"
          font-family="system-ui, -apple-system, Segoe UI, Roboto, Arial" font-size="56" font-weight="800" fill="${fg}">
          ${initials}
        </text>
      </svg>
    `;
    return `data:image/svg+xml;charset=UTF-8,${encodeURIComponent(svg)}`;
  };

  const shareJob = (platform, job) => {
    const url = encodeURIComponent(getJobShareUrl(job));
    const text = encodeURIComponent(`${job?.title || 'Job'} at ${job?.company || 'Company'}`);

    let shareUrl = '';
    if (platform === 'linkedin') {
      shareUrl = `https://www.linkedin.com/sharing/share-offsite/?url=${url}`;
    } else if (platform === 'twitter') {
      shareUrl = `https://twitter.com/intent/tweet?text=${text}&url=${url}`;
    } else if (platform === 'facebook') {
      shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${url}`;
    }

    if (shareUrl) window.open(shareUrl, '_blank', 'noopener,noreferrer');
  };

  useEffect(() => {
    fetchJobs();
    
    // Load saved jobs from localStorage
    const saved = localStorage.getItem('savedJobs');
    if (saved) setSavedJobs(JSON.parse(saved));

    // Load saved searches
    try {
      const rawSearches = localStorage.getItem(savedSearchStorageKey);
      if (rawSearches) setSavedSearches(JSON.parse(rawSearches));
    } catch (e) {
      console.warn('Failed to load saved searches', e);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    if (user) {
      fetchMyApplications();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [user]);

  useEffect(() => {
    // Save to localStorage whenever savedJobs changes
    localStorage.setItem('savedJobs', JSON.stringify(savedJobs));
  }, [savedJobs]);

  const fetchJobs = async () => {
    try {
      const response = await api.get('/jobs');
      setJobs(response.data.jobs || []);
      setLoadError('');
    } catch (error) {
      console.error('Error fetching jobs:', error);
      setLoadError('Failed to load jobs. Please refresh and try again.');
    } finally {
      setLoading(false);
    }
  };

  const fetchMyApplications = async () => {
    if (!user) return;
    try {
      const response = await api.get('/jobs/applications/my-applications');
      setApplications(response.data.applications || []);
    } catch (error) {
      console.error('Error fetching applications:', error);
    }
  };

  const fetchMyPostedJobs = async () => {
    if (!user) return;
    try {
      const response = await api.get('/jobs/employer/my-jobs');
      setMyPostedJobs(response.data.jobs || []);
    } catch (error) {
      console.error('Error fetching posted jobs:', error);
    }
  };

  const fetchJobApplications = async (jobId) => {
    try {
      const response = await api.get(`/jobs/${jobId}/applications`);
      setJobApplications(response.data.applications || []);
    } catch (error) {
      console.error('Error fetching job applications:', error);
    }
  };

  const handlePostJob = async () => {
    if (!user) {
      alert('Please login to post jobs');
      navigate('/login');
      return;
    }

    if (!jobFormData.title || !jobFormData.company || !jobFormData.description || !jobFormData.location) {
      alert('Please fill in all required fields');
      return;
    }

    try {
      await api.post('/jobs', jobFormData);
      
      setShowPostJobModal(false);
      setJobFormData({
        title: '',
        company: '',
        companyLogo: '',
        description: '',
        location: '',
        type: 'fulltime',
        level: 'mid',
        salary: '',
        techStack: [],
        applyUrl: '',
        companySize: '',
        benefits: [],
        isRemote: false,
      });

      alert('Job posted successfully!');
      fetchJobs();
      fetchMyPostedJobs();
    } catch (error) {
      console.error('Failed to post job:', error);
      alert(error.response?.data?.message || 'Failed to post job. Please try again.');
    }
  };

  const handleDeleteJob = async (jobId) => {
    if (!window.confirm('Are you sure you want to delete this job posting?')) return;

    try {
      await api.delete(`/jobs/${jobId}`);
      alert('Job deleted successfully');
      fetchJobs();
      fetchMyPostedJobs();
    } catch (error) {
      console.error('Failed to delete job:', error);
      alert('Failed to delete job. Please try again.');
    }
  };

  const updateApplicationStatusEmployer = async (applicationId, status, notes = '') => {
    try {
      await api.patch(`/jobs/applications/${applicationId}/employer-status`, {
        status,
        employerNotes: notes,
      });
      
      // Refresh applications
      if (selectedPostedJob) {
        fetchJobApplications(selectedPostedJob._id);
      }
      
      alert('Application status updated');
    } catch (error) {
      console.error('Failed to update application:', error);
      alert('Failed to update application status');
    }
  };

  const persistSavedSearches = (next) => {
    setSavedSearches(next);
    try {
      localStorage.setItem(savedSearchStorageKey, JSON.stringify(next));
    } catch (e) {
      console.warn('Failed to persist saved searches', e);
    }
  };

  const getCurrentSearchState = () => {
    return {
      searchQuery,
      filters,
      selectedTechStack,
      selectedCompanySize,
      selectedBenefits,
      datePosted,
      salaryRange,
      sortBy,
    };
  };

  const applySavedSearch = (savedSearch) => {
    if (!savedSearch?.state) return;
    setSearchQuery(savedSearch.state.searchQuery || '');
    setFilters(savedSearch.state.filters || { location: 'all', type: 'all', level: 'all' });
    setSelectedTechStack(savedSearch.state.selectedTechStack || []);
    setSelectedCompanySize(savedSearch.state.selectedCompanySize || []);
    setSelectedBenefits(savedSearch.state.selectedBenefits || []);
    setDatePosted(savedSearch.state.datePosted || 'all');
    setSalaryRange(savedSearch.state.salaryRange || DEFAULT_SALARY_RANGE);
    setSortBy(savedSearch.state.sortBy || 'relevance');
  };

  const handleSaveSearch = () => {
    const name = window.prompt('Name this search (e.g., "Remote React roles")');
    if (!name) return;

    const next = [
      {
        id: String(Date.now()),
        name: name.trim(),
        state: getCurrentSearchState(),
        createdAt: new Date().toISOString(),
      },
      ...savedSearches,
    ].slice(0, 25);

    persistSavedSearches(next);
    setSelectedSavedSearchId(next[0].id);
  };

  const deleteSavedSearch = (id) => {
    const next = savedSearches.filter(s => s.id !== id);
    persistSavedSearches(next);
    if (selectedSavedSearchId === id) setSelectedSavedSearchId('');
  };

  // Cover letter templates
  const coverLetterTemplates = {
    professional: `Dear Hiring Manager,\n\nI am writing to express my strong interest in the [POSITION] position at [COMPANY]. With my background in software development and passion for creating innovative solutions, I am confident I would be a valuable addition to your team.\n\nMy experience aligns well with the requirements outlined in the job description, particularly in [SKILLS]. I am excited about the opportunity to contribute to [COMPANY]'s mission and grow as part of your talented team.\n\nThank you for considering my application. I look forward to discussing how my skills and experiences can benefit your organization.\n\nBest regards,`,
    
    enthusiastic: `Hello [COMPANY] Team!\n\nI'm thrilled to apply for the [POSITION] role! I've been following [COMPANY]'s work and am genuinely excited about the innovative solutions you're building.\n\nAs someone passionate about [SKILLS], I believe I can bring fresh perspectives and strong technical expertise to your team. I love tackling challenging problems and collaborating with talented developers.\n\nI'd be honored to discuss how I can contribute to [COMPANY]'s continued success. Let's chat!\n\nCheers,`,
    
    concise: `Dear Hiring Manager,\n\nI'm applying for the [POSITION] position at [COMPANY]. My expertise in [SKILLS] and proven track record in delivering high-quality solutions make me an ideal candidate.\n\nI'm eager to bring my skills to your team and contribute to [COMPANY]'s goals.\n\nThank you for your consideration.\n\nSincerely,`
  };

  // Filter configuration data
  const techStackOptions = ['React', 'Node.js', 'Python', 'TypeScript', 'AWS', 'Docker', 'MongoDB', 'PostgreSQL', 'GraphQL', 'Vue.js', 'Angular', 'Django', 'Flask', 'Kubernetes', 'Redis'];
  const companySizeOptions = ['Startup (1-50)', 'Small (51-200)', 'Medium (201-1000)', 'Large (1000+)'];
  const benefitsOptions = ['Health Insurance', 'Dental', 'Vision', '401k', 'Stock Options', 'Remote Work', 'Flexible Hours', 'Unlimited PTO', 'Gym Membership', 'Learning Budget'];
  const datePostedOptions = [
    { value: 'all', label: 'All Time' },
    { value: '24h', label: 'Last 24 Hours' },
    { value: 'week', label: 'Last Week' },
    { value: 'month', label: 'Last Month' }
  ];

  // Filter jobs based on all criteria
  const getFilteredJobs = () => {
    let filtered = jobs;

    // Search query
    if (searchQuery) {
      const query = searchQuery.toLowerCase();
      filtered = filtered.filter(job => 
        job.title.toLowerCase().includes(query) ||
        job.company.toLowerCase().includes(query) ||
        job.description.toLowerCase().includes(query)
      );
    }

    // Location filter
    if (filters.location !== 'all') {
      filtered = filtered.filter(job => {
        if (filters.location === 'remote') return job.isRemote || job.location.toLowerCase().includes('remote');
        if (filters.location === 'onsite') return !job.isRemote;
        if (filters.location === 'hybrid') return job.location.toLowerCase().includes('hybrid');
        return true;
      });
    }

    // Type filter
    if (filters.type !== 'all') {
      filtered = filtered.filter(job => job.type === filters.type);
    }

    // Level filter
    if (filters.level !== 'all') {
      filtered = filtered.filter(job => job.level === filters.level);
    }

    // Tech stack filter
    if (selectedTechStack.length > 0) {
      filtered = filtered.filter(job => 
        job.techStack?.some(tech => selectedTechStack.includes(tech))
      );
    }

    // Benefits filter
    if (selectedBenefits.length > 0) {
      filtered = filtered.filter(job => 
        job.benefits?.some(benefit => selectedBenefits.includes(benefit))
      );
    }

    // Company size filter
    if (selectedCompanySize.length > 0) {
      filtered = filtered.filter(job => selectedCompanySize.includes(job.companySize));
    }

    // Salary range filter (annualized)
    if (salaryRange[0] !== DEFAULT_SALARY_RANGE[0] || salaryRange[1] !== DEFAULT_SALARY_RANGE[1]) {
      filtered = filtered.filter(job => {
        const parsed = parseSalaryToAnnualRange(job.salary);
        if (!parsed) return false;
        // Keep if ranges overlap
        return parsed.max >= salaryRange[0] && parsed.min <= salaryRange[1];
      });
    }

    // Date posted filter
    if (datePosted !== 'all') {
      const now = new Date();
      filtered = filtered.filter(job => {
        const jobDate = new Date(job.createdAt);
        const diffHours = (now - jobDate) / (1000 * 60 * 60);
        if (datePosted === '24h') return diffHours <= 24;
        if (datePosted === 'week') return diffHours <= 168;
        if (datePosted === 'month') return diffHours <= 720;
        return true;
      });
    }

    return filtered;
  };

  const filteredJobs = getFilteredJobs();

  // Sort jobs
  const sortedJobs = [...filteredJobs].sort((a, b) => {
    if (sortBy === 'date') return new Date(b.createdAt) - new Date(a.createdAt);
    if (sortBy === 'salary') {
      const rangeA = parseSalaryToAnnualRange(a.salary);
      const rangeB = parseSalaryToAnnualRange(b.salary);
      const maxA = rangeA?.max || 0;
      const maxB = rangeB?.max || 0;
      return maxB - maxA;
    }
    // Default: featured/pinned first
    if (a.isPinned !== b.isPinned) return b.isPinned - a.isPinned;
    if (a.isFeatured !== b.isFeatured) return b.isFeatured - a.isFeatured;
    return new Date(b.createdAt) - new Date(a.createdAt);
  });

  // Count active filters
  const activeFilterCount = 
    (searchQuery ? 1 : 0) +
    selectedTechStack.length +
    selectedCompanySize.length +
    selectedBenefits.length +
    (datePosted !== 'all' ? 1 : 0) +
    ((salaryRange[0] !== DEFAULT_SALARY_RANGE[0] || salaryRange[1] !== DEFAULT_SALARY_RANGE[1]) ? 1 : 0) +
    (filters.location !== 'all' ? 1 : 0) +
    (filters.type !== 'all' ? 1 : 0) +
    (filters.level !== 'all' ? 1 : 0);

  const clearAllFilters = () => {
    setSearchQuery('');
    setFilters({ location: 'all', type: 'all', level: 'all' });
    setSelectedTechStack([]);
    setSelectedCompanySize([]);
    setSelectedBenefits([]);
    setDatePosted('all');
    setSalaryRange(DEFAULT_SALARY_RANGE);
  };

  const withdrawApplication = async (appId) => {
    if (!window.confirm('Withdraw this application?')) return;
    
    try {
      await api.patch(`/jobs/applications/${appId}/status`, { status: 'withdrawn' });
      fetchMyApplications();
      alert('Application withdrawn successfully');
    } catch (error) {
      console.error('Failed to withdraw application:', error);
      alert('Failed to withdraw application');
    }
  };

  const recommendedJobs = [...jobs]
    .map(job => ({ job, score: getJobMatchScore(job) }))
    .filter(x => x.score > 0)
    .sort((a, b) => b.score - a.score)
    .slice(0, 3);

  const trendingJobs = [...jobs]
    .sort((a, b) => (b.views || 0) - (a.views || 0))
    .slice(0, 3);

  const toggleSaveJob = (jobId) => {
    if (savedJobs.includes(jobId)) {
      setSavedJobs(savedJobs.filter(id => id !== jobId));
    } else {
      setSavedJobs([...savedJobs, jobId]);
    }
  };

  const handleTrackView = async (jobId) => {
    try {
      await api.post(`/job-payments/track/view/${jobId}`);
    } catch (error) {
      console.error('Failed to track view:', error);
    }
  };

  const handleApply = async (job) => {
    if (!user) {
      alert('Please login to apply for jobs');
      navigate('/login');
      return;
    }
    
    // Open apply modal
    openApplyModal(job);
  };

  const isMyJob = (job) => {
    return user && job.postedBy === user._id;
  };

  const getSimilarJobs = (job) => {
    return jobs
      .filter(j => j._id !== job._id)
      .filter(j => 
        j.level === job.level ||
        j.techStack?.some(tech => job.techStack?.includes(tech))
      )
      .slice(0, 3);
  };

  const openJobDetail = (job) => {
    setSelectedJob(job);
    setActiveTab('overview');
    handleTrackView(job._id);
  };

  const closeJobDetail = () => {
    setSelectedJob(null);
  };

  const openApplyModal = (job) => {
    setSelectedJob(job);
    setShowApplyModal(true);
    // Pre-fill template
    const template = coverLetterTemplates[selectedTemplate]
      .replace('[POSITION]', job.title)
      .replace('[COMPANY]', job.company)
      .replace('[COMPANY]', job.company)
      .replace('[SKILLS]', job.techStack?.slice(0, 3).join(', ') || 'technology');
    setApplicationData(prev => ({ ...prev, coverLetter: template }));
  };

  const handleFileUpload = (e) => {
    const file = e.target.files[0];
    if (file) {
      setApplicationData(prev => ({ ...prev, resume: file }));
    }
  };

  const handleTemplateChange = (template) => {
    setSelectedTemplate(template);
    if (selectedJob) {
      const templateText = coverLetterTemplates[template]
        .replace('[POSITION]', selectedJob.title)
        .replace('[COMPANY]', selectedJob.company)
        .replace('[COMPANY]', selectedJob.company)
        .replace('[SKILLS]', selectedJob.techStack?.slice(0, 3).join(', ') || 'technology');
      setApplicationData(prev => ({ ...prev, coverLetter: templateText }));
    }
  };

  const submitApplication = async () => {
    if (!applicationData.coverLetter.trim()) {
      alert('Please write a cover letter');
      return;
    }

    setUploading(true);
    try {
      // Submit application through API
      const response = await api.post(`/jobs/${selectedJob._id}/apply`, {
        resume: applicationData.resume?.name || 'Resume.pdf',
        coverLetter: applicationData.coverLetter,
        email: applicationData.email,
        phone: applicationData.phone,
        linkedIn: applicationData.linkedIn,
        portfolio: applicationData.portfolio,
      });

      // Track metrics
      await api.post(`/job-payments/track/click/${selectedJob._id}`).catch(() => {});
      await api.post(`/job-payments/track/application/${selectedJob._id}`).catch(() => {});

      // Refresh applications list
      fetchMyApplications();

      setShowApplyModal(false);
      setApplicationData({
        resume: null,
        coverLetter: '',
        email: user?.email || '',
        phone: '',
        linkedIn: '',
        portfolio: ''
      });

      alert('Application submitted successfully!');
    } catch (error) {
      console.error('Failed to submit application:', error);
      if (error.response?.data?.message === 'You have already applied to this job') {
        alert('You have already applied to this job!');
      } else {
        alert(error.response?.data?.message || 'Failed to submit application. Please try again.');
      }
    } finally {
      setUploading(false);
    }
  };

  const getApplicationStatus = (jobId) => {
    return applications.find(app => app.job?._id === jobId);
  };

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="mb-8 flex justify-between items-center flex-wrap gap-4">
          <div>
            <h1 className="text-4xl font-bold text-white mb-4">Find Your Next Opportunity</h1>
            <p className="text-gray-300">Browse {jobs.length} job openings from top companies</p>
          </div>
          <div className="flex gap-3">
            <button
              onClick={() => setShowApplicationHistory(!showApplicationHistory)}
              className="btn-secondary flex items-center gap-2"
            >
              <FaHistory /> My Applications ({applications.length})
            </button>
            {user && (
              <>
                <button
                  onClick={() => {
                    setShowEmployerDashboard(!showEmployerDashboard);
                    if (!showEmployerDashboard) fetchMyPostedJobs();
                  }}
                  className="btn-secondary flex items-center gap-2"
                >
                  <FaUserTie /> Employer Dashboard
                </button>
                <button
                  onClick={() => setShowPostJobModal(true)}
                  className="btn-primary flex items-center gap-2 bg-gradient-to-r from-turquoise to-gold"
                >
                  <FaBriefcase /> Post a Job
                </button>
              </>
            )}
          </div>
        </div>

        {/* Recommendations + Trending */}
        {jobs.length > 0 && (recommendedJobs.length > 0 || trendingJobs.length > 0) && (
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
            {recommendedJobs.length > 0 && (
              <div className="bg-gray-800 rounded-xl p-6 border border-gray-700">
                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-xl font-bold text-white">Recommended for you</h2>
                  <span className="text-sm text-gray-400">Based on your skills</span>
                </div>
                <div className="space-y-3">
                  {recommendedJobs.map(({ job, score }) => (
                    <button
                      key={job._id}
                      onClick={() => openJobDetail(job)}
                      className="w-full text-left bg-gray-700/40 hover:bg-gray-700 transition-colors rounded-lg p-4 border border-gray-600"
                    >
                      <div className="flex items-start justify-between gap-3">
                        <div>
                          <div className="font-semibold text-white">{job.title}</div>
                          <div className="text-sm text-turquoise-400">{job.company}</div>
                        </div>
                        <div className={`text-xs font-semibold px-2 py-1 rounded-full ${score >= 70 ? 'bg-green-600/20 text-green-400' : 'bg-blue-600/20 text-blue-400'}`}>
                          {score}% match
                        </div>
                      </div>
                      <div className="text-xs text-gray-400 mt-2">{job.location} • {job.type}</div>
                    </button>
                  ))}
                </div>
              </div>
            )}

            {trendingJobs.length > 0 && (
              <div className="bg-gray-800 rounded-xl p-6 border border-gray-700">
                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-xl font-bold text-white">Trending jobs</h2>
                  <span className="text-sm text-gray-400">Most viewed</span>
                </div>
                <div className="space-y-3">
                  {trendingJobs.map((job) => (
                    <button
                      key={job._id}
                      onClick={() => openJobDetail(job)}
                      className="w-full text-left bg-gray-700/40 hover:bg-gray-700 transition-colors rounded-lg p-4 border border-gray-600"
                    >
                      <div className="flex items-start justify-between gap-3">
                        <div>
                          <div className="font-semibold text-white">{job.title}</div>
                          <div className="text-sm text-turquoise-400">{job.company}</div>
                        </div>
                        <div className="text-xs font-semibold px-2 py-1 rounded-full bg-purple-600/20 text-purple-400">
                          {job.views || 0} views
                        </div>
                      </div>
                      <div className="text-xs text-gray-400 mt-2">{job.location} • {job.type}</div>
                    </button>
                  ))}
                </div>
              </div>
            )}
          </div>
        )}

        {/* Advanced Search Bar */}
        <div className="bg-gray-800 rounded-xl p-6 mb-6 border border-gray-700">
          <div className="flex gap-4 mb-4">
            <div className="flex-1 relative">
              <FaSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400" />
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search jobs by title, company, or keywords..."
                className="w-full pl-12 pr-4 py-3 bg-gray-700 border border-gray-600 rounded-lg text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-turquoise-500"
              />
            </div>
            <button
              onClick={() => setShowFilters(!showFilters)}
              className="px-6 py-3 bg-gray-700 text-white rounded-lg hover:bg-gray-600 transition-colors flex items-center gap-2"
            >
              <FaFilter />
              Filters {activeFilterCount > 0 && `(${activeFilterCount})`}
            </button>

            <button
              onClick={handleSaveSearch}
              className="px-6 py-3 bg-gray-700 text-white rounded-lg hover:bg-gray-600 transition-colors flex items-center gap-2"
              title="Save this filter combination"
            >
              <FaBookmark /> Save Search
            </button>
          </div>

          {/* Saved Searches */}
          {savedSearches.length > 0 && (
            <div className="flex flex-col md:flex-row gap-3 mb-4">
              <div className="flex-1">
                <label className="block text-sm font-medium text-gray-300 mb-1">Saved searches</label>
                <select
                  value={selectedSavedSearchId}
                  onChange={(e) => {
                    const id = e.target.value;
                    setSelectedSavedSearchId(id);
                    const saved = savedSearches.find(s => s.id === id);
                    if (saved) applySavedSearch(saved);
                  }}
                  className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                >
                  <option value="">Select a saved search…</option>
                  {savedSearches.map(s => (
                    <option key={s.id} value={s.id}>{s.name}</option>
                  ))}
                </select>
              </div>
              {selectedSavedSearchId && (
                <div className="flex items-end gap-2">
                  <button
                    onClick={() => deleteSavedSearch(selectedSavedSearchId)}
                    className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                  >
                    Delete
                  </button>
                </div>
              )}
            </div>
          )}

          {/* Active Filter Tags */}
          {activeFilterCount > 0 && (
            <div className="flex flex-wrap gap-2 mb-4">
              {searchQuery && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Search: {searchQuery}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setSearchQuery('')} />
                </span>
              )}
              {selectedTechStack.map(tech => (
                <span key={tech} className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  {tech}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setSelectedTechStack(selectedTechStack.filter(t => t !== tech))} />
                </span>
              ))}
              {selectedBenefits.map(benefit => (
                <span key={benefit} className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  {benefit}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setSelectedBenefits(selectedBenefits.filter(b => b !== benefit))} />
                </span>
              ))}
              {selectedCompanySize.map(size => (
                <span key={size} className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Size: {size}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setSelectedCompanySize(selectedCompanySize.filter(s => s !== size))} />
                </span>
              ))}
              {(salaryRange[0] !== DEFAULT_SALARY_RANGE[0] || salaryRange[1] !== DEFAULT_SALARY_RANGE[1]) && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Salary: {formatCurrency(salaryRange[0])} – {formatCurrency(salaryRange[1])}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setSalaryRange(DEFAULT_SALARY_RANGE)} />
                </span>
              )}
              {filters.location !== 'all' && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Location: {filters.location}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setFilters({...filters, location: 'all'})} />
                </span>
              )}
              {filters.type !== 'all' && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Type: {filters.type}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setFilters({...filters, type: 'all'})} />
                </span>
              )}
              {filters.level !== 'all' && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Level: {filters.level}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setFilters({...filters, level: 'all'})} />
                </span>
              )}
              {datePosted !== 'all' && (
                <span className="px-3 py-1 bg-turquoise-600 text-white rounded-full text-sm flex items-center gap-2">
                  Posted: {datePostedOptions.find(o => o.value === datePosted)?.label}
                  <FaTimesCircle className="cursor-pointer" onClick={() => setDatePosted('all')} />
                </span>
              )}
              <button
                onClick={clearAllFilters}
                className="px-3 py-1 bg-red-600 text-white rounded-full text-sm hover:bg-red-700 transition-colors"
              >
                Clear All
              </button>
            </div>
          )}

          {/* Expanded Filters */}
          {showFilters && (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 pt-6 border-t border-gray-700">
              {/* Tech Stack Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Tech Stack</label>
                <div className="flex flex-wrap gap-2 max-h-40 overflow-y-auto">
                  {techStackOptions.map(tech => (
                    <button
                      key={tech}
                      onClick={() => {
                        if (selectedTechStack.includes(tech)) {
                          setSelectedTechStack(selectedTechStack.filter(t => t !== tech));
                        } else {
                          setSelectedTechStack([...selectedTechStack, tech]);
                        }
                      }}
                      className={`px-3 py-1 rounded-full text-sm transition-colors ${
                        selectedTechStack.includes(tech)
                          ? 'bg-turquoise-600 text-white'
                          : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                      }`}
                    >
                      {tech}
                    </button>
                  ))}
                </div>
              </div>

              {/* Location Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Location Type</label>
                <select
                  value={filters.location}
                  onChange={(e) => setFilters({ ...filters, location: e.target.value })}
                  className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                >
                  <option value="all">All Locations</option>
                  <option value="remote">Remote</option>
                  <option value="onsite">On-site</option>
                  <option value="hybrid">Hybrid</option>
                </select>
              </div>

              {/* Job Type Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Job Type</label>
                <select
                  value={filters.type}
                  onChange={(e) => setFilters({ ...filters, type: e.target.value })}
                  className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                >
                  <option value="all">All Types</option>
                  <option value="fulltime">Full-time</option>
                  <option value="parttime">Part-time</option>
                  <option value="contract">Contract</option>
                </select>
              </div>

              {/* Experience Level Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Experience Level</label>
                <select
                  value={filters.level}
                  onChange={(e) => setFilters({ ...filters, level: e.target.value })}
                  className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                >
                  <option value="all">All Levels</option>
                  <option value="junior">Junior</option>
                  <option value="mid">Mid-level</option>
                  <option value="senior">Senior</option>
                </select>
              </div>

              {/* Date Posted Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Date Posted</label>
                <select
                  value={datePosted}
                  onChange={(e) => setDatePosted(e.target.value)}
                  className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                >
                  {datePostedOptions.map(option => (
                    <option key={option.value} value={option.value}>{option.label}</option>
                  ))}
                </select>
              </div>

              {/* Benefits Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Benefits</label>
                <div className="flex flex-wrap gap-2 max-h-40 overflow-y-auto">
                  {benefitsOptions.map(benefit => (
                    <button
                      key={benefit}
                      onClick={() => {
                        if (selectedBenefits.includes(benefit)) {
                          setSelectedBenefits(selectedBenefits.filter(b => b !== benefit));
                        } else {
                          setSelectedBenefits([...selectedBenefits, benefit]);
                        }
                      }}
                      className={`px-3 py-1 rounded-full text-sm transition-colors ${
                        selectedBenefits.includes(benefit)
                          ? 'bg-turquoise-600 text-white'
                          : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                      }`}
                    >
                      {benefit}
                    </button>
                  ))}
                </div>
              </div>

              {/* Company Size Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Company Size</label>
                <div className="flex flex-wrap gap-2 max-h-40 overflow-y-auto">
                  {companySizeOptions.map(size => (
                    <button
                      key={size}
                      onClick={() => {
                        if (selectedCompanySize.includes(size)) {
                          setSelectedCompanySize(selectedCompanySize.filter(s => s !== size));
                        } else {
                          setSelectedCompanySize([...selectedCompanySize, size]);
                        }
                      }}
                      className={`px-3 py-1 rounded-full text-sm transition-colors ${
                        selectedCompanySize.includes(size)
                          ? 'bg-turquoise-600 text-white'
                          : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                      }`}
                    >
                      {size}
                    </button>
                  ))}
                </div>
              </div>

              {/* Salary Range Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-300 mb-2">Salary Range (annualized)</label>
                <div className="bg-gray-700/40 border border-gray-600 rounded-lg p-4">
                  <div className="flex justify-between text-sm text-gray-300 mb-3">
                    <span>{formatCurrency(salaryRange[0])}</span>
                    <span>{formatCurrency(salaryRange[1])}</span>
                  </div>
                  <div className="space-y-3">
                    <input
                      type="range"
                      min="0"
                      max="300000"
                      step="5000"
                      value={salaryRange[0]}
                      onChange={(e) => {
                        const nextMin = Number(e.target.value);
                        setSalaryRange([Math.min(nextMin, salaryRange[1]), salaryRange[1]]);
                      }}
                      className="w-full"
                    />
                    <input
                      type="range"
                      min="0"
                      max="300000"
                      step="5000"
                      value={salaryRange[1]}
                      onChange={(e) => {
                        const nextMax = Number(e.target.value);
                        setSalaryRange([salaryRange[0], Math.max(nextMax, salaryRange[0])]);
                      }}
                      className="w-full"
                    />
                    <button
                      onClick={() => setSalaryRange(DEFAULT_SALARY_RANGE)}
                      className="text-sm text-turquoise-400 hover:text-turquoise-300"
                    >
                      Reset salary range
                    </button>
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* Sort Options */}
          <div className="flex justify-between items-center mt-4 pt-4 border-t border-gray-700">
            <div className="text-sm text-gray-400">
              Showing {sortedJobs.length} of {jobs.length} jobs
            </div>
            <div className="flex items-center gap-2">
              <label className="text-sm text-gray-400">Sort by:</label>
              <select
                value={sortBy}
                onChange={(e) => setSortBy(e.target.value)}
                className="px-3 py-1 bg-gray-700 border border-gray-600 rounded-lg text-white text-sm focus:outline-none focus:ring-2 focus:ring-turquoise-500"
              >
                <option value="relevance">Relevance</option>
                <option value="date">Newest First</option>
                <option value="salary">Highest Salary</option>
              </select>
            </div>
          </div>
        </div>

        {/* Jobs List */}
        {loading ? (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise mx-auto"></div>
          </div>
        ) : loadError ? (
          <div className="text-center py-12">
            <p className="text-red-400 text-lg">{loadError}</p>
            <button onClick={() => { setLoading(true); fetchJobs(); }} className="mt-4 text-turquoise-500 hover:text-turquoise-400">
              Retry
            </button>
          </div>
        ) : sortedJobs.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-400 text-lg">No jobs match your filters.</p>
            <button onClick={clearAllFilters} className="mt-4 text-turquoise-500 hover:text-turquoise-400">
              Clear all filters
            </button>
            {jobs.length === 0 && (
              <div className="mt-6 text-sm text-gray-500">
                <div>No jobs are currently available in the database.</div>
                <div className="mt-2">Dev tip: run the server seed script to create demo jobs.</div>
              </div>
            )}
          </div>
        ) : (
          <div className="space-y-4">
            {sortedJobs.map((job) => (
              <div 
                key={job._id} 
                className={`card hover:border-turquoise/50 transition-all cursor-pointer ${
                  job.highlightColor ? `border-l-4` : ''
                }`}
                style={job.highlightColor ? { borderLeftColor: job.highlightColor } : {}}
                onClick={() => openJobDetail(job)}
                onMouseEnter={() => handleTrackView(job._id)}
              >
                {/* Badges (in flow, not absolute) */}
                <div className="flex gap-2 justify-end mb-2">
                  {job.isPinned && (
                    <div className="bg-purple/20 text-purple px-3 py-1 rounded-full text-xs font-semibold flex items-center gap-1">
                      <FaThumbtack /> Pinned
                    </div>
                  )}
                  {job.isFeatured && (
                    <div className="bg-turquoise/20 text-turquoise px-3 py-1 rounded-full text-xs font-semibold flex items-center gap-1">
                      <FaStar /> Featured
                    </div>
                  )}
                </div>

                <div className="flex flex-col md:flex-row justify-between gap-4 pt-2 md:pt-0">
                  {/* Job Info */}
                  <div className="flex-1">
                    <div className="flex items-start gap-4">
                      {/* Company Logo */}
                      <div className="w-16 h-16 bg-gray-800 rounded-lg flex items-center justify-center flex-shrink-0">
                        <img
                          src={job.companyLogo || companyLogoFallbackDataUri(job.company, job.highlightColor)}
                          alt={job.company}
                          className="w-12 h-12 object-contain"
                          loading="lazy"
                          onError={(e) => {
                            e.currentTarget.onerror = null;
                            e.currentTarget.src = companyLogoFallbackDataUri(job.company, job.highlightColor);
                          }}
                        />
                      </div>

                      <div className="flex-1">
                        <h3 className="text-xl font-bold mb-1">{job.title}</h3>
                        <p className="text-turquoise font-semibold mb-2">{job.company}</p>
                        
                        {/* Job Details */}
                        <div className="flex flex-wrap gap-4 text-sm text-gray-400 mb-3">
                          <div className="flex items-center gap-1">
                            <FaMapMarkerAlt className="text-turquoise" />
                            <span>{job.location}</span>
                          </div>
                          <div className="flex items-center gap-1">
                            <FaBriefcase className="text-turquoise" />
                            <span>{job.type}</span>
                          </div>
                          {job.companySize && (
                            <div className="flex items-center gap-1">
                              <FaUsers className="text-turquoise" />
                              <span>{job.companySize}</span>
                            </div>
                          )}
                          {job.salary && (
                            <div className="flex items-center gap-1">
                              <FaDollarSign className="text-turquoise" />
                              <span>{job.salary}</span>
                            </div>
                          )}
                          <div className="flex items-center gap-1">
                            <FaClock className="text-turquoise" />
                            <span>{new Date(job.createdAt).toLocaleDateString()}</span>
                          </div>
                        </div>

                        <p className="text-gray-400 mb-3 line-clamp-2">{job.description}</p>

                        {/* Tech Stack */}
                        <div className="flex flex-wrap gap-2">
                          {job.techStack?.map((tech, index) => (
                            <span
                              key={index}
                              className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded"
                            >
                              {tech}
                            </span>
                          ))}
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Action Buttons */}
                  <div className="flex flex-col gap-2 items-end mt-20 md:mt-0 min-h-[80px]">
                    {/* Save Button - Available for everyone */}
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        toggleSaveJob(job._id);
                      }}
                      className="p-2 text-gray-400 hover:text-turquoise-500 transition-colors"
                      title={savedJobs.includes(job._id) ? "Remove from saved" : "Save job"}
                    >
                      {savedJobs.includes(job._id) ? (
                        <FaBookmark className="text-xl text-turquoise-500" />
                      ) : (
                        <FaRegBookmark className="text-xl" />
                      )}
                    </button>

                    {isMyJob(job) ? (
                      <>
                        <button 
                          onClick={(e) => {
                            e.stopPropagation();
                            navigate(`/jobs/${job._id}/analytics`);
                          }}
                          className="btn-secondary whitespace-nowrap flex items-center gap-2"
                        >
                          <FaChartBar /> Analytics
                        </button>
                        {job.listingTier === 'free' && (
                          <button 
                            onClick={(e) => {
                              e.stopPropagation();
                              navigate(`/jobs/${job._id}/pricing`);
                            }}
                            className="btn-primary whitespace-nowrap"
                          >
                            Boost Job
                          </button>
                        )}
                      </>
                    ) : (
                      <div className="flex flex-col gap-2">
                        {getApplicationStatus(job._id) ? (
                          <button 
                            onClick={(e) => e.stopPropagation()}
                            className="btn-secondary whitespace-nowrap flex items-center gap-2"
                            disabled
                          >
                            <FaCheckCircle /> Applied
                          </button>
                        ) : (
                          <button 
                            onClick={(e) => {
                              e.stopPropagation();
                              openApplyModal(job);
                            }}
                            className="btn-primary whitespace-nowrap"
                          >
                            Apply Now
                          </button>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}

        {/* Job Detail Modal */}
        {selectedJob && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4 overflow-y-auto">
            <div className="bg-gray-800 rounded-2xl max-w-5xl w-full max-h-[90vh] overflow-y-auto border border-gray-700 my-8">
              {/* Modal Header */}
              <div className="sticky top-0 bg-gray-800 border-b border-gray-700 p-6 z-10">
                <div className="flex justify-between items-start">
                  <div className="flex gap-4 flex-1">
                    <div className="w-20 h-20 bg-gray-700 rounded-lg flex items-center justify-center flex-shrink-0">
                      <img
                        src={selectedJob.companyLogo || companyLogoFallbackDataUri(selectedJob.company, selectedJob.highlightColor)}
                        alt={selectedJob.company}
                        className="w-16 h-16 object-contain"
                        loading="lazy"
                        onError={(e) => {
                          e.currentTarget.onerror = null;
                          e.currentTarget.src = companyLogoFallbackDataUri(selectedJob.company, selectedJob.highlightColor);
                        }}
                      />
                    </div>
                    <div className="flex-1">
                      <h2 className="text-3xl font-bold text-white mb-2">{selectedJob.title}</h2>
                      <p className="text-turquoise-500 text-xl font-semibold mb-3">{selectedJob.company}</p>
                      <div className="flex flex-wrap gap-4 text-sm text-gray-400">
                        <div className="flex items-center gap-1">
                          <FaMapMarkerAlt className="text-turquoise-500" />
                          <span>{selectedJob.location}</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <FaBriefcase className="text-turquoise-500" />
                          <span className="capitalize">{selectedJob.type}</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <FaUserTie className="text-turquoise-500" />
                          <span className="capitalize">{selectedJob.level}</span>
                        </div>
                        {selectedJob.salary && (
                          <div className="flex items-center gap-1">
                            <FaDollarSign className="text-turquoise-500" />
                            <span>{selectedJob.salary}</span>
                          </div>
                        )}
                        {selectedJob.companySize && (
                          <div className="flex items-center gap-1">
                            <FaUsers className="text-turquoise-500" />
                            <span>{selectedJob.companySize}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                  <button
                    onClick={closeJobDetail}
                    className="p-2 hover:bg-gray-700 rounded-lg transition-colors"
                  >
                    <FaTimes className="text-2xl text-gray-400 hover:text-white" />
                  </button>
                </div>

                <div className="flex gap-2 mt-4">
                  <button
                    onClick={() => shareJob('linkedin', selectedJob)}
                    className="p-2 bg-gray-700 rounded-lg hover:bg-[#0077B5] transition-colors group"
                    title="Share on LinkedIn"
                  >
                    <FaLinkedin className="text-white group-hover:text-white" />
                  </button>
                  <button
                    onClick={() => shareJob('twitter', selectedJob)}
                    className="p-2 bg-gray-700 rounded-lg hover:bg-black transition-colors group"
                    title="Share on X/Twitter"
                  >
                    <FaTwitter className="text-white group-hover:text-white" />
                  </button>
                  <button
                    onClick={() => shareJob('facebook', selectedJob)}
                    className="p-2 bg-gray-700 rounded-lg hover:bg-[#1877F2] transition-colors group"
                    title="Share on Facebook"
                  >
                    <FaFacebook className="text-white group-hover:text-white" />
                  </button>
                </div>

                {/* Tabs */}
                <div className="flex gap-2 mt-6 overflow-x-auto">
                  {['overview', 'company', 'benefits', 'team', 'similar'].map((tab) => (
                    <button
                      key={tab}
                      onClick={() => setActiveTab(tab)}
                      className={`px-4 py-2 rounded-lg font-medium whitespace-nowrap transition-colors ${
                        activeTab === tab
                          ? 'bg-turquoise-600 text-white'
                          : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                      }`}
                    >
                      {tab.charAt(0).toUpperCase() + tab.slice(1)}
                    </button>
                  ))}
                </div>
              </div>

              {/* Modal Content */}
              <div className="p-6">
                {/* Overview Tab */}
                {activeTab === 'overview' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-xl font-bold text-white mb-4 flex items-center gap-2">
                        <FaBriefcase className="text-turquoise-500" />
                        Job Description
                      </h3>
                      <div className="text-gray-300 whitespace-pre-line leading-relaxed">
                        {selectedJob.description}
                      </div>
                    </div>

                    <div className="bg-gray-700/40 rounded-xl p-5 border border-gray-600">
                      <h3 className="text-xl font-bold text-white mb-3 flex items-center gap-2">
                        <FaCalendarAlt className="text-turquoise-500" />
                        Application Timeline
                      </h3>
                      <div className="text-sm text-gray-300 space-y-2">
                        <div><span className="text-gray-400">Response time:</span> 3–7 business days</div>
                        <div><span className="text-gray-400">Interview process:</span> Recruiter screen → Technical interview → Final round</div>
                        <div><span className="text-gray-400">Tip:</span> Save the job and track your application status.</div>
                      </div>
                    </div>

                    <div>
                      <h3 className="text-xl font-bold text-white mb-4">Required Skills</h3>
                      <div className="flex flex-wrap gap-2">
                        {selectedJob.techStack?.map((tech, index) => (
                          <span
                            key={index}
                            className="px-4 py-2 bg-turquoise-600/20 text-turquoise-400 rounded-lg font-medium"
                          >
                            {tech}
                          </span>
                        ))}
                      </div>
                    </div>

                    {selectedJob.companyBio && (
                      <div>
                        <h3 className="text-xl font-bold text-white mb-4">About the Role</h3>
                        <p className="text-gray-300 leading-relaxed">{selectedJob.companyBio}</p>
                      </div>
                    )}

                    <div className="flex gap-4 pt-4">
                      <button
                        onClick={() => handleApply(selectedJob)}
                        className="btn-primary flex-1 flex items-center justify-center gap-2"
                      >
                        Apply Now <FaExternalLinkAlt />
                      </button>
                      <button
                        onClick={() => toggleSaveJob(selectedJob._id)}
                        className={`px-6 py-3 rounded-lg font-semibold transition-colors flex items-center gap-2 ${
                          savedJobs.includes(selectedJob._id)
                            ? 'bg-turquoise-600 text-white'
                            : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                        }`}
                      >
                        {savedJobs.includes(selectedJob._id) ? <FaBookmark /> : <FaRegBookmark />}
                        {savedJobs.includes(selectedJob._id) ? 'Saved' : 'Save Job'}
                      </button>
                    </div>
                  </div>
                )}

                {/* Company Tab */}
                {activeTab === 'company' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-2xl font-bold text-white mb-4 flex items-center gap-2">
                        <FaBuilding className="text-turquoise-500" />
                        About {selectedJob.company}
                      </h3>
                      <p className="text-gray-300 leading-relaxed mb-6">
                        {selectedJob.companyBio || `${selectedJob.company} is a leading company in the tech industry, committed to innovation and excellence. Join our team and help shape the future of technology.`}
                      </p>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="bg-gray-700/50 rounded-xl p-4">
                        <div className="flex items-center gap-2 mb-2">
                          <FaUsers className="text-turquoise-500" />
                          <h4 className="font-semibold text-white">Company Size</h4>
                        </div>
                        <p className="text-gray-300">500-1000 employees</p>
                      </div>

                      <div className="bg-gray-700/50 rounded-xl p-4">
                        <div className="flex items-center gap-2 mb-2">
                          <FaGlobeAmericas className="text-turquoise-500" />
                          <h4 className="font-semibold text-white">Industry</h4>
                        </div>
                        <p className="text-gray-300">Technology & Software</p>
                      </div>

                      <div className="bg-gray-700/50 rounded-xl p-4">
                        <div className="flex items-center gap-2 mb-2">
                          <FaRocket className="text-turquoise-500" />
                          <h4 className="font-semibold text-white">Founded</h4>
                        </div>
                        <p className="text-gray-300">2015</p>
                      </div>

                      <div className="bg-gray-700/50 rounded-xl p-4">
                        <div className="flex items-center gap-2 mb-2">
                          <FaMapMarkerAlt className="text-turquoise-500" />
                          <h4 className="font-semibold text-white">Headquarters</h4>
                        </div>
                        <p className="text-gray-300">{selectedJob.location}</p>
                      </div>
                    </div>

                    <div>
                      <h4 className="font-semibold text-white mb-3 flex items-center gap-2">
                        <FaLightbulb className="text-turquoise-500" />
                        Company Culture
                      </h4>
                      <div className="flex flex-wrap gap-2">
                        {['Innovation-Driven', 'Collaborative', 'Work-Life Balance', 'Growth Mindset', 'Inclusive'].map((value) => (
                          <span key={value} className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-sm">
                            {value}
                          </span>
                        ))}
                      </div>
                    </div>

                    <div>
                      <h4 className="font-semibold text-white mb-3">Connect With Us</h4>
                      <div className="flex gap-3">
                        <button className="p-3 bg-gray-700 rounded-lg hover:bg-[#0077B5] transition-colors group">
                          <FaLinkedin className="text-xl text-white group-hover:text-white" />
                        </button>
                        <button className="p-3 bg-gray-700 rounded-lg hover:bg-black transition-colors group">
                          <FaTwitter className="text-xl text-white group-hover:text-white" />
                        </button>
                        <button className="p-3 bg-gray-700 rounded-lg hover:bg-gray-600 transition-colors">
                          <FaGlobeAmericas className="text-xl text-turquoise-400" />
                        </button>
                      </div>
                    </div>
                  </div>
                )}

                {/* Benefits Tab */}
                {activeTab === 'benefits' && (
                  <div className="space-y-6">
                    <h3 className="text-2xl font-bold text-white mb-4">Benefits & Perks</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      {(selectedJob.benefits || [
                        'Health Insurance',
                        'Dental & Vision',
                        '401k Matching',
                        'Stock Options',
                        'Unlimited PTO',
                        'Remote Work',
                        'Flexible Hours',
                        'Learning Budget',
                        'Gym Membership',
                        'Free Lunch',
                        'Team Events',
                        'Parental Leave'
                      ]).map((benefit, index) => (
                        <div
                          key={index}
                          className="flex items-center gap-3 bg-gray-700/50 rounded-xl p-4 hover:bg-gray-700 transition-colors"
                        >
                          <FaCheckCircle className="text-turquoise-500 text-xl flex-shrink-0" />
                          <span className="text-gray-300 font-medium">{benefit}</span>
                        </div>
                      ))}
                    </div>

                    <div className="bg-gradient-to-r from-turquoise-600/20 to-purple-600/20 rounded-xl p-6 border border-turquoise-500/30">
                      <h4 className="text-xl font-bold text-white mb-3 flex items-center gap-2">
                        <FaAward className="text-gold-500" />
                        Why Join Us?
                      </h4>
                      <ul className="space-y-2 text-gray-300">
                        <li className="flex items-start gap-2">
                          <span className="text-turquoise-500 mt-1">•</span>
                          <span>Work on cutting-edge technology with industry experts</span>
                        </li>
                        <li className="flex items-start gap-2">
                          <span className="text-turquoise-500 mt-1">•</span>
                          <span>Competitive salary and equity compensation</span>
                        </li>
                        <li className="flex items-start gap-2">
                          <span className="text-turquoise-500 mt-1">•</span>
                          <span>Professional development and career growth opportunities</span>
                        </li>
                        <li className="flex items-start gap-2">
                          <span className="text-turquoise-500 mt-1">•</span>
                          <span>Inclusive and diverse work environment</span>
                        </li>
                      </ul>
                    </div>
                  </div>
                )}

                {/* Team Tab */}
                {activeTab === 'team' && (
                  <div className="space-y-6">
                    <h3 className="text-2xl font-bold text-white mb-4">Meet The Team</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      {[
                        { name: 'Sarah Johnson', role: 'Engineering Manager', avatar: '👩‍💼' },
                        { name: 'Michael Chen', role: 'Senior Developer', avatar: '👨‍💻' },
                        { name: 'Emily Rodriguez', role: 'Product Lead', avatar: '👩‍🔬' },
                        { name: 'David Kim', role: 'Tech Lead', avatar: '👨‍🏫' }
                      ].map((member, index) => (
                        <div
                          key={index}
                          className="bg-gray-700/50 rounded-xl p-6 hover:bg-gray-700 transition-colors"
                        >
                          <div className="flex items-center gap-4">
                            <div className="w-16 h-16 bg-gradient-to-br from-turquoise-500 to-purple-600 rounded-full flex items-center justify-center text-3xl">
                              {member.avatar}
                            </div>
                            <div>
                              <h4 className="text-lg font-bold text-white">{member.name}</h4>
                              <p className="text-turquoise-400">{member.role}</p>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>

                    <div className="bg-gray-700/50 rounded-xl p-6">
                      <h4 className="text-xl font-bold text-white mb-4">Team Culture</h4>
                      <p className="text-gray-300 leading-relaxed mb-4">
                        Our team values collaboration, innovation, and continuous learning. We believe in empowering each team member to bring their unique perspective and contribute to our shared success.
                      </p>
                      <div className="flex flex-wrap gap-2">
                        <span className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-sm">Weekly Team Lunches</span>
                        <span className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-sm">Hackathons</span>
                        <span className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-sm">Mentorship Program</span>
                        <span className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-sm">Team Retreats</span>
                      </div>
                    </div>
                  </div>
                )}

                {/* Similar Jobs Tab */}
                {activeTab === 'similar' && (
                  <div className="space-y-6">
                    <h3 className="text-2xl font-bold text-white mb-4">Similar Opportunities</h3>
                    <div className="space-y-4">
                      {getSimilarJobs(selectedJob).map((job) => (
                        <div
                          key={job._id}
                          onClick={() => openJobDetail(job)}
                          className="bg-gray-700/50 rounded-xl p-4 hover:bg-gray-700 transition-colors cursor-pointer"
                        >
                          <div className="flex justify-between items-start">
                            <div className="flex gap-3 flex-1">
                              <div className="w-12 h-12 bg-gray-800 rounded-lg flex items-center justify-center">
                                <img
                                  src={job.companyLogo || companyLogoFallbackDataUri(job.company, job.highlightColor)}
                                  alt={job.company}
                                  className="w-10 h-10 object-contain"
                                  loading="lazy"
                                  onError={(e) => {
                                    e.currentTarget.onerror = null;
                                    e.currentTarget.src = companyLogoFallbackDataUri(job.company, job.highlightColor);
                                  }}
                                />
                              </div>
                              <div className="flex-1">
                                <h4 className="text-lg font-bold text-white mb-1">{job.title}</h4>
                                <p className="text-turquoise-400 mb-2">{job.company}</p>
                                <div className="flex flex-wrap gap-3 text-sm text-gray-400">
                                  <span className="flex items-center gap-1">
                                    <FaMapMarkerAlt className="text-turquoise-500" />
                                    {job.location}
                                  </span>
                                  <span className="flex items-center gap-1">
                                    <FaBriefcase className="text-turquoise-500" />
                                    {job.type}
                                  </span>
                                  {job.salary && (
                                    <span className="flex items-center gap-1">
                                      <FaDollarSign className="text-turquoise-500" />
                                      {job.salary}
                                    </span>
                                  )}
                                </div>
                              </div>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>

                    {getSimilarJobs(selectedJob).length === 0 && (
                      <div className="text-center py-12">
                        <p className="text-gray-400">No similar jobs found at the moment.</p>
                      </div>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Application Modal */}
        {showApplyModal && selectedJob && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4 overflow-y-auto">
            <div className="bg-gray-800 rounded-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto border border-gray-700 my-8">
              {/* Modal Header */}
              <div className="sticky top-0 bg-gray-800 border-b border-gray-700 p-6 z-10">
                <div className="flex justify-between items-start">
                  <div>
                    <h2 className="text-2xl font-bold text-white mb-2">Apply for {selectedJob.title}</h2>
                    <p className="text-turquoise-500">{selectedJob.company}</p>
                  </div>
                  <button
                    onClick={() => setShowApplyModal(false)}
                    className="p-2 hover:bg-gray-700 rounded-lg transition-colors"
                  >
                    <FaTimes className="text-xl text-gray-400 hover:text-white" />
                  </button>
                </div>
              </div>

              {/* Modal Content */}
              <div className="p-6 space-y-6">
                {/* Contact Information */}
                <div>
                  <h3 className="text-lg font-bold text-white mb-4">Contact Information</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-300 mb-2">Email *</label>
                      <input
                        type="email"
                        value={applicationData.email}
                        onChange={(e) => setApplicationData({ ...applicationData, email: e.target.value })}
                        className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                        placeholder="your.email@example.com"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-300 mb-2">Phone</label>
                      <input
                        type="tel"
                        value={applicationData.phone}
                        onChange={(e) => setApplicationData({ ...applicationData, phone: e.target.value })}
                        className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                        placeholder="+1 (555) 123-4567"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-300 mb-2">LinkedIn</label>
                      <input
                        type="url"
                        value={applicationData.linkedIn}
                        onChange={(e) => setApplicationData({ ...applicationData, linkedIn: e.target.value })}
                        className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                        placeholder="https://linkedin.com/in/yourprofile"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-300 mb-2">Portfolio</label>
                      <input
                        type="url"
                        value={applicationData.portfolio}
                        onChange={(e) => setApplicationData({ ...applicationData, portfolio: e.target.value })}
                        className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500"
                        placeholder="https://yourportfolio.com"
                      />
                    </div>
                  </div>
                </div>

                {/* Resume Upload */}
                <div>
                  <h3 className="text-lg font-bold text-white mb-4">Resume *</h3>
                  <div className="border-2 border-dashed border-gray-600 rounded-lg p-6 text-center hover:border-turquoise-500 transition-colors">
                    <input
                      type="file"
                      id="resume-upload"
                      accept=".pdf,.doc,.docx"
                      onChange={handleFileUpload}
                      className="hidden"
                    />
                    <label htmlFor="resume-upload" className="cursor-pointer">
                      {applicationData.resume ? (
                        <div className="flex items-center justify-center gap-2 text-turquoise-500">
                          <FaFilePdf className="text-3xl" />
                          <div className="text-left">
                            <p className="font-semibold">{applicationData.resume.name}</p>
                            <p className="text-sm text-gray-400">{(applicationData.resume.size / 1024).toFixed(2)} KB</p>
                          </div>
                        </div>
                      ) : (
                        <div>
                          <FaUpload className="text-4xl text-gray-400 mx-auto mb-2" />
                          <p className="text-gray-300 font-medium">Click to upload resume</p>
                          <p className="text-sm text-gray-400 mt-1">PDF, DOC, or DOCX (Max 5MB)</p>
                        </div>
                      )}
                    </label>
                  </div>
                </div>

                {/* Cover Letter */}
                <div>
                  <div className="flex justify-between items-center mb-4">
                    <h3 className="text-lg font-bold text-white">Cover Letter</h3>
                    <div className="flex gap-2">
                      {Object.keys(coverLetterTemplates).map((template) => (
                        <button
                          key={template}
                          onClick={() => handleTemplateChange(template)}
                          className={`px-3 py-1 rounded-lg text-sm transition-colors ${
                            selectedTemplate === template
                              ? 'bg-turquoise-600 text-white'
                              : 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                          }`}
                        >
                          {template.charAt(0).toUpperCase() + template.slice(1)}
                        </button>
                      ))}
                    </div>
                  </div>
                  <textarea
                    value={applicationData.coverLetter}
                    onChange={(e) => setApplicationData({ ...applicationData, coverLetter: e.target.value })}
                    rows={12}
                    className="w-full px-4 py-3 bg-gray-700 border border-gray-600 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-turquoise-500 resize-none"
                    placeholder="Write your cover letter here..."
                  />
                  <div className="flex justify-between items-center mt-2">
                    <p className="text-sm text-gray-400">{applicationData.coverLetter.length} characters</p>
                    <button
                      onClick={() => {
                        navigator.clipboard.writeText(applicationData.coverLetter);
                        alert('Cover letter copied to clipboard!');
                      }}
                      className="text-sm text-turquoise-500 hover:text-turquoise-400 flex items-center gap-1"
                    >
                      <FaCopy /> Copy
                    </button>
                  </div>
                </div>

                {/* Action Buttons */}
                <div className="flex gap-4 pt-4 border-t border-gray-700">
                  <button
                    onClick={() => setShowApplyModal(false)}
                    className="flex-1 px-6 py-3 bg-gray-700 text-white rounded-lg font-semibold hover:bg-gray-600 transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={submitApplication}
                    disabled={uploading || !applicationData.resume}
                    className="flex-1 btn-primary flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    {uploading ? (
                      <>
                        <FaSpinner className="animate-spin" /> Submitting...
                      </>
                    ) : (
                      <>
                        <FaPaperPlane /> Submit Application
                      </>
                    )}
                  </button>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Application History Modal */}
        {showApplicationHistory && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4 overflow-y-auto">
            <div className="bg-gray-800 rounded-2xl max-w-4xl w-full max-h-[90vh] overflow-y-auto border border-gray-700 my-8">
              {/* Modal Header */}
              <div className="sticky top-0 bg-gray-800 border-b border-gray-700 p-6 z-10">
                <div className="flex justify-between items-start">
                  <div>
                    <h2 className="text-2xl font-bold text-white mb-2">Application History</h2>
                    <p className="text-gray-400">{applications.length} applications submitted</p>
                  </div>
                  <button
                    onClick={() => setShowApplicationHistory(false)}
                    className="p-2 hover:bg-gray-700 rounded-lg transition-colors"
                  >
                    <FaTimes className="text-xl text-gray-400 hover:text-white" />
                  </button>
                </div>
              </div>

              {/* Modal Content */}
              <div className="p-6">
                {applications.length === 0 ? (
                  <div className="text-center py-12">
                    <FaHistory className="text-6xl text-gray-600 mx-auto mb-4" />
                    <p className="text-gray-400 text-lg">No applications yet</p>
                    <p className="text-gray-500 text-sm mt-2">Start applying to jobs to track your applications here</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {applications.map((app) => {
                      // Skip if job was deleted
                      if (!app.job) return null;
                      
                      return (
                      <div
                        key={app._id}
                        className="bg-gray-700/50 rounded-xl p-6 border border-gray-600 hover:border-turquoise-500/50 transition-colors"
                      >
                        <div className="flex justify-between items-start mb-4">
                          <div className="flex gap-4 flex-1">
                            <div className="w-16 h-16 bg-gray-800 rounded-lg flex items-center justify-center flex-shrink-0">
                              <img
                                src={app.job.companyLogo || companyLogoFallbackDataUri(app.job.company, app.job.highlightColor)}
                                alt={app.job.company}
                                className="w-12 h-12 object-contain"
                                loading="lazy"
                                onError={(e) => {
                                  e.currentTarget.onerror = null;
                                  e.currentTarget.src = companyLogoFallbackDataUri(app.job.company, app.job.highlightColor);
                                }}
                              />
                            </div>
                            <div className="flex-1">
                              <h3 className="text-xl font-bold text-white mb-1">{app.job.title}</h3>
                              <p className="text-turquoise-500 font-semibold mb-2">{app.job.company}</p>
                              <div className="flex items-center gap-4 text-sm text-gray-400">
                                <span className="flex items-center gap-1">
                                  <FaCalendarAlt className="text-turquoise-500" />
                                  Applied {new Date(app.appliedAt).toLocaleDateString()}
                                </span>
                                <span className="flex items-center gap-1">
                                  <FaFile className="text-turquoise-500" />
                                  {app.resume}
                                </span>
                              </div>
                            </div>
                          </div>
                          
                          {/* Status Badge (read-only for applicants, employer controls it) */}
                          <div className={`px-4 py-2 rounded-lg font-semibold text-sm border-2 ${
                            app.status === 'submitted' ? 'bg-blue-600/20 text-blue-400 border-blue-500' :
                            app.status === 'reviewing' ? 'bg-yellow-600/20 text-yellow-400 border-yellow-500' :
                            app.status === 'interview' ? 'bg-purple-600/20 text-purple-400 border-purple-500' :
                            app.status === 'offer' ? 'bg-green-600/20 text-green-400 border-green-500' :
                            app.status === 'withdrawn' ? 'bg-gray-600/20 text-gray-400 border-gray-500' :
                            'bg-red-600/20 text-red-400 border-red-500'
                          }`}>
                            {app.status === 'submitted' ? 'Submitted' :
                             app.status === 'reviewing' ? 'Under Review' :
                             app.status === 'interview' ? 'Interview' :
                             app.status === 'offer' ? 'Offer Received' :
                             app.status === 'withdrawn' ? 'Withdrawn' :
                             'Not Selected'}
                          </div>
                        </div>

                        <div className="flex justify-end gap-2">
                          {app.status !== 'withdrawn' && app.status !== 'rejected' && (
                            <button
                              onClick={() => withdrawApplication(app._id)}
                              className="px-4 py-2 bg-red-600/20 text-red-400 rounded-lg hover:bg-red-600/30 transition-colors text-sm font-semibold border border-red-500/40"
                            >
                              Withdraw
                            </button>
                          )}
                        </div>

                        {/* Cover Letter Preview */}
                        {app.coverLetter && (
                          <div className="mt-4 pt-4 border-t border-gray-600">
                            <div className="flex justify-between items-center mb-2">
                              <h4 className="text-sm font-semibold text-gray-300">Cover Letter</h4>
                              <button
                                onClick={() => {
                                  navigator.clipboard.writeText(app.coverLetter);
                                  alert('Cover letter copied!');
                                }}
                                className="text-sm text-turquoise-500 hover:text-turquoise-400 flex items-center gap-1"
                              >
                                <FaCopy /> Copy
                              </button>
                            </div>
                            <div className="text-sm text-gray-400 bg-gray-800/50 rounded-lg p-3 max-h-32 overflow-y-auto whitespace-pre-wrap">
                              {app.coverLetter.length > 200 ? `${app.coverLetter.substring(0, 200)}...` : app.coverLetter}
                            </div>
                          </div>
                        )}

                        {/* Interview Details */}
                        {app.status === 'interview' && app.interviewDate && (
                          <div className="mt-4 pt-4 border-t border-gray-600">
                            <div className="flex items-center gap-2 text-purple-400 mb-2">
                              <FaCalendarAlt />
                              <span className="text-sm font-semibold">Interview Scheduled</span>
                            </div>
                            <p className="text-sm text-gray-300">
                              {new Date(app.interviewDate).toLocaleString()}
                            </p>
                            {app.interviewLocation && (
                              <p className="text-sm text-gray-400 mt-1">
                                Location: {app.interviewLocation}
                              </p>
                            )}
                          </div>
                        )}

                        {/* Employer Notes */}
                        {app.employerNotes && (
                          <div className="mt-4 pt-4 border-t border-gray-600">
                            <h4 className="text-sm font-semibold text-gray-300 mb-2">Notes from Employer</h4>
                            <div className="text-sm text-gray-400 bg-gray-800/50 rounded-lg p-3">
                              {app.employerNotes}
                            </div>
                          </div>
                        )}
                        {app.status === 'interview' && (
                          <div className="mt-4 pt-4 border-t border-gray-600">
                            <div className="flex items-center gap-2 text-purple-400">
                              <FaCalendarAlt />
                              <span className="text-sm font-semibold">Interview Scheduled</span>
                            </div>
                            <p className="text-sm text-gray-400 mt-2">
                              Keep track of your interview dates and preparation notes
                            </p>
                          </div>
                        )}
                      </div>
                      );
                    })}
                  </div>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Post Job Modal */}
        {showPostJobModal && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="bg-gray-800 rounded-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto border border-gray-700">
              <div className="sticky top-0 bg-gray-800 border-b border-gray-700 p-6 flex justify-between items-center">
                <h2 className="text-2xl font-bold text-white">Post a Job</h2>
                <button onClick={() => setShowPostJobModal(false)} className="text-gray-400 hover:text-white">
                  <FaTimes size={24} />
                </button>
              </div>

              <div className="p-6 space-y-6">
                {/* Job Title */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Job Title *</label>
                  <input
                    type="text"
                    value={jobFormData.title}
                    onChange={(e) => setJobFormData({ ...jobFormData, title: e.target.value })}
                    placeholder="e.g. Senior Full Stack Developer"
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Company */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Company *</label>
                  <input
                    type="text"
                    value={jobFormData.company}
                    onChange={(e) => setJobFormData({ ...jobFormData, company: e.target.value })}
                    placeholder="Your company name"
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Company Logo URL */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Company Logo URL</label>
                  <input
                    type="text"
                    value={jobFormData.companyLogo}
                    onChange={(e) => setJobFormData({ ...jobFormData, companyLogo: e.target.value })}
                    placeholder="https://..."
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Description */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Description *</label>
                  <textarea
                    value={jobFormData.description}
                    onChange={(e) => setJobFormData({ ...jobFormData, description: e.target.value })}
                    placeholder="Describe the role, responsibilities, and requirements..."
                    rows={6}
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Location */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-semibold text-gray-300 mb-2">Location *</label>
                    <input
                      type="text"
                      value={jobFormData.location}
                      onChange={(e) => setJobFormData({ ...jobFormData, location: e.target.value })}
                      placeholder="e.g. San Francisco, CA or Remote"
                      className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                    />
                  </div>

                  <div className="flex items-end">
                    <label className="flex items-center gap-2 text-gray-300 cursor-pointer">
                      <input
                        type="checkbox"
                        checked={jobFormData.isRemote}
                        onChange={(e) => setJobFormData({ ...jobFormData, isRemote: e.target.checked })}
                        className="w-5 h-5 bg-gray-700 border-gray-600 rounded text-turquoise-500 focus:ring-turquoise-500"
                      />
                      <span className="font-semibold">Remote Position</span>
                    </label>
                  </div>
                </div>

                {/* Type and Level */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-semibold text-gray-300 mb-2">Job Type *</label>
                    <select
                      value={jobFormData.type}
                      onChange={(e) => setJobFormData({ ...jobFormData, type: e.target.value })}
                      className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                    >
                      <option value="fulltime">Full-time</option>
                      <option value="parttime">Part-time</option>
                      <option value="contract">Contract</option>
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-semibold text-gray-300 mb-2">Experience Level *</label>
                    <select
                      value={jobFormData.level}
                      onChange={(e) => setJobFormData({ ...jobFormData, level: e.target.value })}
                      className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                    >
                      <option value="junior">Junior</option>
                      <option value="mid">Mid-level</option>
                      <option value="senior">Senior</option>
                    </select>
                  </div>
                </div>

                {/* Salary */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Salary Range</label>
                  <input
                    type="text"
                    value={jobFormData.salary}
                    onChange={(e) => setJobFormData({ ...jobFormData, salary: e.target.value })}
                    placeholder="e.g. $120,000 - $150,000"
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Tech Stack */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Tech Stack (comma-separated)</label>
                  <input
                    type="text"
                    value={jobFormData.techStack.join(', ')}
                    onChange={(e) => setJobFormData({ ...jobFormData, techStack: e.target.value.split(',').map(s => s.trim()).filter(Boolean) })}
                    placeholder="React, Node.js, MongoDB, AWS"
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Apply URL */}
                <div>
                  <label className="block text-sm font-semibold text-gray-300 mb-2">Application URL or Email *</label>
                  <input
                    type="text"
                    value={jobFormData.applyUrl}
                    onChange={(e) => setJobFormData({ ...jobFormData, applyUrl: e.target.value })}
                    placeholder="https://yourcompany.com/apply or jobs@company.com"
                    className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:ring-2 focus:ring-turquoise-500 focus:outline-none"
                  />
                </div>

                {/* Submit Button */}
                <div className="flex gap-3 pt-4">
                  <button
                    onClick={handlePostJob}
                    className="flex-1 btn-primary bg-gradient-to-r from-turquoise to-gold"
                  >
                    <FaPaperPlane className="inline mr-2" />
                    Post Job
                  </button>
                  <button
                    onClick={() => setShowPostJobModal(false)}
                    className="px-6 py-3 bg-gray-700 text-white rounded-lg hover:bg-gray-600 transition-colors"
                  >
                    Cancel
                  </button>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Employer Dashboard Modal */}
        {showEmployerDashboard && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
            <div className="bg-gray-800 rounded-2xl max-w-6xl w-full max-h-[90vh] overflow-y-auto border border-gray-700">
              <div className="sticky top-0 bg-gray-800 border-b border-gray-700 p-6 flex justify-between items-center">
                <h2 className="text-2xl font-bold text-white">Employer Dashboard</h2>
                <button onClick={() => setShowEmployerDashboard(false)} className="text-gray-400 hover:text-white">
                  <FaTimes size={24} />
                </button>
              </div>

              <div className="p-6">
                {myPostedJobs.length === 0 ? (
                  <div className="text-center py-12">
                    <FaBriefcase className="mx-auto text-6xl text-gray-600 mb-4" />
                    <p className="text-gray-400 text-lg mb-6">You haven't posted any jobs yet</p>
                    <button
                      onClick={() => {
                        setShowEmployerDashboard(false);
                        setShowPostJobModal(true);
                      }}
                      className="btn-primary bg-gradient-to-r from-turquoise to-gold"
                    >
                      <FaBriefcase className="inline mr-2" />
                      Post Your First Job
                    </button>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {myPostedJobs.map((job) => (
                      <div
                        key={job._id}
                        className="bg-gray-700/50 rounded-xl p-6 border border-gray-600 hover:border-turquoise-500/50 transition-colors"
                      >
                        <div className="flex justify-between items-start mb-4">
                          <div className="flex-1">
                            <h3 className="text-xl font-bold text-white mb-1">{job.title}</h3>
                            <p className="text-turquoise-500 font-semibold mb-2">{job.company}</p>
                            <div className="flex items-center gap-4 text-sm text-gray-400">
                              <span className="flex items-center gap-1">
                                <FaMapMarkerAlt className="text-turquoise-500" />
                                {job.location}
                              </span>
                              <span className="flex items-center gap-1">
                                <FaClock className="text-turquoise-500" />
                                {new Date(job.createdAt).toLocaleDateString()}
                              </span>
                              <span className="flex items-center gap-1">
                                <FaChartBar className="text-turquoise-500" />
                                {job.views || 0} views
                              </span>
                            </div>
                          </div>

                          <div className="flex gap-2">
                            <button
                              onClick={() => {
                                setSelectedPostedJob(job);
                                fetchJobApplications(job._id);
                              }}
                              className="px-4 py-2 bg-turquoise-600/20 text-turquoise-400 rounded-lg hover:bg-turquoise-600/30 transition-colors text-sm font-semibold border border-turquoise-500/40"
                            >
                              <FaUsers className="inline mr-2" />
                              Applications ({job.totalApplications || 0})
                              {job.newApplications > 0 && (
                                <span className="ml-2 px-2 py-0.5 bg-red-500 text-white rounded-full text-xs">
                                  {job.newApplications} new
                                </span>
                              )}
                            </button>
                            <button
                              onClick={() => handleDeleteJob(job._id)}
                              className="px-4 py-2 bg-red-600/20 text-red-400 rounded-lg hover:bg-red-600/30 transition-colors text-sm font-semibold border border-red-500/40"
                            >
                              Delete
                            </button>
                          </div>
                        </div>

                        <p className="text-gray-300 text-sm line-clamp-2">{job.description}</p>
                      </div>
                    ))}
                  </div>
                )}

                {/* Applications View */}
                {selectedPostedJob && (
                  <div className="mt-8 border-t border-gray-700 pt-8">
                    <div className="flex justify-between items-center mb-6">
                      <h3 className="text-2xl font-bold text-white">
                        Applications for {selectedPostedJob.title}
                      </h3>
                      <button
                        onClick={() => setSelectedPostedJob(null)}
                        className="text-gray-400 hover:text-white"
                      >
                        <FaTimes size={20} />
                      </button>
                    </div>

                    {jobApplications.length === 0 ? (
                      <div className="text-center py-12 bg-gray-700/30 rounded-xl">
                        <FaUsers className="mx-auto text-5xl text-gray-600 mb-4" />
                        <p className="text-gray-400">No applications yet</p>
                      </div>
                    ) : (
                      <div className="space-y-4">
                        {jobApplications.map((app) => (
                          <div
                            key={app._id}
                            className="bg-gray-700/30 rounded-xl p-6 border border-gray-600"
                          >
                            <div className="flex justify-between items-start mb-4">
                              <div>
                                <h4 className="text-lg font-bold text-white mb-1">
                                  {app.applicant?.username || 'Applicant'}
                                </h4>
                                <p className="text-gray-400 text-sm mb-2">{app.email}</p>
                                <div className="flex items-center gap-4 text-sm text-gray-400">
                                  <span>Applied {new Date(app.appliedAt).toLocaleDateString()}</span>
                                  {app.phone && <span>Phone: {app.phone}</span>}
                                </div>
                                {app.linkedIn && (
                                  <a
                                    href={app.linkedIn}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="text-turquoise-500 hover:text-turquoise-400 text-sm flex items-center gap-1 mt-2"
                                  >
                                    <FaLinkedin /> LinkedIn Profile
                                  </a>
                                )}
                                {app.portfolio && (
                                  <a
                                    href={app.portfolio}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="text-turquoise-500 hover:text-turquoise-400 text-sm flex items-center gap-1 mt-1"
                                  >
                                    <FaExternalLinkAlt /> Portfolio
                                  </a>
                                )}
                              </div>

                              <select
                                value={app.status}
                                onChange={(e) => updateApplicationStatusEmployer(app._id, e.target.value)}
                                className={`px-4 py-2 rounded-lg font-semibold text-sm border-2 focus:outline-none ${
                                  app.status === 'submitted' ? 'bg-blue-600/20 text-blue-400 border-blue-500' :
                                  app.status === 'reviewing' ? 'bg-yellow-600/20 text-yellow-400 border-yellow-500' :
                                  app.status === 'interview' ? 'bg-purple-600/20 text-purple-400 border-purple-500' :
                                  app.status === 'offer' ? 'bg-green-600/20 text-green-400 border-green-500' :
                                  'bg-red-600/20 text-red-400 border-red-500'
                                }`}
                              >
                                <option value="submitted">New</option>
                                <option value="reviewing">Reviewing</option>
                                <option value="interview">Interview</option>
                                <option value="offer">Offer</option>
                                <option value="rejected">Rejected</option>
                              </select>
                            </div>

                            {/* Cover Letter */}
                            <div className="mt-4 pt-4 border-t border-gray-600">
                              <h5 className="text-sm font-semibold text-gray-300 mb-2">Cover Letter</h5>
                              <div className="text-sm text-gray-400 bg-gray-800/50 rounded-lg p-4 max-h-48 overflow-y-auto whitespace-pre-wrap">
                                {app.coverLetter}
                              </div>
                            </div>

                            {/* Applicant Skills */}
                            {app.applicant?.skills && app.applicant.skills.length > 0 && (
                              <div className="mt-4 pt-4 border-t border-gray-600">
                                <h5 className="text-sm font-semibold text-gray-300 mb-2">Skills</h5>
                                <div className="flex flex-wrap gap-2">
                                  {app.applicant.skills.map((skill, idx) => (
                                    <span
                                      key={idx}
                                      className="px-3 py-1 bg-turquoise-600/20 text-turquoise-400 rounded-full text-xs border border-turquoise-500/40"
                                    >
                                      {skill}
                                    </span>
                                  ))}
                                </div>
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default Jobs;
