import { Link } from 'react-router-dom';
import { FaCode, FaBriefcase, FaTrophy, FaUsers, FaRocket, FaChartLine } from 'react-icons/fa';
import { motion } from 'framer-motion';
import broImg from '/src/assets/vibe-coding-bros.png';
import girlsImg from '/src/assets/vibe-coding-girls.png';
import promoVideo from '/src/assets/vibe-coding-bros-prom-2026.mp4';

const Landing = () => {
  const features = [
    {
      icon: <FaUsers size={40} />,
      title: 'Community Hub',
      description: 'Share your projects, get feedback, and collaborate with fellow developers',
      link: '/community'
    },
    {
      icon: <FaBriefcase size={40} />,
      title: 'Jobs Board',
      description: 'Find vibe coding jobs that match your skills and interests',
      link: '/jobs'
    },
    {
      icon: <FaTrophy size={40} />,
      title: 'Challenges',
      description: 'Compete in coding challenges and win prizes',
      link: '/challenges'
    },
    {
      icon: <FaRocket size={40} />,
      title: 'Summit Events',
      description: 'Join exclusive events and network with industry leaders',
      link: '/summit'
    },
    {
      icon: <FaCode size={40} />,
      title: 'Workflows',
      description: 'Share and discover productivity tips and coding workflows',
      link: '/workflows'
    },
    {
      icon: <FaChartLine size={40} />,
      title: 'Career Growth',
      description: 'Level up your skills and advance your coding career',
      link: '/dashboard'
    },
  ];

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-turquoise/10 via-vibe-black to-gold/10"></div>
        
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 text-center">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
          >
            <h1 className="text-5xl md:text-7xl font-bold mb-6">
              Welcome to <span className="text-gradient">Vibe Coding Bros©</span>
            </h1>
            <p className="text-xl md:text-2xl text-gray-300 mb-8 max-w-3xl mx-auto">
               We unify learning, building, and earning in one platform. 
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link to="/signup" className="btn-primary text-lg">
                Join the Community
              </Link>
              <Link to="/community" className="btn-outline text-lg">
                Explore Projects
              </Link>
            </div>
          </motion.div>

          {/* Animated Code Symbol */}
          <motion.div
            className="mt-16 text-turquoise text-6xl"
            animate={{ 
              scale: [1, 1.2, 1],
              rotate: [0, 360],
            }}
            transition={{ 
              duration: 10,
              repeat: Infinity,
              ease: "linear"
            }}
          >
            {'</>'}
          </motion.div>
        </div>
      </section>

      {/* Promo Video Section */}
      <section className="py-20 bg-gradient-to-b from-vibe-black to-gray-900">
        <div className="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
            viewport={{ once: true }}
            className="text-center mb-12"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">
              <span className="text-gradient">2026 Promo</span>
            </h2>
            <p className="text-xl text-gray-400">
              Watch what we're building together
            </p>
          </motion.div>

          <motion.div
            initial={{ opacity: 0, scale: 0.95 }}
            whileInView={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.8 }}
            viewport={{ once: true }}
            className="relative rounded-2xl overflow-hidden shadow-2xl border-2 border-turquoise/30"
          >
            <video
              className="w-full h-auto"
              controls
              autoPlay
              muted
              loop
              playsInline
            >
              <source src={promoVideo} type="video/mp4" />
              Your browser does not support the video tag.
            </video>
          </motion.div>
        </div>
      </section>

      {/* Vibe Coding Images Section */}
      <section className="flex flex-col md:flex-row items-center justify-center gap-8 py-12 px-4">
        <div className="max-w-md w-full">
          <img 
            src={broImg} 
            alt="Vibe Coding Bro" 
            className="w-full h-auto object-cover rounded-xl shadow-2xl border-2 border-turquoise/30 hover:border-turquoise transition-all duration-300 hover:scale-105" 
          />
        </div>
        <div className="max-w-md w-full">
          <img 
            src={girlsImg} 
            alt="Vibe Coding Girls" 
            className="w-full h-auto object-cover rounded-xl shadow-2xl border-2 border-gold/30 hover:border-gold transition-all duration-300 hover:scale-105" 
          />
        </div>
      </section>

      {/* Features Section */}
      <section className="py-20 bg-gray-900/50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            transition={{ duration: 0.8 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">
              Everything You Need to <span className="text-gradient">Level Up</span>
            </h2>
            <p className="text-xl text-gray-400">
              All the tools and resources to grow as a developer
            </p>
            <p>Learn → Build → Earn → Teach → Repeat</p>
          </motion.div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {features.map((feature, index) => (
              <Link
                key={index}
                to={feature.link}
              >
                <motion.div
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  transition={{ duration: 0.5, delay: index * 0.1 }}
                  viewport={{ once: true }}
                  className="card hover:scale-105 transition-transform duration-300 cursor-pointer h-full"
                >
                  <div className="text-turquoise mb-4">{feature.icon}</div>
                  <h3 className="text-xl font-bold mb-2">{feature.title}</h3>
                  <p className="text-gray-400">{feature.description}</p>
                </motion.div>
              </Link>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            whileInView={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.8 }}
            viewport={{ once: true }}
            className="glass-effect rounded-2xl p-12"
          >
            <h2 className="text-4xl font-bold mb-4">
              Ready to Start Your <span className="text-gradient">Vibe Journey</span>?
            </h2>
            <p className="text-xl text-gray-300 mb-8">
              Join thousands of developers who are already vibing and coding together
            </p>
            <Link to="/signup" className="btn-secondary text-lg">
              Get Started Free
            </Link>
          </motion.div>
        </div>
      </section>

      {/* Stats Section */}
      <section className="py-20 bg-gray-900/50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 text-center">
            <div>
              <div className="text-4xl font-bold text-turquoise mb-2">10K+</div>
              <div className="text-gray-400">Developers</div>
            </div>
            <div>
              <div className="text-4xl font-bold text-gold mb-2">5K+</div>
              <div className="text-gray-400">Projects Shared</div>
            </div>
            <div>
              <div className="text-4xl font-bold text-turquoise mb-2">500+</div>
              <div className="text-gray-400">Jobs Posted</div>
            </div>
            <div>
              <div className="text-4xl font-bold text-gold mb-2">100+</div>
              <div className="text-gray-400">Active Challenges</div>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
};

export default Landing;
