import React, { useState, useEffect } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import api from '../services/api';
import { useAuth } from '../context/AuthContext';
import { FaSearch, FaFilter, FaShoppingCart, FaStar, FaDownload, FaRocket, FaCrown, FaCode, FaPalette, FaChartLine, FaShieldAlt, FaBolt, FaTimes, FaCheck } from 'react-icons/fa';

const PREMIUM_UPGRADES = [
  {
    id: 'premium-templates',
    icon: <FaRocket />,
    title: 'Premium Templates Bundle',
    description: 'Access 50+ production-ready templates for React, Vue, and Next.js',
    fullDescription: 'Comprehensive collection of modern, responsive templates designed for rapid development. Each template is production-ready, fully documented, and includes best practices for performance and accessibility.',
    price: 149,
    features: ['50+ Templates', 'Lifetime Updates', 'Commercial License', 'Priority Support'],
    gradient: 'from-turquoise to-blue-500',
    included: [
      'Landing pages, dashboards, e-commerce, SaaS templates',
      'Dark mode support out of the box',
      'Mobile-first responsive design',
      'SEO optimized structure',
      'Regular monthly updates with new templates',
      'Dedicated support channel',
    ],
  },
  {
    id: 'elite-components',
    icon: <FaCrown />,
    title: 'Elite Component Library',
    description: 'Professional UI components with animations and accessibility built-in',
    fullDescription: 'A complete library of beautifully crafted, accessible components that integrate seamlessly into any React project. Built with performance and developer experience in mind.',
    price: 199,
    features: ['100+ Components', 'Dark Mode', 'TypeScript', 'Storybook Docs'],
    gradient: 'from-purple to-pink-500',
    included: [
      'Forms, tables, charts, modals, navigation components',
      'Full TypeScript support with type definitions',
      'Comprehensive Storybook documentation',
      'WCAG 2.1 AA compliant',
      'Smooth animations with Framer Motion',
      'Copy-paste ready code snippets',
    ],
  },
  {
    id: 'fullstack-starter',
    icon: <FaCode />,
    title: 'Full-Stack Starter Kit',
    description: 'Complete auth, payments, and admin dashboard boilerplate',
    fullDescription: 'Skip months of setup and get straight to building your product. This comprehensive starter kit includes everything you need for a modern SaaS application.',
    price: 299,
    features: ['Auth System', 'Stripe Integration', 'Admin Panel', '6 Months Support'],
    gradient: 'from-gold to-orange-500',
    included: [
      'Email/password + OAuth authentication',
      'Stripe subscription & payment processing',
      'Role-based access control',
      'Admin dashboard with analytics',
      'Email notifications system',
      '6 months of dedicated support',
    ],
  },
  {
    id: 'designer-pro',
    icon: <FaPalette />,
    title: 'Designer Pro Pack',
    description: 'Premium themes, Figma files, and design system templates',
    fullDescription: 'Everything a designer needs to create stunning, consistent interfaces. Includes design tokens, component libraries, and production-ready assets.',
    price: 179,
    features: ['10 Themes', 'Figma Files', 'Icon Library', 'Brand Guidelines'],
    gradient: 'from-pink-500 to-rose-500',
    included: [
      '10 complete UI themes with variants',
      'Editable Figma design files',
      '500+ custom icon set',
      'Design system documentation',
      'Brand guideline templates',
      'Color palette generators',
    ],
  },
  {
    id: 'analytics-suite',
    icon: <FaChartLine />,
    title: 'Analytics Dashboard Suite',
    description: 'Advanced data visualization and reporting dashboards',
    fullDescription: 'Professional analytics dashboards that turn raw data into actionable insights. Perfect for SaaS apps, admin panels, and data-driven applications.',
    price: 249,
    features: ['Chart Components', 'Real-time Data', 'Export Options', 'Custom Widgets'],
    gradient: 'from-blue-500 to-cyan-500',
    included: [
      '20+ customizable chart types',
      'Real-time data streaming support',
      'CSV/PDF export functionality',
      'Drag-and-drop dashboard builder',
      'Custom widget creation tools',
      'Integration with popular analytics services',
    ],
  },
  {
    id: 'security-auth',
    icon: <FaShieldAlt />,
    title: 'Security & Auth Bundle',
    description: 'Enterprise-grade authentication and security implementations',
    fullDescription: 'Robust security solutions for modern applications. Implement industry-standard authentication and protect your users with enterprise-grade features.',
    price: 199,
    features: ['OAuth/SSO', '2FA', 'Role-Based Access', 'Audit Logging'],
    gradient: 'from-green-500 to-emerald-500',
    included: [
      'OAuth 2.0 & SSO integration',
      'Two-factor authentication (2FA)',
      'Granular role-based permissions',
      'Complete audit logging system',
      'Session management & security',
      'OWASP security best practices',
    ],
  },
  {
    id: 'performance-kit',
    icon: <FaBolt />,
    title: 'Performance Optimization Kit',
    description: 'Tools and configs to supercharge your app performance',
    fullDescription: 'Optimize your application for blazing-fast performance. Includes configurations, tools, and best practices to achieve top Lighthouse scores.',
    price: 129,
    features: ['Code Splitting', 'Image Optimization', 'Caching Strategies', 'CDN Setup'],
    gradient: 'from-yellow-500 to-orange-500',
    included: [
      'Webpack/Vite optimization configs',
      'Automatic image compression pipeline',
      'Service worker & caching setup',
      'CDN deployment guides',
      'Bundle analyzer tools',
      'Performance monitoring setup',
    ],
  },
];

const Marketplace = () => {
  const navigate = useNavigate();
  const { isAuthenticated, user } = useAuth();
  
  const [products, setProducts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedUpgrade, setSelectedUpgrade] = useState(null);
  const [purchasing, setPurchasing] = useState(false);
  const [filters, setFilters] = useState({
    category: 'all',
    minPrice: '',
    maxPrice: '',
    search: '',
    sort: '-createdAt',
  });
  const [pagination, setPagination] = useState({});
  const [showFilters, setShowFilters] = useState(false);

  useEffect(() => {
    fetchProducts();
  }, [filters]);

  const fetchProducts = async (page = 1) => {
    try {
      setLoading(true);
      const params = new URLSearchParams({
        ...filters,
        page,
      });
      
      const response = await api.get(`/products?${params}`);
      console.log('Fetched products from API:', response.data.products);
      setProducts(response.data.products);
      setPagination(response.data.pagination);
    } catch (error) {
      console.error('Failed to fetch products:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleFilterChange = (key, value) => {
    setFilters(prev => ({ ...prev, [key]: value }));
  };

  const handlePurchaseUpgrade = async (upgrade) => {
    if (!isAuthenticated) {
      navigate('/signin');
      return;
    }

    setPurchasing(true);
    try {
      const response = await api.post('/stripe/create-premium-upgrade-checkout', {
        productId: upgrade.id,
        name: upgrade.title,
        price: upgrade.price,
        description: upgrade.description,
      });

      if (response.data?.url) {
        window.location.href = response.data.url;
      }
    } catch (error) {
      console.error('Purchase error:', error);
      alert(error.response?.data?.message || 'Failed to start checkout');
    } finally {
      setPurchasing(false);
    }
  };

  const getDiscountedPrice = (price) => {
    if (user?.subscriptionTier === 'creator') {
      return (price * 0.8).toFixed(2); // 20% off
    }
    return price;
  };

  const categories = [
    { value: 'all', label: 'All Categories' },
    { value: 'template', label: 'Templates' },
    { value: 'component', label: 'Components' },
    { value: 'snippet', label: 'Code Snippets' },
    { value: 'plugin', label: 'Plugins' },
    { value: 'theme', label: 'Themes' },
    { value: 'other', label: 'Other' },
  ];

  const sortOptions = [
    { value: '-createdAt', label: 'Newest First' },
    { value: 'createdAt', label: 'Oldest First' },
    { value: '-rating', label: 'Highest Rated' },
    { value: '-downloads', label: 'Most Downloaded' },
    { value: 'price', label: 'Price: Low to High' },
    { value: '-price', label: 'Price: High to Low' },
  ];

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-5xl font-bold mb-4">
            <span className="text-gradient">Marketplace</span>
          </h1>
          <p className="text-xl text-gray-400 max-w-2xl mx-auto">
            Discover premium templates, components, and code snippets from talented creators
          </p>
        </div>

        {/* Premium Upgrades Section */}
        <div className="mb-16">
          <div className="flex items-center justify-between mb-8">
            <div>
              <h2 className="text-3xl font-bold mb-2">
                <span className="text-gradient">Premium Upgrades</span>
              </h2>
              <p className="text-gray-400">High-level professional packages to supercharge your development</p>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mb-12">
            {PREMIUM_UPGRADES.map((upgrade) => (
              <div
                key={upgrade.id}
                className="card relative overflow-hidden hover:scale-105 transition-transform group"
              >
                <div className={`absolute inset-0 bg-gradient-to-br ${upgrade.gradient} opacity-5 group-hover:opacity-10 transition-opacity`}></div>
                
                <div className="relative p-6">
                  <div className={`text-4xl mb-4 bg-gradient-to-r ${upgrade.gradient} bg-clip-text text-transparent`}>
                    {upgrade.icon}
                  </div>
                  
                  <h3 className="text-xl font-bold mb-2">{upgrade.title}</h3>
                  <p className="text-gray-400 text-sm mb-4 min-h-[40px]">{upgrade.description}</p>
                  
                  <div className="mb-4">
                    <div className="text-3xl font-bold text-turquoise mb-3">
                      ${upgrade.price}
                    </div>
                    
                    <ul className="space-y-2 mb-4">
                      {upgrade.features.map((feature, idx) => (
                        <li key={idx} className="text-sm text-gray-300 flex items-center gap-2">
                          <span className="text-turquoise">✓</span>
                          {feature}
                        </li>
                      ))}
                    </ul>
                  </div>
                  
                  <button className={`w-full btn-primary bg-gradient-to-r ${upgrade.gradient} hover:opacity-90`}
                    onClick={() => setSelectedUpgrade(upgrade)}
                  >
                    View Details
                  </button>
                </div>
              </div>
            ))}
          </div>

          <div className="text-center mb-16">
            <div className="inline-block p-1 rounded-xl bg-gradient-to-r from-turquoise via-purple to-gold">
              <div className="bg-vibe-black px-8 py-4 rounded-lg">
                <p className="text-lg">
                  <span className="font-bold text-turquoise">Creator Tier Members</span> get{' '}
                  <span className="font-bold text-gold">20% off</span> all premium upgrades
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Divider */}
        <div className="border-t border-gray-800 mb-12"></div>

        {/* Community Products Section */}
        <div className="mb-8">
          <h2 className="text-2xl font-bold mb-6">Community Products</h2>
        </div>

        {/* Search and Filters Bar */}
        <div className="mb-8 space-y-4">
          {/* Search */}
          <div className="flex gap-4">
            <div className="flex-1 relative">
              <FaSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-500" />
              <input
                type="text"
                placeholder="Search products..."
                value={filters.search}
                onChange={(e) => handleFilterChange('search', e.target.value)}
                className="input-field pl-12 w-full"
              />
            </div>
            <button
              onClick={() => setShowFilters(!showFilters)}
              className="btn-secondary flex items-center gap-2"
            >
              <FaFilter />
              Filters
            </button>
            <Link to="/cart" className="btn-secondary flex items-center gap-2">
              <FaShoppingCart />
              Cart
            </Link>
          </div>

          {/* Filters Panel */}
          {showFilters && (
            <div className="card p-6">
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-2">Category</label>
                  <select
                    value={filters.category}
                    onChange={(e) => handleFilterChange('category', e.target.value)}
                    className="input-field"
                  >
                    {categories.map(cat => (
                      <option key={cat.value} value={cat.value}>{cat.label}</option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Min Price ($)</label>
                  <input
                    type="number"
                    placeholder="0"
                    value={filters.minPrice}
                    onChange={(e) => handleFilterChange('minPrice', e.target.value)}
                    className="input-field"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Max Price ($)</label>
                  <input
                    type="number"
                    placeholder="Any"
                    value={filters.maxPrice}
                    onChange={(e) => handleFilterChange('maxPrice', e.target.value)}
                    className="input-field"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">Sort By</label>
                  <select
                    value={filters.sort}
                    onChange={(e) => handleFilterChange('sort', e.target.value)}
                    className="input-field"
                  >
                    {sortOptions.map(opt => (
                      <option key={opt.value} value={opt.value}>{opt.label}</option>
                    ))}
                  </select>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Products Grid */}
        {loading ? (
          <div className="flex justify-center py-20">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
          </div>
        ) : products.length === 0 ? (
          <div className="text-center py-20">
            <p className="text-gray-400 text-lg">No products found</p>
          </div>
        ) : (
          <>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {products.map(product => (
                <Link
                  key={product._id}
                  to={`/marketplace/${product._id}`}
                  className="card hover:scale-105 transition-transform"
                >
                  {/* Preview Image */}
                  {product.previewImages && product.previewImages[0] ? (
                    <img
                      src={product.previewImages[0]}
                      alt={product.title}
                      className="w-full h-48 object-cover rounded-t-lg"
                    />
                  ) : (
                    <div className="w-full h-48 bg-gradient-to-br from-turquoise/20 to-purple/20 rounded-t-lg flex items-center justify-center">
                      <span className="text-4xl">📦</span>
                    </div>
                  )}

                  <div className="p-4">
                    {/* Category Badge */}
                    <span className="inline-block px-2 py-1 text-xs rounded bg-turquoise/20 text-turquoise mb-2">
                      {product.category}
                    </span>

                    <h3 className="font-bold text-lg mb-2 line-clamp-2">{product.title}</h3>
                    <p className="text-gray-400 text-sm mb-4 line-clamp-2">
                      {product.shortDescription}
                    </p>

                    {/* Creator */}
                    <div className="flex items-center gap-2 mb-3">
                      <img
                        src={product.creator?.avatar || '/default-avatar.png'}
                        alt={product.creator?.username}
                        className="w-6 h-6 rounded-full"
                      />
                      <span className="text-sm text-gray-400">{product.creator?.username}</span>
                    </div>

                    {/* Rating & Stats */}
                    <div className="flex items-center justify-between text-sm mb-4">
                      <div className="flex items-center gap-1">
                        <FaStar className="text-yellow-500" />
                        <span>{product.rating.toFixed(1)}</span>
                        <span className="text-gray-500">({product.reviewCount})</span>
                      </div>
                      <div className="flex items-center gap-1 text-gray-400">
                        <FaDownload />
                        <span>{product.downloads}</span>
                      </div>
                    </div>

                    {/* Price */}
                    <div className="flex items-center justify-between">
                      <span className="text-2xl font-bold text-turquoise">
                        ${product.price.toFixed(2)}
                      </span>
                      <button className="btn-primary py-1 px-4 text-sm">
                        View Details
                      </button>
                    </div>
                  </div>
                </Link>
              ))}
            </div>

            {/* Pagination */}
            {pagination.pages > 1 && (
              <div className="flex justify-center gap-2 mt-8">
                <button
                  disabled={pagination.page === 1}
                  onClick={() => fetchProducts(pagination.page - 1)}
                  className="btn-secondary disabled:opacity-50"
                >
                  Previous
                </button>
                <span className="px-4 py-2">
                  Page {pagination.page} of {pagination.pages}
                </span>
                <button
                  disabled={pagination.page === pagination.pages}
                  onClick={() => fetchProducts(pagination.page + 1)}
                  className="btn-secondary disabled:opacity-50"
                >
                  Next
                </button>
              </div>
            )}
          </>
        )}

        {/* Premium Upgrade Detail Modal */}
        {selectedUpgrade && (
          <div className="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4"
            onClick={() => setSelectedUpgrade(null)}
          >
            <div className="bg-gray-900 rounded-2xl max-w-3xl w-full max-h-[90vh] overflow-y-auto border border-gray-800"
              onClick={(e) => e.stopPropagation()}
            >
              <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between">
                <div className="flex items-center gap-4">
                  <div className={`text-4xl bg-gradient-to-r ${selectedUpgrade.gradient} bg-clip-text text-transparent`}>
                    {selectedUpgrade.icon}
                  </div>
                  <div>
                    <h2 className="text-2xl font-bold">{selectedUpgrade.title}</h2>
                    <p className="text-gray-400">{selectedUpgrade.description}</p>
                  </div>
                </div>
                <button
                  onClick={() => setSelectedUpgrade(null)}
                  className="text-gray-400 hover:text-white transition-colors"
                >
                  <FaTimes size={24} />
                </button>
              </div>

              <div className="p-6">
                <div className="mb-6">
                  <h3 className="text-xl font-bold mb-3">Full Description</h3>
                  <p className="text-gray-300">{selectedUpgrade.fullDescription}</p>
                </div>

                <div className="mb-6">
                  <h3 className="text-xl font-bold mb-3">What's Included</h3>
                  <ul className="space-y-3">
                    {selectedUpgrade.included.map((item, idx) => (
                      <li key={idx} className="flex items-start gap-3">
                        <FaCheck className="text-turquoise mt-1 flex-shrink-0" />
                        <span className="text-gray-300">{item}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div className="border-t border-gray-800 pt-6">
                  <div className="flex items-end justify-between mb-6">
                    <div>
                      <div className="text-sm text-gray-400 mb-1">One-time payment</div>
                      {user?.subscriptionTier === 'creator' ? (
                        <div>
                          <div className="text-2xl text-gray-500 line-through">${selectedUpgrade.price}</div>
                          <div className="text-4xl font-bold text-turquoise">
                            ${getDiscountedPrice(selectedUpgrade.price)}
                            <span className="text-sm text-gold ml-2">20% Creator Discount</span>
                          </div>
                        </div>
                      ) : (
                        <div className="text-4xl font-bold text-turquoise">${selectedUpgrade.price}</div>
                      )}
                    </div>
                    
                    <button
                      onClick={() => handlePurchaseUpgrade(selectedUpgrade)}
                      disabled={purchasing}
                      className={`px-8 py-4 rounded-xl font-bold text-lg bg-gradient-to-r ${selectedUpgrade.gradient} hover:opacity-90 transition-opacity disabled:opacity-50`}
                    >
                      {purchasing ? 'Processing...' : 'Purchase Now'}
                    </button>
                  </div>

                  {!isAuthenticated && (
                    <div className="bg-yellow-500/10 border border-yellow-500/20 rounded-lg p-4 text-sm text-yellow-300">
                      Please sign in to purchase this upgrade
                    </div>
                  )}

                  <div className="text-xs text-gray-500 mt-4">
                    Secure payment processing via Stripe. Instant access after purchase. All purchases include lifetime updates.
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default Marketplace;
