import React, { useState, useEffect, useRef } from 'react';
import { useLocation } from 'react-router-dom';
import { useSocket } from '../context/SocketContext';
import { FiSend, FiSearch, FiMoreVertical, FiPlus } from 'react-icons/fi';
import { FaCircle } from 'react-icons/fa';
import api from '../services/api';

const Messages = () => {
  const { socket, connected } = useSocket();
  const location = useLocation();
  const [conversations, setConversations] = useState([]);
  const [selectedConversation, setSelectedConversation] = useState(null);
  const [messages, setMessages] = useState([]);
  const [newMessage, setNewMessage] = useState('');
  const [loading, setLoading] = useState(true);
  const [typing, setTyping] = useState(null);
  const [showUserList, setShowUserList] = useState(false);
  const [users, setUsers] = useState([]);
  const [searchUsers, setSearchUsers] = useState('');
  const messagesEndRef = useRef(null);
  const typingTimeoutRef = useRef(null);

  useEffect(() => {
    fetchConversations();
    
    // Check if navigated with a conversation
    if (location.state?.conversation) {
      selectConversation(location.state.conversation);
    }
  }, [location.state]);

  useEffect(() => {
    if (!socket) return;

    // Listen for new messages
    socket.on('message:new', ({ conversationId, message }) => {
      if (selectedConversation?._id === conversationId) {
        setMessages((prev) => [...prev, message]);
        scrollToBottom();
        
        // Mark as read
        socket.emit('message:read', { conversationId });
      }
      
      // Update conversation list
      fetchConversations();
    });

    // Listen for typing indicators
    socket.on('typing:start', ({ conversationId, username }) => {
      if (selectedConversation?._id === conversationId) {
        setTyping(username);
      }
    });

    socket.on('typing:stop', ({ conversationId }) => {
      if (selectedConversation?._id === conversationId) {
        setTyping(null);
      }
    });

    // Listen for read receipts
    socket.on('message:read', ({ conversationId }) => {
      if (selectedConversation?._id === conversationId) {
        fetchMessages(conversationId);
      }
    });

    return () => {
      socket.off('message:new');
      socket.off('typing:start');
      socket.off('typing:stop');
      socket.off('message:read');
    };
  }, [socket, selectedConversation]);

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const fetchUsers = async () => {
    try {
      const response = await api.get('/users');
      setUsers(response.data.data || []);
    } catch (error) {
      console.error('Error fetching users:', error);
    }
  };

  const startConversation = async (userId) => {
    try {
      const response = await api.get(`/messages/conversations/${userId}`);
      setShowUserList(false);
      selectConversation(response.data.data);
      fetchConversations();
    } catch (error) {
      console.error('Error starting conversation:', error);
    }
  };

  const fetchConversations = async () => {
    try {
      const response = await api.get('/messages/conversations');
      setConversations(response.data.data);
    } catch (error) {
      console.error('Error fetching conversations:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchMessages = async (conversationId) => {
    try {
      const response = await api.get(`/messages/${conversationId}`);
      setMessages(response.data.data);
      
      // Mark messages as read
      if (socket) {
        socket.emit('message:read', { conversationId });
      }
    } catch (error) {
      console.error('Error fetching messages:', error);
    }
  };

  const selectConversation = (conversation) => {
    setSelectedConversation(conversation);
    fetchMessages(conversation._id);
    
    // Join conversation room
    if (socket) {
      if (selectedConversation) {
        socket.emit('conversation:leave', selectedConversation._id);
      }
      socket.emit('conversation:join', conversation._id);
    }
  };

  const handleSendMessage = (e) => {
    e.preventDefault();
    if (!newMessage.trim() || !socket || !selectedConversation) return;

    // Emit message through socket
    socket.emit('message:send', {
      conversationId: selectedConversation._id,
      content: newMessage.trim(),
    });

    setNewMessage('');
    
    // Stop typing indicator
    socket.emit('typing:stop', { conversationId: selectedConversation._id });
  };

  const handleTyping = (e) => {
    setNewMessage(e.target.value);
    
    if (!socket || !selectedConversation) return;

    // Clear previous timeout
    if (typingTimeoutRef.current) {
      clearTimeout(typingTimeoutRef.current);
    }

    // Start typing indicator
    if (e.target.value) {
      socket.emit('typing:start', {
        conversationId: selectedConversation._id,
        username: 'You',
      });

      // Stop typing after 2 seconds of inactivity
      typingTimeoutRef.current = setTimeout(() => {
        socket.emit('typing:stop', { conversationId: selectedConversation._id });
      }, 2000);
    } else {
      socket.emit('typing:stop', { conversationId: selectedConversation._id });
    }
  };

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  const getOtherParticipant = (conversation) => {
    return conversation.participants.find(
      (p) => p._id !== localStorage.getItem('userId')
    );
  };

  const formatTime = (date) => {
    return new Date(date).toLocaleTimeString('en-US', {
      hour: 'numeric',
      minute: '2-digit',
    });
  };

  const filteredUsers = users.filter(user => 
    user._id !== localStorage.getItem('userId') &&
    user.username.toLowerCase().includes(searchUsers.toLowerCase())
  );

  return (
    <div className="min-h-screen bg-dark-bg text-white pt-20 pb-8">
      <div className="max-w-7xl mx-auto px-4">
        <div className="bg-gray-800 rounded-lg overflow-hidden" style={{ height: 'calc(100vh - 160px)' }}>
          <div className="flex h-full">
            {/* Conversations List */}
            <div className="w-1/3 border-r border-gray-700 flex flex-col">
              {/* Header */}
              <div className="p-4 border-b border-gray-700">
                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-2xl font-bold">Messages</h2>
                  <button
                    onClick={() => {
                      fetchUsers();
                      setShowUserList(true);
                    }}
                    className="bg-light-sea-green text-white p-2 rounded-lg hover:bg-opacity-90 transition-colors"
                    title="New Message"
                  >
                    <FiPlus size={20} />
                  </button>
                </div>
                <div className="relative">
                  <FiSearch className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                  <input
                    type="text"
                    placeholder="Search conversations..."
                    className="w-full bg-gray-700 text-white pl-10 pr-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                  />
                </div>
              </div>

              {/* Conversations */}
              <div className="flex-1 overflow-y-auto">
                {loading ? (
                  <div className="flex items-center justify-center h-full">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-light-sea-green"></div>
                  </div>
                ) : conversations.length === 0 ? (
                  <div className="flex flex-col items-center justify-center h-full text-gray-400">
                    <p>No conversations yet</p>
                    <p className="text-sm mt-2">Start chatting with other users!</p>
                  </div>
                ) : (
                  conversations.map((conversation) => {
                    const otherUser = getOtherParticipant(conversation);
                    return (
                      <div
                        key={conversation._id}
                        onClick={() => selectConversation(conversation)}
                        className={`p-4 border-b border-gray-700 cursor-pointer hover:bg-gray-750 transition-colors ${
                          selectedConversation?._id === conversation._id ? 'bg-gray-750' : ''
                        }`}
                      >
                        <div className="flex items-center gap-3">
                          <div className="relative">
                            <div className="w-12 h-12 rounded-full bg-light-sea-green flex items-center justify-center text-lg font-bold">
                              {otherUser?.username?.charAt(0).toUpperCase()}
                            </div>
                            {connected && (
                              <FaCircle className="absolute bottom-0 right-0 text-green-500 text-xs" />
                            )}
                          </div>
                          <div className="flex-1 min-w-0">
                            <div className="flex items-center justify-between">
                              <h3 className="font-semibold truncate">{otherUser?.username}</h3>
                              {conversation.lastMessage && (
                                <span className="text-xs text-gray-400">
                                  {formatTime(conversation.lastMessageAt)}
                                </span>
                              )}
                            </div>
                            {conversation.lastMessage && (
                              <p className="text-sm text-gray-400 truncate">
                                {conversation.lastMessage.content}
                              </p>
                            )}
                          </div>
                        </div>
                      </div>
                    );
                  })
                )}
              </div>
            </div>

            {/* Chat Area */}
            <div className="flex-1 flex flex-col">
              {selectedConversation ? (
                <>
                  {/* Chat Header */}
                  <div className="p-4 border-b border-gray-700 flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 rounded-full bg-light-sea-green flex items-center justify-center font-bold">
                        {getOtherParticipant(selectedConversation)?.username?.charAt(0).toUpperCase()}
                      </div>
                      <div>
                        <h3 className="font-semibold">
                          {getOtherParticipant(selectedConversation)?.username}
                        </h3>
                        {typing && (
                          <p className="text-xs text-light-sea-green">typing...</p>
                        )}
                      </div>
                    </div>
                    <button className="text-gray-400 hover:text-white">
                      <FiMoreVertical size={20} />
                    </button>
                  </div>

                  {/* Messages */}
                  <div className="flex-1 overflow-y-auto p-4 space-y-4">
                    {messages.map((message) => {
                      const isOwn = message.sender._id === localStorage.getItem('userId');
                      return (
                        <div
                          key={message._id}
                          className={`flex ${isOwn ? 'justify-end' : 'justify-start'}`}
                        >
                          <div
                            className={`max-w-[70%] rounded-lg p-3 ${
                              isOwn
                                ? 'bg-light-sea-green text-white'
                                : 'bg-gray-700 text-white'
                            }`}
                          >
                            <p className="break-words">{message.content}</p>
                            <p className="text-xs mt-1 opacity-75">
                              {formatTime(message.createdAt)}
                            </p>
                          </div>
                        </div>
                      );
                    })}
                    <div ref={messagesEndRef} />
                  </div>

                  {/* Message Input */}
                  <form onSubmit={handleSendMessage} className="p-4 border-t border-gray-700">
                    <div className="flex items-center gap-2">
                      <input
                        type="text"
                        value={newMessage}
                        onChange={handleTyping}
                        placeholder="Type a message..."
                        className="flex-1 bg-gray-700 text-white px-4 py-3 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                      />
                      <button
                        type="submit"
                        disabled={!newMessage.trim()}
                        className="bg-light-sea-green text-white p-3 rounded-lg hover:bg-opacity-90 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        <FiSend size={20} />
                      </button>
                    </div>
                  </form>
                </>
              ) : (
                <div className="flex-1 flex items-center justify-center text-gray-400">
                  <div className="text-center">
                    <p className="text-xl mb-2">Select a conversation</p>
                    <p className="text-sm">Choose a conversation from the list to start chatting</p>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* New Conversation Modal */}
      {showUserList && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50" onClick={() => setShowUserList(false)}>
          <div className="bg-gray-800 rounded-lg p-6 max-w-md w-full mx-4" onClick={(e) => e.stopPropagation()}>
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold">New Message</h3>
              <button onClick={() => setShowUserList(false)} className="text-gray-400 hover:text-white">
                ✕
              </button>
            </div>
            
            <div className="mb-4">
              <input
                type="text"
                value={searchUsers}
                onChange={(e) => setSearchUsers(e.target.value)}
                placeholder="Search users..."
                className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
              />
            </div>

            <div className="max-h-96 overflow-y-auto">
              {filteredUsers.length === 0 ? (
                <p className="text-center text-gray-400 py-8">No users found</p>
              ) : (
                filteredUsers.map((user) => (
                  <div
                    key={user._id}
                    onClick={() => startConversation(user._id)}
                    className="flex items-center gap-3 p-3 hover:bg-gray-700 rounded-lg cursor-pointer transition-colors"
                  >
                    <div className="w-10 h-10 rounded-full bg-light-sea-green flex items-center justify-center font-bold">
                      {user.username.charAt(0).toUpperCase()}
                    </div>
                    <div className="flex-1">
                      <h4 className="font-semibold">{user.username}</h4>
                      <p className="text-sm text-gray-400">{user.email}</p>
                    </div>
                  </div>
                ))
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Messages;
