import { useEffect, useState } from 'react';
import { useParams, Link } from 'react-router-dom';

const ModelDetail = () => {
  const { modelId } = useParams();
  const [model, setModel] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const load = async () => {
      try {
        setLoading(true);
        const res = await fetch(`/api/hf/models/${encodeURIComponent(modelId)}`);
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data = await res.json();
        setModel(data);
      } catch (err) {
        setError(err.message || 'Failed to load model');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, [modelId]);

  if (loading) return <div className="min-h-screen pt-20 flex items-center justify-center">Loading…</div>;
  if (error) return <div className="min-h-screen pt-20 text-red-400">Error: {error}</div>;
  if (!model) return <div className="min-h-screen pt-20">No model found</div>;

  return (
    <div className="min-h-screen bg-slate-900 text-gray-100 pt-20">
      <div className="max-w-4xl mx-auto px-6 sm:px-8 lg:px-12 py-12">
        <div className="mb-6">
          <h1 className="text-3xl font-bold mb-2">{model.name}</h1>
          <p className="text-sm text-gray-400">{model.author} • {model.license || 'No license listed'}</p>
        </div>

        <div className="bg-slate-800 rounded-lg p-6">
          <div className="prose prose-invert max-w-none" dangerouslySetInnerHTML={{ __html: model.readmeHtml || '<p>No README available.</p>' }} />
        </div>

        <div className="mt-6">
          <Link to="/models" className="text-blue-400">← Back to models</Link>
        </div>
      </div>
    </div>
  );
};

export default ModelDetail;
