import { useEffect, useState } from 'react';
import { Link } from 'react-router-dom';
import { FaSearch, FaFilter, FaTimes, FaCheckSquare, FaSquare, FaChartBar, FaPlay, FaRocket, FaFire, FaStar, FaCrown, FaDollarSign, FaCode, FaImage, FaMicrophone, FaVideo, FaLightbulb, FaTh, FaList } from 'react-icons/fa';
import ModelCardsGrid from '../components/ModelCardsGrid';

function useServerModelCards() {
  const [models, setModels] = useState([]);
  useEffect(() => {
    let mounted = true;
    (async () => {
      try {
        const res = await fetch('/api/models/cards');
        if (!res.ok) throw new Error('Failed to fetch model cards');
        const data = await res.json();
        if (mounted) setModels(data || []);
      } catch (e) {
        console.warn('Failed to load /api/models/cards, falling back to Hugging Face list', e);
      }
    })();
    return () => { mounted = false; };
  }, []);
  return models;
}

const Models = () => {
  const premiumModels = useServerModelCards();
  const [models, setModels] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterCreator, setFilterCreator] = useState('all');
  
  // New state for advanced features
  const [showAdvancedFilters, setShowAdvancedFilters] = useState(false);
  const [selectedCapabilities, setSelectedCapabilities] = useState([]);
  const [paramSizeRange, setParamSizeRange] = useState([0, 200]);
  const [contextRange, setContextRange] = useState([0, 200]);
  const [selectedTasks, setSelectedTasks] = useState([]);
  const [pricingFilter, setPricingFilter] = useState('all');
  const [selectedModels, setSelectedModels] = useState([]);
  const [showComparison, setShowComparison] = useState(false);
  const [activeCollection, setActiveCollection] = useState('all');
  const [viewMode, setViewMode] = useState('grid');
  const [selectedModel, setSelectedModel] = useState(null);
  const [showPlayground, setShowPlayground] = useState(false);
  const [playgroundPrompt, setPlaygroundPrompt] = useState('');
  const [playgroundResult, setPlaygroundResult] = useState('');
  const [sortBy, setSortBy] = useState('popular');

  useEffect(() => {
    const load = async () => {
        try {
        setLoading(true);
        const res = await fetch('https://huggingface.co/api/models?limit=50');
        if (!res.ok) throw new Error(`Failed to load (${res.status})`);
        const raw = await res.json();
        const data = raw.map(m => ({
          id: m.modelId || m.id,
          tags: m.tags || [],
          pipeline: m.pipeline_tag || (m.cardData && m.cardData.pipeline_tag) || null,
          lastModified: m.lastModified || m.last_updated || null,
        }));
        setModels(data || []);
      } catch (err) {
        setError(err.message || 'Failed to fetch models');
      } finally {
        setLoading(false);
      }
    };

    load();
  }, []);

  const capabilities = ['tools', 'vision', 'reasoning', 'humor'];
  const tasks = ['Text Generation', 'Code', 'Image', 'Audio', 'Multimodal'];
  const pricingTiers = ['all', 'free', 'api-only', 'self-hostable'];
  
  const collections = [
    { id: 'all', label: 'All Models', icon: <FaTh /> },
    { id: 'trending', label: 'Trending', icon: <FaFire /> },
    { id: 'popular', label: 'Most Popular', icon: <FaStar /> },
    { id: 'new', label: 'New Releases', icon: <FaRocket /> },
    { id: 'code', label: 'Best for Code', icon: <FaCode /> },
    { id: 'enterprise', label: 'Enterprise Ready', icon: <FaCrown /> },
  ];

  const samplePrompts = [
    'Write a Python function to calculate fibonacci numbers',
    'Explain quantum computing in simple terms',
    'Create a React component for a todo list',
    'Summarize the key points of machine learning',
  ];

  const toggleCapability = (cap) => {
    setSelectedCapabilities(prev =>
      prev.includes(cap) ? prev.filter(c => c !== cap) : [...prev, cap]
    );
  };

  const toggleTask = (task) => {
    setSelectedTasks(prev =>
      prev.includes(task) ? prev.filter(t => t !== task) : [...prev, task]
    );
  };

  const toggleModelSelection = (model) => {
    setSelectedModels(prev => {
      const exists = prev.find(m => m.model === model.model);
      if (exists) {
        return prev.filter(m => m.model !== model.model);
      }
      if (prev.length >= 4) {
        alert('Maximum 4 models can be compared');
        return prev;
      }
      return [...prev, model];
    });
  };

  const clearFilters = () => {
    setSearchTerm('');
    setFilterCreator('all');
    setSelectedCapabilities([]);
    setParamSizeRange([0, 200]);
    setContextRange([0, 200]);
    setSelectedTasks([]);
    setPricingFilter('all');
  };

  const applyFilters = (modelsList) => {
    return modelsList.filter(model => {
      // Search filter
      const matchesSearch = model.model.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           model.creator.toLowerCase().includes(searchTerm.toLowerCase());
      
      // Creator filter
      const matchesCreator = filterCreator === 'all' || model.creator === filterCreator;
      
      // Capabilities filter
      const matchesCapabilities = selectedCapabilities.length === 0 || 
        selectedCapabilities.some(cap => model.capabilities?.includes(cap));
      
      // Collection filter
      const matchesCollection = activeCollection === 'all' || 
        (activeCollection === 'trending' && Math.random() > 0.6) ||
        (activeCollection === 'popular' && (model.rating || 0) >= 4) ||
        (activeCollection === 'code' && model.capabilities?.includes('tools'));

      return matchesSearch && matchesCreator && matchesCapabilities && matchesCollection;
    });
  };

  const sortModels = (modelsList) => {
    const sorted = [...modelsList];
    switch(sortBy) {
      case 'popular':
        return sorted.sort((a, b) => (b.rating || 0) - (a.rating || 0));
      case 'name':
        return sorted.sort((a, b) => a.model.localeCompare(b.model));
      case 'params':
        return sorted.sort((a, b) => {
          const aSize = parseInt(a.paramSize) || 0;
          const bSize = parseInt(b.paramSize) || 0;
          return bSize - aSize;
        });
      default:
        return sorted;
    }
  };

  const filteredPremiumModels = sortModels(applyFilters(premiumModels));

  const creators = ['all', ...new Set(premiumModels.map(m => m.creator))];

  const activeFiltersCount = 
    (searchTerm ? 1 : 0) +
    (filterCreator !== 'all' ? 1 : 0) +
    selectedCapabilities.length +
    selectedTasks.length +
    (pricingFilter !== 'all' ? 1 : 0);

  return (
    <div className="min-h-screen bg-slate-900 text-gray-100 pt-20">
      <div className="max-w-7xl mx-auto px-6 sm:px-8 lg:px-12 py-12">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-4xl font-bold mb-2">
              AI <span className="text-gradient">Models</span>
            </h1>
            <p className="text-gray-400">
              Explore {filteredPremiumModels.length} premium models with benchmarks, testing & comparisons
            </p>
          </div>
          <div className="flex gap-2">
            <button
              onClick={() => setViewMode('grid')}
              className={`p-3 rounded-lg transition-colors ${
                viewMode === 'grid' ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400'
              }`}
            >
              <FaTh />
            </button>
            <button
              onClick={() => setViewMode('list')}
              className={`p-3 rounded-lg transition-colors ${
                viewMode === 'list' ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400'
              }`}
            >
              <FaList />
            </button>
          </div>
        </div>

        {/* Collections Bar */}
        <div className="mb-6 flex gap-3 overflow-x-auto pb-2">
          {collections.map((col) => (
            <button
              key={col.id}
              onClick={() => setActiveCollection(col.id)}
              className={`flex items-center gap-2 px-4 py-2 rounded-lg whitespace-nowrap transition-all ${
                activeCollection === col.id
                  ? 'bg-gradient-to-r from-turquoise to-gold text-black font-semibold'
                  : 'bg-gray-800 text-gray-400 hover:bg-gray-700'
              }`}
            >
              {col.icon}
              <span>{col.label}</span>
            </button>
          ))}
        </div>

        {/* Search and Controls */}
        <div className="mb-6 flex flex-col lg:flex-row gap-4">
          <div className="flex-1 relative">
            <FaSearch className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
            <input
              type="text"
              placeholder="Search models, creators, capabilities..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full pl-10 pr-10 py-3 bg-slate-800 border border-gray-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-turquoise text-gray-300 placeholder-gray-500"
            />
            {searchTerm && (
              <button
                onClick={() => setSearchTerm('')}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-300"
              >
                <FaTimes />
              </button>
            )}
          </div>

          <div className="flex gap-2">
            <button
              onClick={() => setShowAdvancedFilters(!showAdvancedFilters)}
              className={`px-4 py-3 rounded-lg transition-colors flex items-center gap-2 ${
                showAdvancedFilters ? 'bg-turquoise text-black' : 'bg-gray-800 text-gray-400'
              }`}
            >
              <FaFilter />
              Filters
              {activeFiltersCount > 0 && (
                <span className="ml-1 px-2 py-0.5 bg-gold text-black text-xs rounded-full font-bold">
                  {activeFiltersCount}
                </span>
              )}
            </button>

            <select
              value={sortBy}
              onChange={(e) => setSortBy(e.target.value)}
              className="px-4 py-3 bg-slate-800 border border-gray-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-turquoise text-gray-300"
            >
              <option value="popular">Most Popular</option>
              <option value="name">Name A-Z</option>
              <option value="params">Parameter Size</option>
            </select>
          </div>
        </div>

        {/* Advanced Filters Panel */}
        {showAdvancedFilters && (
          <div className="mb-6 p-6 bg-gray-800 rounded-lg border border-gray-700 space-y-6">
            <div className="flex items-center justify-between">
              <h3 className="text-lg font-semibold">Advanced Filters</h3>
              <button onClick={clearFilters} className="text-sm text-turquoise hover:underline">
                Clear All
              </button>
            </div>

            {/* Capabilities */}
            <div>
              <h4 className="text-sm font-semibold text-gray-400 mb-3">Capabilities</h4>
              <div className="flex flex-wrap gap-2">
                {capabilities.map((cap) => (
                  <button
                    key={cap}
                    onClick={() => toggleCapability(cap)}
                    className={`px-4 py-2 rounded-lg text-sm flex items-center gap-2 transition-colors ${
                      selectedCapabilities.includes(cap)
                        ? 'bg-turquoise text-black font-semibold'
                        : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                    }`}
                  >
                    {selectedCapabilities.includes(cap) ? <FaCheckSquare /> : <FaSquare />}
                    {cap.charAt(0).toUpperCase() + cap.slice(1)}
                  </button>
                ))}
              </div>
            </div>

            {/* Tasks */}
            <div>
              <h4 className="text-sm font-semibold text-gray-400 mb-3">Task Categories</h4>
              <div className="flex flex-wrap gap-2">
                {tasks.map((task) => (
                  <button
                    key={task}
                    onClick={() => toggleTask(task)}
                    className={`px-4 py-2 rounded-lg text-sm transition-colors ${
                      selectedTasks.includes(task)
                        ? 'bg-gold text-black font-semibold'
                        : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                    }`}
                  >
                    {task}
                  </button>
                ))}
              </div>
            </div>

            {/* Creator */}
            <div>
              <h4 className="text-sm font-semibold text-gray-400 mb-3">Creator</h4>
              <select
                value={filterCreator}
                onChange={(e) => setFilterCreator(e.target.value)}
                className="w-full px-4 py-2 bg-gray-700 border border-gray-600 rounded-lg text-gray-300 focus:outline-none focus:ring-2 focus:ring-turquoise"
              >
                {creators.map(creator => (
                  <option key={creator} value={creator}>
                    {creator === 'all' ? 'All Creators' : creator}
                  </option>
                ))}
              </select>
            </div>

            {/* Pricing */}
            <div>
              <h4 className="text-sm font-semibold text-gray-400 mb-3">Pricing Tier</h4>
              <div className="flex gap-2">
                {pricingTiers.map((tier) => (
                  <button
                    key={tier}
                    onClick={() => setPricingFilter(tier)}
                    className={`px-4 py-2 rounded-lg text-sm flex-1 transition-colors ${
                      pricingFilter === tier
                        ? 'bg-gradient-to-r from-turquoise to-gold text-black font-semibold'
                        : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                    }`}
                  >
                    {tier === 'all' ? 'All' : tier.split('-').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' ')}
                  </button>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Active Filters Chips */}
        {activeFiltersCount > 0 && (
          <div className="mb-6 flex flex-wrap gap-2">
            {searchTerm && (
              <span className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                Search: {searchTerm}
                <button onClick={() => setSearchTerm('')}><FaTimes size={12} /></button>
              </span>
            )}
            {filterCreator !== 'all' && (
              <span className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                Creator: {filterCreator}
                <button onClick={() => setFilterCreator('all')}><FaTimes size={12} /></button>
              </span>
            )}
            {selectedCapabilities.map(cap => (
              <span key={cap} className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                {cap}
                <button onClick={() => toggleCapability(cap)}><FaTimes size={12} /></button>
              </span>
            ))}
          </div>
        )}

        {/* Premium Models Section */}
        <div className="mb-10">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-2xl font-bold">
              Premium Model Cards ({filteredPremiumModels.length})
            </h2>
            {selectedModels.length > 0 && (
              <button
                onClick={() => setShowComparison(true)}
                className="btn-primary flex items-center gap-2"
              >
                <FaChartBar /> Compare {selectedModels.length} Models
              </button>
            )}
          </div>
          
          {filteredPremiumModels.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-400 text-lg">No models match your filters.</p>
              <button onClick={clearFilters} className="mt-4 text-turquoise hover:underline">
                Clear all filters
              </button>
            </div>
          ) : (
            <ModelCardsGrid 
              models={filteredPremiumModels} 
              viewMode={viewMode}
              selectedModels={selectedModels}
              onToggleSelect={toggleModelSelection}
              onOpenDetail={setSelectedModel}
              onTestModel={(model) => {
                setSelectedModel(model);
                setShowPlayground(true);
              }}
            />
          )}
        </div>

        {/* Comparison Sticky Bar */}
        {selectedModels.length > 0 && !showComparison && (
          <div className="fixed bottom-0 left-0 right-0 bg-gray-900 border-t border-gray-700 p-4 z-40">
            <div className="max-w-7xl mx-auto flex items-center justify-between">
              <div className="flex items-center gap-4">
                <span className="text-sm text-gray-400">
                  {selectedModels.length} model{selectedModels.length !== 1 ? 's' : ''} selected
                </span>
                <div className="flex gap-2">
                  {selectedModels.map(m => (
                    <span key={m.model} className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                      {m.model}
                      <button onClick={() => toggleModelSelection(m)}>
                        <FaTimes size={12} />
                      </button>
                    </span>
                  ))}
                </div>
              </div>
              <div className="flex gap-2">
                <button
                  onClick={() => setSelectedModels([])}
                  className="px-4 py-2 bg-gray-800 text-gray-400 rounded-lg hover:bg-gray-700"
                >
                  Clear
                </button>
                <button
                  onClick={() => setShowComparison(true)}
                  className="btn-primary flex items-center gap-2"
                >
                  <FaChartBar /> Compare
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Model Comparison Modal */}
        {showComparison && selectedModels.length > 0 && (
          <ModelComparisonModal 
            models={selectedModels}
            onClose={() => setShowComparison(false)}
          />
        )}

        {/* Model Detail Modal */}
        {selectedModel && !showPlayground && (
          <ModelDetailModal 
            model={selectedModel}
            onClose={() => setSelectedModel(null)}
            onTest={() => setShowPlayground(true)}
          />
        )}

        {/* Playground Modal */}
        {showPlayground && selectedModel && (
          <PlaygroundModal 
            model={selectedModel}
            onClose={() => {
              setShowPlayground(false);
              setSelectedModel(null);
            }}
            samplePrompts={samplePrompts}
          />
        )}

        <div className="mt-12">
          <h2 className="text-2xl font-bold mb-4">Public Models (Hugging Face sample)</h2>
          {loading && <p>Loading models…</p>}
          {error && <p className="text-red-400">Error: {error}</p>}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {models.map((m, idx) => {
              const id = m.modelId || m.id || m.modelId?.toString() || `model-${idx}`;
              const url = `https://huggingface.co/${id}`;
              const tags = Array.isArray(m.tags) ? m.tags.slice(0, 3).join(', ') : '';

              return (
                <Link
                  key={id}
                  to={`/models/${encodeURIComponent(id)}`}
                  className="block bg-slate-800 rounded-lg p-4 hover:shadow-lg transition-shadow hover:bg-slate-700"
                >
                  <h3 className="text-lg font-semibold text-white mb-2 truncate">{id}</h3>
                  {tags && <p className="text-sm text-gray-400 mb-2">{tags}</p>}
                  {m.pipeline_tag && <p className="text-xs text-gray-500">Pipeline: {m.pipeline_tag}</p>}
                  <div className="mt-3 text-xs text-gray-400">View details →</div>
                </Link>
              );
            })}
          </div>
        </div>
      </div>
    </div>
  );
};

// Model Comparison Modal Component
const ModelComparisonModal = ({ models, onClose }) => {
  const metrics = ['rating', 'paramSize', 'context', 'maxOutput'];
  
  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-6xl w-full max-h-[90vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between">
          <h3 className="text-2xl font-bold">Model Comparison</h3>
          <button onClick={onClose} className="text-gray-400 hover:text-white">
            <FaTimes size={20} />
          </button>
        </div>

        <div className="p-6">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-800">
                  <th className="text-left py-4 px-4 text-gray-400">Feature</th>
                  {models.map(model => (
                    <th key={model.model} className="text-left py-4 px-4">
                      <div className="font-bold text-turquoise">{model.model}</div>
                      <div className="text-xs text-gray-500 font-normal">{model.creator}</div>
                    </th>
                  ))}
                </tr>
              </thead>
              <tbody>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4 text-gray-400">Rating</td>
                  {models.map(model => (
                    <td key={model.model} className="py-4 px-4">
                      <div className="flex items-center gap-1">
                        {'★'.repeat(model.rating || 0)}
                        <span className="text-gray-500">{'★'.repeat(5 - (model.rating || 0))}</span>
                      </div>
                    </td>
                  ))}
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4 text-gray-400">Parameters</td>
                  {models.map(model => (
                    <td key={model.model} className="py-4 px-4">{model.paramSize || '—'}</td>
                  ))}
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4 text-gray-400">Context Window</td>
                  {models.map(model => (
                    <td key={model.model} className="py-4 px-4">{model.context || '—'}</td>
                  ))}
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4 text-gray-400">Max Output</td>
                  {models.map(model => (
                    <td key={model.model} className="py-4 px-4">{model.maxOutput || '—'}</td>
                  ))}
                </tr>
                <tr className="border-b border-gray-800">
                  <td className="py-4 px-4 text-gray-400">Capabilities</td>
                  {models.map(model => (
                    <td key={model.model} className="py-4 px-4">
                      <div className="flex flex-wrap gap-1">
                        {model.capabilities?.map(cap => (
                          <span key={cap} className="text-xs px-2 py-1 bg-turquoise/20 text-turquoise rounded">
                            {cap}
                          </span>
                        ))}
                      </div>
                    </td>
                  ))}
                </tr>
                {models[0]?.benchmarks && (
                  <>
                    <tr className="bg-gray-800/50">
                      <td colSpan={models.length + 1} className="py-3 px-4 font-semibold">Benchmarks</td>
                    </tr>
                    {Object.keys(models[0].benchmarks).map(benchmark => (
                      <tr key={benchmark} className="border-b border-gray-800">
                        <td className="py-3 px-4 text-gray-400">{benchmark}</td>
                        {models.map(model => {
                          const value = model.benchmarks?.[benchmark] || 0;
                          const max = Math.max(...models.map(m => m.benchmarks?.[benchmark] || 0));
                          const isMax = value === max;
                          return (
                            <td key={model.model} className="py-3 px-4">
                              <div className="flex items-center gap-2">
                                <span className={isMax ? 'text-gold font-bold' : ''}>{value}%</span>
                                {isMax && <FaStar className="text-gold" size={12} />}
                              </div>
                            </td>
                          );
                        })}
                      </tr>
                    ))}
                  </>
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
  );
};

// Model Detail Modal Component
const ModelDetailModal = ({ model, onClose, onTest }) => {
  const [imageError, setImageError] = useState(false);

  // Generate consistent color from model creator name
  const getColorFromString = (str) => {
    const colors = [
      { from: 'turquoise', to: 'cyan-500', icon: 'turquoise' },
      { from: 'purple', to: 'pink-500', icon: 'purple-400' },
      { from: 'blue-500', to: 'indigo-500', icon: 'blue-400' },
      { from: 'green-500', to: 'emerald-500', icon: 'green-400' },
      { from: 'orange-500', to: 'red-500', icon: 'orange-400' },
      { from: 'yellow-500', to: 'amber-500', icon: 'yellow-400' },
      { from: 'pink-500', to: 'rose-500', icon: 'pink-400' },
      { from: 'indigo-500', to: 'violet-500', icon: 'indigo-400' },
      { from: 'teal-500', to: 'cyan-400', icon: 'teal-400' },
      { from: 'red-500', to: 'orange-600', icon: 'red-400' },
    ];
    
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      hash = ((hash << 5) - hash) + str.charCodeAt(i);
      hash = hash & hash;
    }
    
    return colors[Math.abs(hash) % colors.length];
  };

  const modelColor = getColorFromString(model.creator || model.model || 'default');

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between">
          <div className="flex items-center gap-4">
            {imageError || !model.companyIcon ? (
              <div className={`w-12 h-12 rounded-lg bg-gradient-to-br from-${modelColor.from}/20 to-${modelColor.to}/20 flex items-center justify-center`}>
                <FaRocket className={`text-2xl text-${modelColor.icon}`} />
              </div>
            ) : (
              <img 
                src={model.companyIcon} 
                alt={model.creator} 
                className="w-12 h-12 rounded-lg object-cover"
                onError={() => setImageError(true)}
              />
            )}
            <div>
              <h3 className="text-2xl font-bold">{model.model}</h3>
              <p className="text-gray-400">{model.creator}</p>
            </div>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-white">
            <FaTimes size={20} />
          </button>
        </div>

        <div className="p-6 space-y-6">
          {/* Quick Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div className="bg-gray-800 p-4 rounded-lg">
              <div className="text-sm text-gray-400">Parameters</div>
              <div className="text-xl font-bold text-turquoise">{model.paramSize || '—'}</div>
            </div>
            <div className="bg-gray-800 p-4 rounded-lg">
              <div className="text-sm text-gray-400">Context</div>
              <div className="text-xl font-bold text-turquoise">{model.context || '—'}</div>
            </div>
            <div className="bg-gray-800 p-4 rounded-lg">
              <div className="text-sm text-gray-400">Max Output</div>
              <div className="text-xl font-bold text-turquoise">{model.maxOutput || '—'}</div>
            </div>
            <div className="bg-gray-800 p-4 rounded-lg">
              <div className="text-sm text-gray-400">Rating</div>
              <div className="text-xl font-bold text-gold">
                {'★'.repeat(model.rating || 0)}
              </div>
            </div>
          </div>

          {/* Capabilities */}
          <div>
            <h4 className="text-lg font-semibold mb-3">Capabilities</h4>
            <div className="flex flex-wrap gap-2">
              {model.capabilities?.map(cap => (
                <span key={cap} className="px-4 py-2 bg-turquoise/20 text-turquoise rounded-lg">
                  {cap.charAt(0).toUpperCase() + cap.slice(1)}
                </span>
              ))}
            </div>
          </div>

          {/* Benchmarks */}
          {model.benchmarks && (
            <div>
              <h4 className="text-lg font-semibold mb-3">Benchmarks</h4>
              <div className="grid grid-cols-2 gap-3">
                {Object.entries(model.benchmarks).map(([key, value]) => (
                  <div key={key} className="bg-gray-800 p-3 rounded-lg">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm text-gray-400">{key}</span>
                      <span className="text-lg font-bold text-turquoise">{value}%</span>
                    </div>
                    <div className="w-full bg-gray-700 rounded-full h-2">
                      <div 
                        className="bg-gradient-to-r from-turquoise to-gold h-2 rounded-full"
                        style={{ width: `${value}%` }}
                      />
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Test Model Button */}
          <button
            onClick={onTest}
            className="w-full btn-primary flex items-center justify-center gap-2 py-4"
          >
            <FaPlay /> Test This Model
          </button>
        </div>
      </div>
    </div>
  );
};

// Playground Modal Component
const PlaygroundModal = ({ model, onClose, samplePrompts }) => {
  const [prompt, setPrompt] = useState('');
  const [result, setResult] = useState('');
  const [loading, setLoading] = useState(false);

  const handleTest = async () => {
    if (!prompt.trim()) return;
    setLoading(true);
    // Simulate API call
    setTimeout(() => {
      setResult(`Demo response from ${model.model}:\n\nThis is a simulated response. In production, this would connect to the actual model API and return real results based on your prompt.\n\nYour prompt was: "${prompt}"`);
      setLoading(false);
    }, 1500);
  };

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-5xl w-full max-h-[90vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between">
          <div>
            <h3 className="text-2xl font-bold">Test {model.model}</h3>
            <p className="text-sm text-gray-400">Try sample prompts or write your own</p>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-white">
            <FaTimes size={20} />
          </button>
        </div>

        <div className="p-6 space-y-6">
          {/* Sample Prompts */}
          <div>
            <h4 className="text-sm font-semibold text-gray-400 mb-3">Sample Prompts</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
              {samplePrompts.map((sample, idx) => (
                <button
                  key={idx}
                  onClick={() => setPrompt(sample)}
                  className="text-left p-3 bg-gray-800 rounded-lg hover:bg-gray-700 transition-colors text-sm text-gray-300"
                >
                  {sample}
                </button>
              ))}
            </div>
          </div>

          {/* Prompt Input */}
          <div>
            <h4 className="text-sm font-semibold text-gray-400 mb-3">Your Prompt</h4>
            <textarea
              value={prompt}
              onChange={(e) => setPrompt(e.target.value)}
              placeholder="Enter your prompt here..."
              className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 placeholder-gray-500 focus:outline-none focus:border-turquoise resize-none"
              rows="4"
            />
          </div>

          {/* Test Button */}
          <button
            onClick={handleTest}
            disabled={loading || !prompt.trim()}
            className="w-full btn-primary flex items-center justify-center gap-2 py-3 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading ? (
              <>
                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black"></div>
                Processing...
              </>
            ) : (
              <>
                <FaRocket /> Run Test
              </>
            )}
          </button>

          {/* Result */}
          {result && (
            <div>
              <h4 className="text-sm font-semibold text-gray-400 mb-3">Response</h4>
              <div className="p-4 bg-gray-800 rounded-lg border border-turquoise/30">
                <pre className="text-gray-300 whitespace-pre-wrap text-sm">{result}</pre>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Models;
