import { useState } from 'react';
import { Link } from 'react-router-dom';
import {
  Trophy,
  Calendar,
  Users,
  DollarSign,
  Award,
  CheckCircle,
  Code,
  Sparkles,
  Clock,
  Target,
  Star,
  ArrowRight,
  ExternalLink,
  Upload,
  Github,
  Zap,
  TrendingUp,
  Globe,
  MessageSquare
} from 'lucide-react';

const OpenAIChallengeDemo = () => {
  const [activeTab, setActiveTab] = useState('overview');

  const judges = [
    {
      name: 'Sarah Chen',
      role: 'Head of Developer Relations',
      company: 'OpenAI',
      avatar: 'https://ui-avatars.com/api/?name=Sarah+Chen&background=10b981&color=fff&size=128'
    },
    {
      name: 'Marcus Rodriguez',
      role: 'Senior AI Engineer',
      company: 'OpenAI',
      avatar: 'https://ui-avatars.com/api/?name=Marcus+Rodriguez&background=3b82f6&color=fff&size=128'
    },
    {
      name: 'Dr. Emily Watson',
      role: 'Research Lead',
      company: 'OpenAI',
      avatar: 'https://ui-avatars.com/api/?name=Emily+Watson&background=8b5cf6&color=fff&size=128'
    }
  ];

  const prizes = [
    {
      place: '1st Place',
      amount: '$10,000',
      icon: Trophy,
      color: 'from-yellow-400 to-orange-500',
      benefits: [
        'Featured on VibeCodingBros homepage',
        'OpenAI API credits worth $5,000',
        'Direct meeting with OpenAI DevRel team',
        'Case study written about your project',
        'Lifetime Pro membership'
      ]
    },
    {
      place: '2nd Place',
      amount: '$5,000',
      icon: Award,
      color: 'from-gray-300 to-gray-400',
      benefits: [
        'Featured in newsletter',
        'OpenAI API credits worth $2,500',
        'Interview with OpenAI team',
        '1-year Creator membership'
      ]
    },
    {
      place: '3rd Place',
      amount: '$2,500',
      icon: Star,
      color: 'from-orange-400 to-amber-600',
      benefits: [
        'Community spotlight',
        'OpenAI API credits worth $1,000',
        '1-year Pro membership'
      ]
    }
  ];

  const requirements = [
    {
      title: 'Use OpenAI APIs',
      description: 'Must integrate at least one OpenAI API (GPT-4, GPT-4 Turbo, DALL-E 3, Whisper, or Assistants API)',
      icon: Zap,
      mandatory: true
    },
    {
      title: 'Original Work',
      description: 'Project must be your original creation, built during the challenge period',
      icon: Sparkles,
      mandatory: true
    },
    {
      title: 'Open Source',
      description: 'Code must be publicly available on GitHub with clear documentation',
      icon: Github,
      mandatory: true
    },
    {
      title: 'Working Demo',
      description: 'Submit a live, functional demo that judges can test',
      icon: Globe,
      mandatory: true
    },
    {
      title: 'Video Walkthrough',
      description: 'Create a 3-5 minute video explaining your project',
      icon: Code,
      mandatory: true
    }
  ];

  const judgingCriteria = [
    {
      category: 'Innovation & Creativity',
      weight: '30%',
      description: 'Originality of concept, creative use of OpenAI APIs, novel problem-solving approach'
    },
    {
      category: 'Technical Excellence',
      weight: '25%',
      description: 'Code quality, architecture, performance, scalability, security best practices'
    },
    {
      category: 'User Experience',
      weight: '20%',
      description: 'Interface design, usability, accessibility, polish, attention to detail'
    },
    {
      category: 'Real-World Impact',
      weight: '15%',
      description: 'Practical utility, potential to solve real problems, market viability'
    },
    {
      category: 'Documentation',
      weight: '10%',
      description: 'Clear README, code comments, setup instructions, API usage examples'
    }
  ];

  const timeline = [
    {
      date: 'January 15, 2026',
      event: 'Challenge Launches',
      description: 'Registration opens, submit your project idea',
      status: 'upcoming'
    },
    {
      date: 'January 22, 2026',
      event: 'Kickoff Webinar',
      description: 'Live Q&A with OpenAI team, technical walkthrough',
      status: 'upcoming'
    },
    {
      date: 'February 15, 2026',
      event: 'Mid-Challenge Check-in',
      description: 'Optional project review sessions with mentors',
      status: 'upcoming'
    },
    {
      date: 'March 15, 2026',
      event: 'Submissions Close',
      description: 'Final deadline for project submissions (11:59 PM PT)',
      status: 'upcoming'
    },
    {
      date: 'March 22, 2026',
      event: 'Judging Complete',
      description: 'OpenAI team evaluates all submissions',
      status: 'upcoming'
    },
    {
      date: 'March 29, 2026',
      event: 'Winners Announced',
      description: 'Live announcement event, prizes distributed',
      status: 'upcoming'
    }
  ];

  const topSubmissions = [
    {
      id: 1,
      title: 'MediAI - AI Medical Assistant',
      author: 'Jessica Park',
      description: 'HIPAA-compliant medical documentation assistant using GPT-4 for clinical note generation',
      apis: ['GPT-4', 'Whisper'],
      votes: 342,
      image: 'https://images.unsplash.com/photo-1576091160399-112ba8d25d1d?w=400&h=300&fit=crop'
    },
    {
      id: 2,
      title: 'CodeMentor - Interactive Tutor',
      author: 'Alex Thompson',
      description: 'Real-time coding tutor with live feedback using GPT-4 and code execution sandbox',
      apis: ['GPT-4 Turbo', 'Assistants API'],
      votes: 298,
      image: 'https://images.unsplash.com/photo-1516116216624-53e697fedbea?w=400&h=300&fit=crop'
    },
    {
      id: 3,
      title: 'StoryForge - AI Novel Writer',
      author: 'Maria Santos',
      description: 'Collaborative fiction writing tool with DALL-E illustrations and GPT-4 story generation',
      apis: ['GPT-4', 'DALL-E 3'],
      votes: 287,
      image: 'https://images.unsplash.com/photo-1456513080510-7bf3a84b82f8?w=400&h=300&fit=crop'
    }
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-purple-50">
      {/* Hero Section */}
      <div className="relative bg-gradient-to-r from-blue-600 via-purple-600 to-pink-500 text-white overflow-hidden">
        <div className="absolute inset-0 bg-black/10"></div>
        <div className="absolute inset-0" style={{
          backgroundImage: 'url("data:image/svg+xml,%3Csvg width="60" height="60" viewBox="0 0 60 60" xmlns="http://www.w3.org/2000/svg"%3E%3Cg fill="none" fill-rule="evenodd"%3E%3Cg fill="%23ffffff" fill-opacity="0.05"%3E%3Cpath d="M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z"/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")'
        }}></div>
        
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 md:py-24">
          <div className="flex items-center justify-between mb-8">
            <Link to="/challenges" className="text-white/90 hover:text-white text-sm flex items-center gap-2">
              ← Back to Challenges
            </Link>
            <div className="flex items-center gap-2 bg-white/10 backdrop-blur-sm border border-white/20 px-4 py-2 rounded-full">
              <img src="https://openai.com/favicon.ico" alt="OpenAI" className="w-5 h-5" />
              <span className="font-semibold text-sm">Sponsored by OpenAI</span>
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-white/10 backdrop-blur-sm border border-white/20 px-4 py-2 rounded-full mb-6">
                <Sparkles className="w-4 h-4" />
                <span className="text-sm font-medium">Featured Challenge</span>
              </div>
              
              <h1 className="text-5xl md:text-6xl font-bold mb-6 leading-tight">
                OpenAI Innovation Challenge
              </h1>
              
              <p className="text-xl text-blue-100 mb-8 leading-relaxed">
                Build the most innovative application using GPT-4, DALL-E, Whisper, or Assistants API. 
                Win $10,000 and get featured by OpenAI.
              </p>

              <div className="flex flex-wrap gap-4 mb-8">
                <div className="bg-white/10 backdrop-blur-sm border border-white/20 rounded-lg px-6 py-4">
                  <div className="flex items-center gap-2 text-white/80 mb-1">
                    <DollarSign className="w-5 h-5" />
                    <span className="text-sm font-medium">Total Prize Pool</span>
                  </div>
                  <p className="text-3xl font-bold">$17,500</p>
                </div>
                
                <div className="bg-white/10 backdrop-blur-sm border border-white/20 rounded-lg px-6 py-4">
                  <div className="flex items-center gap-2 text-white/80 mb-1">
                    <Calendar className="w-5 h-5" />
                    <span className="text-sm font-medium">Time Remaining</span>
                  </div>
                  <p className="text-3xl font-bold">45 Days</p>
                </div>
                
                <div className="bg-white/10 backdrop-blur-sm border border-white/20 rounded-lg px-6 py-4">
                  <div className="flex items-center gap-2 text-white/80 mb-1">
                    <Users className="w-5 h-5" />
                    <span className="text-sm font-medium">Participants</span>
                  </div>
                  <p className="text-3xl font-bold">1,247</p>
                </div>
              </div>

              <div className="flex flex-wrap gap-4">
                <button className="bg-white text-blue-600 px-8 py-4 rounded-lg font-semibold text-lg hover:bg-blue-50 transition-all shadow-xl flex items-center gap-2 group">
                  <span>Submit Your Project</span>
                  <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
                </button>
                
                <button className="bg-white/10 backdrop-blur-sm border-2 border-white/30 text-white px-8 py-4 rounded-lg font-semibold text-lg hover:bg-white/20 transition-all flex items-center gap-2">
                  <span>View Rules</span>
                  <ExternalLink className="w-5 h-5" />
                </button>
              </div>
            </div>

            <div className="relative">
              <div className="relative bg-white/10 backdrop-blur-md border border-white/20 rounded-2xl p-8 shadow-2xl">
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 bg-gradient-to-br from-yellow-400 to-orange-500 rounded-full flex items-center justify-center shadow-lg">
                    <Trophy className="w-8 h-8 text-white" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Grand Prize</h3>
                    <p className="text-blue-100">1st Place Winner</p>
                  </div>
                </div>
                
                <ul className="space-y-3">
                  {prizes[0].benefits.map((benefit, idx) => (
                    <li key={idx} className="flex items-start gap-3">
                      <CheckCircle className="w-5 h-5 text-green-300 flex-shrink-0 mt-0.5" />
                      <span className="text-white/90">{benefit}</span>
                    </li>
                  ))}
                </ul>
              </div>
              
              {/* Floating Elements */}
              <div className="absolute -top-4 -right-4 w-24 h-24 bg-yellow-400/20 backdrop-blur-sm rounded-full flex items-center justify-center border border-yellow-400/30 animate-pulse">
                <DollarSign className="w-12 h-12 text-yellow-300" />
              </div>
              
              <div className="absolute -bottom-4 -left-4 w-20 h-20 bg-purple-400/20 backdrop-blur-sm rounded-full flex items-center justify-center border border-purple-400/30 animate-bounce">
                <Sparkles className="w-10 h-10 text-purple-300" />
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Tabs Navigation */}
      <div className="bg-white border-b sticky top-0 z-10 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex gap-8 overflow-x-auto">
            {['overview', 'prizes', 'rules', 'timeline', 'submissions', 'judges'].map((tab) => (
              <button
                key={tab}
                onClick={() => setActiveTab(tab)}
                className={`py-4 px-2 text-sm font-medium border-b-2 whitespace-nowrap transition-colors ${
                  activeTab === tab
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                {tab.charAt(0).toUpperCase() + tab.slice(1)}
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* Overview Tab */}
        {activeTab === 'overview' && (
          <div className="space-y-12">
            <section>
              <h2 className="text-3xl font-bold text-gray-900 mb-6">About the Challenge</h2>
              <div className="bg-white rounded-xl shadow-sm p-8 border">
                <p className="text-lg text-gray-700 leading-relaxed mb-6">
                  The OpenAI Innovation Challenge is your opportunity to showcase your creativity and technical skills 
                  by building groundbreaking applications with OpenAI's most advanced APIs. Whether you're solving real-world 
                  problems, creating delightful experiences, or pushing the boundaries of what's possible with AI, 
                  this is your chance to shine.
                </p>
                
                <div className="grid md:grid-cols-2 gap-6 mt-8">
                  <div className="bg-gradient-to-br from-blue-50 to-purple-50 rounded-lg p-6 border border-blue-100">
                    <div className="flex items-center gap-3 mb-4">
                      <div className="w-10 h-10 bg-blue-500 rounded-lg flex items-center justify-center">
                        <Code className="w-6 h-6 text-white" />
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900">What You Can Build</h3>
                    </div>
                    <ul className="space-y-2 text-gray-700">
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>Web applications</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>Mobile apps (iOS/Android)</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>Browser extensions</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>CLI tools</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>Desktop applications</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span>API services</span>
                      </li>
                    </ul>
                  </div>

                  <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-lg p-6 border border-purple-100">
                    <div className="flex items-center gap-3 mb-4">
                      <div className="w-10 h-10 bg-purple-500 rounded-lg flex items-center justify-center">
                        <Zap className="w-6 h-6 text-white" />
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900">OpenAI APIs</h3>
                    </div>
                    <ul className="space-y-2 text-gray-700">
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>GPT-4</strong> - Advanced language model</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>GPT-4 Turbo</strong> - Faster, cheaper inference</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>DALL-E 3</strong> - Image generation</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>Whisper</strong> - Speech recognition</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>Assistants API</strong> - Stateful agents</span>
                      </li>
                      <li className="flex items-start gap-2">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span><strong>Embeddings</strong> - Vector representations</span>
                      </li>
                    </ul>
                  </div>
                </div>
              </div>
            </section>

            <section>
              <h2 className="text-3xl font-bold text-gray-900 mb-6">Why Participate?</h2>
              <div className="grid md:grid-cols-3 gap-6">
                <div className="bg-white rounded-xl shadow-sm p-6 border hover:shadow-md transition-shadow">
                  <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center mb-4">
                    <DollarSign className="w-6 h-6 text-green-600" />
                  </div>
                  <h3 className="text-xl font-semibold text-gray-900 mb-2">Win Big Prizes</h3>
                  <p className="text-gray-600">
                    $17,500 in cash prizes plus OpenAI API credits, memberships, and exclusive opportunities.
                  </p>
                </div>

                <div className="bg-white rounded-xl shadow-sm p-6 border hover:shadow-md transition-shadow">
                  <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mb-4">
                    <TrendingUp className="w-6 h-6 text-blue-600" />
                  </div>
                  <h3 className="text-xl font-semibold text-gray-900 mb-2">Get Recognized</h3>
                  <p className="text-gray-600">
                    Featured by OpenAI, case studies, interviews, and exposure to thousands of developers.
                  </p>
                </div>

                <div className="bg-white rounded-xl shadow-sm p-6 border hover:shadow-md transition-shadow">
                  <div className="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center mb-4">
                    <Users className="w-6 h-6 text-purple-600" />
                  </div>
                  <h3 className="text-xl font-semibold text-gray-900 mb-2">Network & Learn</h3>
                  <p className="text-gray-600">
                    Connect with OpenAI team, fellow developers, and potential employers or investors.
                  </p>
                </div>
              </div>
            </section>
          </div>
        )}

        {/* Prizes Tab */}
        {activeTab === 'prizes' && (
          <div className="space-y-8">
            <h2 className="text-3xl font-bold text-gray-900">Prize Breakdown</h2>
            
            <div className="grid md:grid-cols-3 gap-8">
              {prizes.map((prize, idx) => (
                <div key={idx} className="bg-white rounded-xl shadow-lg p-8 border-2 border-gray-100 hover:border-blue-200 transition-all">
                  <div className={`w-16 h-16 bg-gradient-to-br ${prize.color} rounded-full flex items-center justify-center mb-6 shadow-lg`}>
                    <prize.icon className="w-8 h-8 text-white" />
                  </div>
                  
                  <h3 className="text-2xl font-bold text-gray-900 mb-2">{prize.place}</h3>
                  <p className="text-4xl font-bold text-blue-600 mb-6">{prize.amount}</p>
                  
                  <ul className="space-y-3">
                    {prize.benefits.map((benefit, bidx) => (
                      <li key={bidx} className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" />
                        <span className="text-gray-700">{benefit}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>

            <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-8 border border-blue-200">
              <h3 className="text-xl font-semibold text-gray-900 mb-4 flex items-center gap-2">
                <Sparkles className="w-6 h-6 text-blue-600" />
                Additional Perks for All Participants
              </h3>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">$100 in OpenAI API credits for all registered participants</span>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Access to exclusive office hours with OpenAI engineers</span>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Certificate of participation</span>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Community voting awards (3 × $500)</span>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Rules Tab */}
        {activeTab === 'rules' && (
          <div className="space-y-8">
            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6">Challenge Requirements</h2>
              <div className="space-y-4">
                {requirements.map((req, idx) => (
                  <div key={idx} className="bg-white rounded-xl shadow-sm p-6 border hover:border-blue-200 transition-colors">
                    <div className="flex items-start gap-4">
                      <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center flex-shrink-0">
                        <req.icon className="w-6 h-6 text-blue-600" />
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <h3 className="text-xl font-semibold text-gray-900">{req.title}</h3>
                          {req.mandatory && (
                            <span className="bg-red-100 text-red-700 text-xs font-semibold px-2 py-1 rounded">
                              Required
                            </span>
                          )}
                        </div>
                        <p className="text-gray-600">{req.description}</p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6">Judging Criteria</h2>
              <div className="bg-white rounded-xl shadow-sm border overflow-hidden">
                <table className="w-full">
                  <thead className="bg-gray-50 border-b">
                    <tr>
                      <th className="text-left px-6 py-4 text-sm font-semibold text-gray-900">Category</th>
                      <th className="text-left px-6 py-4 text-sm font-semibold text-gray-900">Weight</th>
                      <th className="text-left px-6 py-4 text-sm font-semibold text-gray-900">What We Look For</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {judgingCriteria.map((criteria, idx) => (
                      <tr key={idx} className="hover:bg-gray-50">
                        <td className="px-6 py-4 font-medium text-gray-900">{criteria.category}</td>
                        <td className="px-6 py-4">
                          <span className="bg-blue-100 text-blue-700 px-3 py-1 rounded-full text-sm font-semibold">
                            {criteria.weight}
                          </span>
                        </td>
                        <td className="px-6 py-4 text-gray-600">{criteria.description}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          </div>
        )}

        {/* Timeline Tab */}
        {activeTab === 'timeline' && (
          <div>
            <h2 className="text-3xl font-bold text-gray-900 mb-8">Challenge Timeline</h2>
            <div className="relative">
              {/* Timeline Line */}
              <div className="absolute left-8 top-0 bottom-0 w-0.5 bg-gradient-to-b from-blue-500 to-purple-500"></div>
              
              <div className="space-y-8">
                {timeline.map((item, idx) => (
                  <div key={idx} className="relative pl-20">
                    <div className="absolute left-0 w-16 h-16 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-bold shadow-lg">
                      {idx + 1}
                    </div>
                    
                    <div className="bg-white rounded-xl shadow-sm p-6 border hover:shadow-md transition-shadow">
                      <div className="flex items-center gap-3 mb-2">
                        <Clock className="w-5 h-5 text-gray-400" />
                        <span className="text-sm font-medium text-gray-500">{item.date}</span>
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">{item.event}</h3>
                      <p className="text-gray-600">{item.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Submissions Tab */}
        {activeTab === 'submissions' && (
          <div className="space-y-8">
            <div className="flex items-center justify-between">
              <h2 className="text-3xl font-bold text-gray-900">Top Submissions</h2>
              <div className="flex items-center gap-4">
                <select className="px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                  <option>Most Votes</option>
                  <option>Recent</option>
                  <option>Trending</option>
                </select>
              </div>
            </div>

            <div className="grid md:grid-cols-3 gap-6">
              {topSubmissions.map((submission) => (
                <div key={submission.id} className="bg-white rounded-xl shadow-sm border overflow-hidden hover:shadow-lg transition-shadow group">
                  <div className="relative h-48 overflow-hidden bg-gray-200">
                    <img 
                      src={submission.image} 
                      alt={submission.title}
                      className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                    />
                    <div className="absolute top-4 right-4 bg-white rounded-lg px-3 py-1 shadow-lg flex items-center gap-2">
                      <Star className="w-4 h-4 text-yellow-500 fill-current" />
                      <span className="font-semibold text-gray-900">{submission.votes}</span>
                    </div>
                  </div>
                  
                  <div className="p-6">
                    <h3 className="text-xl font-semibold text-gray-900 mb-2">{submission.title}</h3>
                    <p className="text-sm text-gray-500 mb-3">by {submission.author}</p>
                    <p className="text-gray-600 text-sm mb-4">{submission.description}</p>
                    
                    <div className="flex flex-wrap gap-2 mb-4">
                      {submission.apis.map((api, idx) => (
                        <span key={idx} className="bg-blue-50 text-blue-700 px-3 py-1 rounded-full text-xs font-medium">
                          {api}
                        </span>
                      ))}
                    </div>
                    
                    <div className="flex gap-2">
                      <button className="flex-1 px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors text-sm font-medium">
                        View Project
                      </button>
                      <button className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                        <MessageSquare className="w-4 h-4 text-gray-600" />
                      </button>
                    </div>
                  </div>
                </div>
              ))}
            </div>

            <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-8 border border-blue-200 text-center">
              <Upload className="w-12 h-12 text-blue-600 mx-auto mb-4" />
              <h3 className="text-2xl font-bold text-gray-900 mb-2">Ready to Submit?</h3>
              <p className="text-gray-600 mb-6">Show us what you've built with OpenAI's APIs</p>
              <button className="bg-blue-600 text-white px-8 py-3 rounded-lg font-semibold hover:bg-blue-700 transition-colors inline-flex items-center gap-2">
                <span>Submit Your Project</span>
                <ArrowRight className="w-5 h-5" />
              </button>
            </div>
          </div>
        )}

        {/* Judges Tab */}
        {activeTab === 'judges' && (
          <div className="space-y-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">Meet the Judges</h2>
              <p className="text-lg text-gray-600 max-w-3xl mx-auto">
                Your projects will be evaluated by OpenAI's top engineers and developer relations experts.
              </p>
            </div>

            <div className="grid md:grid-cols-3 gap-8">
              {judges.map((judge, idx) => (
                <div key={idx} className="bg-white rounded-xl shadow-lg p-8 border text-center hover:shadow-xl transition-shadow">
                  <img 
                    src={judge.avatar}
                    alt={judge.name}
                    className="w-32 h-32 rounded-full mx-auto mb-6 border-4 border-blue-100"
                  />
                  <h3 className="text-xl font-bold text-gray-900 mb-2">{judge.name}</h3>
                  <p className="text-blue-600 font-medium mb-1">{judge.role}</p>
                  <p className="text-gray-500 text-sm mb-6">{judge.company}</p>
                  
                  <button className="text-blue-600 hover:text-blue-700 font-medium text-sm flex items-center gap-2 mx-auto">
                    <span>View Profile</span>
                    <ExternalLink className="w-4 h-4" />
                  </button>
                </div>
              ))}
            </div>

            <div className="bg-white rounded-xl shadow-sm p-8 border">
              <h3 className="text-2xl font-bold text-gray-900 mb-6 flex items-center gap-3">
                <Target className="w-7 h-7 text-blue-600" />
                What Judges Are Looking For
              </h3>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Technical Excellence</h4>
                  <ul className="space-y-2 text-gray-600">
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Clean, well-structured code</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Effective use of OpenAI APIs</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Performance optimization</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Security best practices</span>
                    </li>
                  </ul>
                </div>
                
                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Innovation & Impact</h4>
                  <ul className="space-y-2 text-gray-600">
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Novel approach to problems</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Real-world utility</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>User-centered design</span>
                    </li>
                    <li className="flex items-start gap-2">
                      <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                      <span>Potential for scale</span>
                    </li>
                  </ul>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Bottom CTA */}
      <div className="bg-gradient-to-r from-blue-600 to-purple-600 text-white py-16">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">Ready to Build Something Amazing?</h2>
          <p className="text-xl text-blue-100 mb-8">
            Join 1,247 developers competing for $17,500 in prizes
          </p>
          <div className="flex flex-wrap gap-4 justify-center">
            <button className="bg-white text-blue-600 px-8 py-4 rounded-lg font-semibold text-lg hover:bg-blue-50 transition-all shadow-xl inline-flex items-center gap-2 group">
              <span>Submit Your Project</span>
              <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
            </button>
            <button className="bg-white/10 backdrop-blur-sm border-2 border-white/30 text-white px-8 py-4 rounded-lg font-semibold text-lg hover:bg-white/20 transition-all inline-flex items-center gap-2">
              <Github className="w-5 h-5" />
              <span>View on GitHub</span>
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default OpenAIChallengeDemo;
