import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  FaUsers, FaDollarSign, FaChartLine, FaTrophy, 
  FaDownload, FaCalendarAlt 
} from 'react-icons/fa';
import {
  LineChart, Line, BarChart, Bar, PieChart, Pie, Cell,
  XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer
} from 'recharts';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

const PlatformAnalytics = () => {
  const navigate = useNavigate();
  const { user } = useAuth();
  const [loading, setLoading] = useState(true);
  const [period, setPeriod] = useState('30d');
  
  const [overview, setOverview] = useState(null);
  const [userGrowth, setUserGrowth] = useState([]);
  const [revenue, setRevenue] = useState([]);
  const [funnel, setFunnel] = useState(null);
  const [topPerformers, setTopPerformers] = useState(null);
  const [engagement, setEngagement] = useState(null);

  useEffect(() => {
    // Check if user is Creator tier
    if (user?.subscriptionTier !== 'creator') {
      navigate('/pricing');
      return;
    }

    fetchAnalytics();
  }, [user, period, navigate]);

  const fetchAnalytics = async () => {
    try {
      setLoading(true);

      const [overviewRes, growthRes, revenueRes, funnelRes, performersRes, engagementRes] = 
        await Promise.all([
          api.get('/analytics/overview'),
          api.get(`/analytics/users/growth?period=${period}`),
          api.get(`/analytics/revenue?period=${period}`),
          api.get('/analytics/conversions'),
          api.get('/analytics/top-performers'),
          api.get(`/analytics/engagement?period=${period}`),
        ]);

      setOverview(overviewRes.data);
      setUserGrowth(growthRes.data);
      setRevenue(revenueRes.data);
      setFunnel(funnelRes.data);
      setTopPerformers(performersRes.data);
      setEngagement(engagementRes.data);
    } catch (error) {
      console.error('Failed to fetch analytics:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleExport = async (format) => {
    try {
      const response = await api.get(`/analytics/export?format=${format}`, {
        responseType: format === 'csv' ? 'blob' : 'json',
      });

      if (format === 'csv') {
        const url = window.URL.createObjectURL(new Blob([response.data]));
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `analytics-${Date.now()}.csv`);
        document.body.appendChild(link);
        link.click();
        link.remove();
      } else {
        const dataStr = JSON.stringify(response.data, null, 2);
        const dataBlob = new Blob([dataStr], { type: 'application/json' });
        const url = window.URL.createObjectURL(dataBlob);
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `analytics-${Date.now()}.json`);
        document.body.appendChild(link);
        link.click();
        link.remove();
      }
    } catch (error) {
      console.error('Failed to export:', error);
    }
  };

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  const COLORS = ['#14F195', '#9945FF', '#FF6B9D', '#FFA500', '#00C9FF'];

  const tierDistributionData = overview ? [
    { name: 'Free', value: overview.userMetrics.freeUsers },
    { name: 'Pro', value: overview.userMetrics.proUsers },
    { name: 'Creator', value: overview.userMetrics.creatorUsers },
  ] : [];

  const revenueDistributionData = overview ? [
    { name: 'Subscriptions', value: parseFloat(overview.revenueMetrics.subscriptionRevenue) },
    { name: 'Marketplace', value: parseFloat(overview.revenueMetrics.marketplaceRevenue) },
    { name: 'Job Listings', value: parseFloat(overview.revenueMetrics.jobListingRevenue) },
    { name: 'Sponsorships', value: parseFloat(overview.revenueMetrics.challengeSponsorRevenue) },
  ] : [];

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center mb-8 gap-4">
          <div>
            <h1 className="text-4xl font-bold mb-2">
              Platform <span className="text-gradient">Analytics</span>
            </h1>
            <p className="text-gray-400">Comprehensive insights and metrics</p>
          </div>

          <div className="flex gap-4">
            {/* Period Selector */}
            <div className="flex items-center gap-2">
              <FaCalendarAlt className="text-gray-400" />
              <select
                value={period}
                onChange={(e) => setPeriod(e.target.value)}
                className="input-field"
              >
                <option value="7d">Last 7 days</option>
                <option value="30d">Last 30 days</option>
                <option value="90d">Last 90 days</option>
                <option value="1y">Last year</option>
              </select>
            </div>

            {/* Export Buttons */}
            <button
              onClick={() => handleExport('json')}
              className="btn-secondary flex items-center gap-2"
            >
              <FaDownload />
              JSON
            </button>
            <button
              onClick={() => handleExport('csv')}
              className="btn-secondary flex items-center gap-2"
            >
              <FaDownload />
              CSV
            </button>
          </div>
        </div>

        {/* Overview Cards */}
        {overview && (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <div className="card p-6">
              <div className="flex items-center justify-between mb-4">
                <FaUsers className="text-3xl text-turquoise" />
                <span className="text-sm text-gray-400">Users</span>
              </div>
              <h3 className="text-3xl font-bold mb-2">{overview.userMetrics.totalUsers}</h3>
              <p className="text-sm text-gray-400">
                {overview.userMetrics.activeUsers} active ({overview.userMetrics.activeRate}%)
              </p>
            </div>

            <div className="card p-6">
              <div className="flex items-center justify-between mb-4">
                <FaDollarSign className="text-3xl text-green-500" />
                <span className="text-sm text-gray-400">Revenue</span>
              </div>
              <h3 className="text-3xl font-bold mb-2">${overview.revenueMetrics.totalRevenue}</h3>
              <p className="text-sm text-gray-400">
                MRR: ${overview.revenueMetrics.mrr}
              </p>
            </div>

            <div className="card p-6">
              <div className="flex items-center justify-between mb-4">
                <FaChartLine className="text-3xl text-purple" />
                <span className="text-sm text-gray-400">ARR</span>
              </div>
              <h3 className="text-3xl font-bold mb-2">${overview.revenueMetrics.arr}</h3>
              <p className="text-sm text-gray-400">
                ARPU: ${overview.revenueMetrics.arpu}
              </p>
            </div>

            <div className="card p-6">
              <div className="flex items-center justify-between mb-4">
                <FaTrophy className="text-3xl text-yellow-500" />
                <span className="text-sm text-gray-400">Content</span>
              </div>
              <h3 className="text-3xl font-bold mb-2">{overview.contentMetrics.totalWorkflows}</h3>
              <p className="text-sm text-gray-400">
                {overview.contentMetrics.totalProducts} products
              </p>
            </div>
          </div>
        )}

        {/* Charts Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
          {/* User Growth */}
          <div className="card p-6">
            <h3 className="text-xl font-bold mb-4">User Growth</h3>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={userGrowth}>
                <CartesianGrid strokeDasharray="3 3" stroke="#333" />
                <XAxis dataKey="date" stroke="#888" />
                <YAxis stroke="#888" />
                <Tooltip 
                  contentStyle={{ backgroundColor: '#1a1a2e', border: '1px solid #333' }}
                />
                <Legend />
                <Line type="monotone" dataKey="total" stroke="#14F195" strokeWidth={2} />
                <Line type="monotone" dataKey="free" stroke="#888" />
                <Line type="monotone" dataKey="pro" stroke="#9945FF" />
                <Line type="monotone" dataKey="creator" stroke="#FF6B9D" />
              </LineChart>
            </ResponsiveContainer>
          </div>

          {/* Revenue Over Time */}
          <div className="card p-6">
            <h3 className="text-xl font-bold mb-4">Revenue Breakdown</h3>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={revenue}>
                <CartesianGrid strokeDasharray="3 3" stroke="#333" />
                <XAxis dataKey="date" stroke="#888" />
                <YAxis stroke="#888" />
                <Tooltip 
                  contentStyle={{ backgroundColor: '#1a1a2e', border: '1px solid #333' }}
                />
                <Legend />
                <Bar dataKey="subscription" stackId="a" fill="#14F195" />
                <Bar dataKey="marketplace" stackId="a" fill="#9945FF" />
                <Bar dataKey="jobListings" stackId="a" fill="#FF6B9D" />
                <Bar dataKey="sponsorships" stackId="a" fill="#FFA500" />
              </BarChart>
            </ResponsiveContainer>
          </div>

          {/* User Tier Distribution */}
          <div className="card p-6">
            <h3 className="text-xl font-bold mb-4">User Tier Distribution</h3>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={tierDistributionData}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                  outerRadius={100}
                  fill="#8884d8"
                  dataKey="value"
                >
                  {tierDistributionData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip 
                  contentStyle={{ backgroundColor: '#1a1a2e', border: '1px solid #333' }}
                />
              </PieChart>
            </ResponsiveContainer>
          </div>

          {/* Revenue Source Distribution */}
          <div className="card p-6">
            <h3 className="text-xl font-bold mb-4">Revenue Sources</h3>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={revenueDistributionData}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                  outerRadius={100}
                  fill="#8884d8"
                  dataKey="value"
                >
                  {revenueDistributionData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip 
                  contentStyle={{ backgroundColor: '#1a1a2e', border: '1px solid #333' }}
                />
              </PieChart>
            </ResponsiveContainer>
          </div>
        </div>

        {/* Conversion Funnel */}
        {funnel && (
          <div className="card p-6 mb-8">
            <h3 className="text-xl font-bold mb-4">Conversion Funnel</h3>
            <div className="space-y-4">
              {funnel.stages.map((stage, idx) => (
                <div key={idx}>
                  <div className="flex justify-between items-center mb-2">
                    <span className="font-semibold">{stage.name}</span>
                    <span className="text-gray-400">{stage.count} ({stage.percentage}%)</span>
                  </div>
                  <div className="w-full bg-gray-700 rounded-full h-4">
                    <div
                      className="bg-gradient-to-r from-turquoise to-purple h-4 rounded-full transition-all"
                      style={{ width: `${stage.percentage}%` }}
                    ></div>
                  </div>
                </div>
              ))}
            </div>

            <div className="mt-6 grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="text-center">
                <p className="text-sm text-gray-400">Free → Pro</p>
                <p className="text-2xl font-bold text-turquoise">{funnel.conversionRates.freeToProRate}%</p>
              </div>
              <div className="text-center">
                <p className="text-sm text-gray-400">Free → Creator</p>
                <p className="text-2xl font-bold text-purple">{funnel.conversionRates.freeToCreatorRate}%</p>
              </div>
              <div className="text-center">
                <p className="text-sm text-gray-400">Pro → Creator</p>
                <p className="text-2xl font-bold text-pink">{funnel.conversionRates.proToCreatorRate}%</p>
              </div>
              <div className="text-center">
                <p className="text-sm text-gray-400">Free → Paid</p>
                <p className="text-2xl font-bold text-green-500">{funnel.conversionRates.freeToPaidRate}%</p>
              </div>
            </div>
          </div>
        )}

        {/* Top Performers */}
        {topPerformers && (
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            {/* Top Products */}
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-4">Top Products</h3>
              <div className="space-y-3">
                {topPerformers.topProducts.slice(0, 5).map((product, idx) => (
                  <div key={idx} className="flex justify-between items-center">
                    <div>
                      <p className="font-semibold">{product.name}</p>
                      <p className="text-sm text-gray-400">{product.sales} sales</p>
                    </div>
                    <p className="text-turquoise font-bold">${product.revenue.toFixed(2)}</p>
                  </div>
                ))}
              </div>
            </div>

            {/* Top Creators */}
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-4">Top Creators</h3>
              <div className="space-y-3">
                {topPerformers.topCreators.slice(0, 5).map((creator, idx) => (
                  <div key={idx} className="flex justify-between items-center">
                    <div>
                      <p className="font-semibold">{creator.username}</p>
                      <p className="text-sm text-gray-400">{creator.sales} sales</p>
                    </div>
                    <p className="text-purple font-bold">${creator.revenue.toFixed(2)}</p>
                  </div>
                ))}
              </div>
            </div>

            {/* Top Job Posters */}
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-4">Top Job Posters</h3>
              <div className="space-y-3">
                {topPerformers.topJobPosters.slice(0, 5).map((poster, idx) => (
                  <div key={idx} className="flex justify-between items-center">
                    <div>
                      <p className="font-semibold">{poster.username}</p>
                      <p className="text-sm text-gray-400">{poster.listings} listings</p>
                    </div>
                    <p className="text-pink font-bold">${poster.spent.toFixed(2)}</p>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default PlatformAnalytics;
