import React from 'react';
import { useNavigate } from 'react-router-dom';
import { FaCheck, FaStar, FaRocket, FaCrown } from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';

import { FeatureComparison } from '../components/PremiumFeatureGate';
import PricingCTAButtons from '../components/PricingCTAButtons';

const Pricing = () => {
  const navigate = useNavigate();
  const { user, isAuthenticated } = useAuth();

  const handleSubscribe = async (tier, priceId) => {
    if (!isAuthenticated) {
      navigate('/signin');
      return;
    }

    try {
      const response = await api.post('/stripe/create-subscription-checkout', {
        priceId,
        tier,
      });

      // Redirect to Stripe checkout
      window.location.href = response.data.url;
    } catch (error) {
      console.error('Subscription error:', error);
      alert(error.response?.data?.message || 'Failed to start subscription');
    }
  };

  const tiers = [
    {
      name: 'Free',
      icon: <FaStar className="text-4xl text-yellow-500" />,
      price: '$0',
      period: 'forever',
      description: 'Perfect for getting started',
      features: [
        'Browse community content',
        'Access free challenges',
        'View job postings',
        'Basic profile',
        'Community discussions',
      ],
      notIncluded: [
        'Premium challenges',
        'Priority support',
        'Advanced analytics',
        'Sell products',
      ],
      cta: user?.subscriptionTier === 'free' ? 'Current Plan' : 'Downgrade',
      current: user?.subscriptionTier === 'free',
    },
    {
      name: 'Pro',
      icon: <FaRocket className="text-4xl text-turquoise" />,
      price: '$9.99',
      period: 'per month',
      priceId: import.meta.env.VITE_STRIPE_PRICE_PRO_MONTHLY || 'price_pro_monthly',
      yearlyPriceId: import.meta.env.VITE_STRIPE_PRICE_PRO_YEARLY || 'price_pro_yearly',
      description: 'For serious developers',
      features: [
        'All Free features',
        'Premium challenges',
        'No ads',
        'Priority support',
        'Advanced analytics',
        'Custom workflows',
        'Early access to features',
      ],
      notIncluded: [
        'Sell products in marketplace',
        'Revenue sharing',
      ],
      cta: 'Upgrade to Pro',
      current: user?.subscriptionTier === 'pro',
      popular: true,
    },
    {
      name: 'Creator',
      icon: <FaCrown className="text-4xl text-purple" />,
      price: '$29.99',
      period: 'per month',
      priceId: import.meta.env.VITE_STRIPE_PRICE_CREATOR_MONTHLY || 'price_creator_monthly',
      yearlyPriceId: import.meta.env.VITE_STRIPE_PRICE_CREATOR_YEARLY || 'price_creator_yearly',
      description: 'For content creators',
      features: [
        'All Pro features',
        'Sell products in marketplace',
        '80% revenue share',
        'Creator dashboard',
        'Analytics & insights',
        'Payout management',
        'Featured creator badge',
        'Dedicated support',
      ],
      notIncluded: [],
      cta: 'Become a Creator',
      current: user?.subscriptionTier === 'creator',
    },
  ];

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-7xl mx-auto">

        {/* Header */}
        <div className="text-center mb-16">
          <h1 className="text-5xl font-bold mb-4">
            Choose Your <span className="text-gradient">Plan</span>
          </h1>
          <p className="text-xl text-gray-400 max-w-2xl mx-auto">
            Unlock premium features and start monetizing your content
          </p>
        </div>

        {/* CTA Buttons */}
        <PricingCTAButtons />

        {/* Pricing Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-6xl mx-auto">
          {tiers.map((tier, idx) => (
            <div
              key={idx}
              className={`card relative ${
                tier.popular
                  ? 'ring-2 ring-turquoise transform scale-105'
                  : ''
              }`}
            >
              {tier.popular && (
                <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 px-4 py-1 bg-turquoise text-black text-sm font-bold rounded-full">
                  MOST POPULAR
                </div>
              )}

              <div className="p-8">
                {/* Icon */}
                <div className="mb-4">{tier.icon}</div>

                {/* Name & Price */}
                <h3 className="text-2xl font-bold mb-2">{tier.name}</h3>
                <div className="mb-4">
                  <span className="text-4xl font-bold">{tier.price}</span>
                  <span className="text-gray-400 ml-2">/ {tier.period}</span>
                </div>
                <p className="text-gray-400 mb-6">{tier.description}</p>

                {/* CTA Button */}
                {tier.current ? (
                  <button className="w-full btn-secondary mb-6" disabled>
                    Current Plan
                  </button>
                ) : tier.name === 'Free' ? (
                  <button className="w-full btn-secondary mb-6" disabled>
                    Free Forever
                  </button>
                ) : (
                  <button
                    onClick={() => handleSubscribe(tier.name.toLowerCase(), tier.priceId)}
                    className="w-full btn-primary mb-6"
                  >
                    {tier.cta}
                  </button>
                )}

                {/* Features */}
                <div className="space-y-3">
                  {tier.features.map((feature, i) => (
                    <div key={i} className="flex items-start gap-3">
                      <FaCheck className="text-green-500 mt-1 flex-shrink-0" />
                      <span className="text-gray-300">{feature}</span>
                    </div>
                  ))}

                  {tier.notIncluded.map((feature, i) => (
                    <div key={i} className="flex items-start gap-3 opacity-50">
                      <span className="text-gray-600 mt-1">✕</span>
                      <span className="text-gray-500 line-through">{feature}</span>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* Feature Comparison */}
        <div className="mt-20">
          <h2 className="text-3xl font-bold text-center mb-10">
            Compare All Features
          </h2>
          <FeatureComparison />
        </div>

        {/* FAQ Section */}
        <div className="mt-20 max-w-3xl mx-auto">
          <h2 className="text-3xl font-bold text-center mb-10">
            Frequently Asked Questions
          </h2>

          <div className="space-y-6">
            <div className="card p-6">
              <h3 className="text-xl font-bold mb-2">Can I change plans anytime?</h3>
              <p className="text-gray-400">
                Yes! You can upgrade or downgrade your plan at any time. Changes take effect at the start of your next billing cycle.
              </p>
            </div>

            <div className="card p-6">
              <h3 className="text-xl font-bold mb-2">What payment methods do you accept?</h3>
              <p className="text-gray-400">
                We accept all major credit cards through our secure Stripe payment processor.
              </p>
            </div>

            <div className="card p-6">
              <h3 className="text-xl font-bold mb-2">How does the Creator revenue share work?</h3>
              <p className="text-gray-400">
                Creators keep 80% of all sales from their products. We handle payments, hosting, and customer support, taking a 20% platform fee.
              </p>
            </div>

            <div className="card p-6">
              <h3 className="text-xl font-bold mb-2">Is there a free trial?</h3>
              <p className="text-gray-400">
                The Free plan is available forever with no credit card required. You can try Pro or Creator features by upgrading anytime.
              </p>
            </div>
          </div>
        </div>

        {/* Manage Subscription */}
        {isAuthenticated && user?.subscriptionTier !== 'free' && (
          <div className="mt-12 text-center">
            <button
              onClick={async () => {
                try {
                  const response = await api.post('/stripe/customer-portal');
                  window.location.href = response.data.url;
                } catch (error) {
                  console.error('Portal error:', error);
                }
              }}
              className="btn-secondary"
            >
              Manage Subscription
            </button>
          </div>
        )}
      </div>
    </div>
  );
};

export default Pricing;
