import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';
import { 
  FaStar, 
  FaShoppingCart, 
  FaDownload, 
  FaExternalLinkAlt,
  FaTag,
  FaCode,
  FaUser
} from 'react-icons/fa';

const ProductDetail = () => {
  const { id } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  const [product, setProduct] = useState(null);
  const [hasPurchased, setHasPurchased] = useState(false);
  const [reviews, setReviews] = useState([]);
  const [loading, setLoading] = useState(true);
  const [addingToCart, setAddingToCart] = useState(false);

  useEffect(() => {
    fetchProduct();
    fetchReviews();
  }, [id]);

  const fetchProduct = async () => {
    try {
      const response = await api.get(`/products/${id}`);
      setProduct(response.data.product);
      setHasPurchased(response.data.hasPurchased);
    } catch (error) {
      console.error('Failed to fetch product:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchReviews = async () => {
    try {
      const response = await api.get(`/reviews/product/${id}`);
      setReviews(response.data.reviews);
    } catch (error) {
      console.error('Failed to fetch reviews:', error);
    }
  };

  const handleAddToCart = async () => {
    if (!user) {
      navigate('/signin');
      return;
    }

    try {
      setAddingToCart(true);
      await api.post('/cart/add', { productId: id });
      navigate('/cart');
    } catch (error) {
      console.error('Failed to add to cart:', error);
      alert(error.response?.data?.message || 'Failed to add to cart');
    } finally {
      setAddingToCart(false);
    }
  };

  const handleDownload = async () => {
    try {
      const response = await api.get(`/products/${id}/download`);
      window.open(response.data.downloadUrl, '_blank');
    } catch (error) {
      console.error('Download failed:', error);
      alert(error.response?.data?.message || 'Download failed');
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  if (!product) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold mb-4">Product not found</h2>
          <button onClick={() => navigate('/marketplace')} className="btn-primary">
            Back to Marketplace
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen px-4 py-20">
      <div className="max-w-6xl mx-auto">
        {/* Back Button */}
        <button
          onClick={() => navigate('/marketplace')}
          className="btn-secondary mb-6"
        >
          ← Back to Marketplace
        </button>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Images and Details */}
          <div className="lg:col-span-2 space-y-6">
            {/* Main Preview */}
            <div className="card overflow-hidden">
              {product.previewImages && product.previewImages[0] ? (
                <img
                  src={product.previewImages[0]}
                  alt={product.title}
                  className="w-full h-96 object-cover"
                />
              ) : (
                <div className="w-full h-96 bg-gradient-to-br from-turquoise/20 to-purple/20 flex items-center justify-center">
                  <span className="text-6xl">📦</span>
                </div>
              )}
            </div>

            {/* Additional Images */}
            {product.previewImages && product.previewImages.length > 1 && (
              <div className="grid grid-cols-3 gap-4">
                {product.previewImages.slice(1).map((img, idx) => (
                  <img
                    key={idx}
                    src={img}
                    alt={`Preview ${idx + 2}`}
                    className="w-full h-32 object-cover rounded-lg"
                  />
                ))}
              </div>
            )}

            {/* Description */}
            <div className="card">
              <h2 className="text-2xl font-bold mb-4">Description</h2>
              <p className="text-gray-300 whitespace-pre-wrap">{product.description}</p>
            </div>

            {/* Reviews */}
            <div className="card">
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-bold">Reviews</h2>
                <div className="flex items-center gap-2">
                  <FaStar className="text-yellow-500" />
                  <span className="text-xl font-bold">{product.rating.toFixed(1)}</span>
                  <span className="text-gray-400">({product.reviewCount} reviews)</span>
                </div>
              </div>

              {reviews.length === 0 ? (
                <p className="text-gray-400">No reviews yet. Be the first to review!</p>
              ) : (
                <div className="space-y-4">
                  {reviews.map(review => (
                    <div key={review._id} className="border-b border-gray-700 pb-4">
                      <div className="flex items-center gap-3 mb-2">
                        <img
                          src={review.user?.avatar || '/default-avatar.png'}
                          alt={review.user?.username}
                          className="w-10 h-10 rounded-full"
                        />
                        <div>
                          <div className="font-medium">{review.user?.username}</div>
                          <div className="flex items-center gap-1">
                            {[...Array(5)].map((_, i) => (
                              <FaStar
                                key={i}
                                className={i < review.rating ? 'text-yellow-500' : 'text-gray-600'}
                              />
                            ))}
                          </div>
                        </div>
                      </div>
                      <h4 className="font-medium mb-1">{review.title}</h4>
                      <p className="text-gray-400">{review.comment}</p>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>

          {/* Right Column - Purchase Card */}
          <div className="lg:col-span-1">
            <div className="card sticky top-24 space-y-6">
              {/* Title */}
              <div>
                <span className="inline-block px-3 py-1 text-sm rounded bg-turquoise/20 text-turquoise mb-3">
                  {product.category}
                </span>
                <h1 className="text-3xl font-bold mb-2">{product.title}</h1>
                <p className="text-gray-400">{product.shortDescription}</p>
              </div>

              {/* Creator */}
              <div className="flex items-center gap-3 pb-4 border-b border-gray-700">
                <img
                  src={product.creator?.avatar || '/default-avatar.png'}
                  alt={product.creator?.username}
                  className="w-12 h-12 rounded-full"
                />
                <div>
                  <div className="text-sm text-gray-400">Created by</div>
                  <div className="font-medium">{product.creator?.username}</div>
                </div>
              </div>

              {/* Price */}
              <div className="pb-4 border-b border-gray-700">
                <div className="text-4xl font-bold text-turquoise mb-2">
                  ${product.price.toFixed(2)}
                </div>
                <div className="flex items-center gap-4 text-sm text-gray-400">
                  <span className="flex items-center gap-1">
                    <FaDownload />
                    {product.downloads} downloads
                  </span>
                  <span>v{product.version}</span>
                </div>
              </div>

              {/* Action Buttons */}
              {hasPurchased ? (
                <button
                  onClick={handleDownload}
                  className="w-full btn-primary flex items-center justify-center gap-2"
                >
                  <FaDownload />
                  Download
                </button>
              ) : (
                <button
                  onClick={handleAddToCart}
                  disabled={addingToCart}
                  className="w-full btn-primary flex items-center justify-center gap-2 disabled:opacity-50"
                >
                  <FaShoppingCart />
                  {addingToCart ? 'Adding...' : 'Add to Cart'}
                </button>
              )}

              {product.demoUrl && (
                <a
                  href={product.demoUrl}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-full btn-secondary flex items-center justify-center gap-2"
                >
                  <FaExternalLinkAlt />
                  Live Demo
                </a>
              )}

              {/* Tags */}
              {product.tags && product.tags.length > 0 && (
                <div className="pt-4 border-t border-gray-700">
                  <div className="flex items-center gap-2 mb-2">
                    <FaTag className="text-gray-400" />
                    <span className="font-medium">Tags</span>
                  </div>
                  <div className="flex flex-wrap gap-2">
                    {product.tags.map((tag, idx) => (
                      <span
                        key={idx}
                        className="px-2 py-1 text-xs rounded bg-gray-800 text-gray-300"
                      >
                        {tag}
                      </span>
                    ))}
                  </div>
                </div>
              )}

              {/* Tech Stack */}
              {product.techStack && product.techStack.length > 0 && (
                <div className="pt-4 border-t border-gray-700">
                  <div className="flex items-center gap-2 mb-2">
                    <FaCode className="text-gray-400" />
                    <span className="font-medium">Tech Stack</span>
                  </div>
                  <div className="flex flex-wrap gap-2">
                    {product.techStack.map((tech, idx) => (
                      <span
                        key={idx}
                        className="px-2 py-1 text-xs rounded bg-purple/20 text-purple"
                      >
                        {tech}
                      </span>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ProductDetail;
