import { useState, useEffect } from 'react';
import { FaUsers, FaDollarSign, FaChartLine, FaCopy, FaCheck, FaExclamationTriangle, FaPaypal, FaCreditCard, FaUniversity, FaCoins } from 'react-icons/fa';
import api from '../services/api';

const ReferralDashboard = () => {
  const [stats, setStats] = useState(null);
  const [recentReferrals, setRecentReferrals] = useState([]);
  const [byType, setByType] = useState({});
  const [payoutInfo, setPayoutInfo] = useState({});
  const [loading, setLoading] = useState(true);
  const [copied, setCopied] = useState(false);
  const [showPayoutSettings, setShowPayoutSettings] = useState(false);
  const [payoutSettings, setPayoutSettings] = useState({
    payoutMethod: 'platform_credit',
    payoutEmail: '',
    minimumPayout: 50
  });

  useEffect(() => {
    fetchDashboard();
  }, []);

  const fetchDashboard = async () => {
    try {
      const response = await api.get('/referrals/dashboard');
      setStats(response.data.stats);
      setRecentReferrals(response.data.recentReferrals);
      setByType(response.data.byType);
      setPayoutInfo(response.data.payoutInfo);
      setPayoutSettings(response.data.payoutInfo);
      setLoading(false);
    } catch (error) {
      console.error('Fetch referral dashboard error:', error);
      setLoading(false);
    }
  };

  const copyReferralLink = () => {
    navigator.clipboard.writeText(stats.referralLink);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  const requestPayout = async () => {
    try {
      const response = await api.post('/referrals/payout/request');
      alert(response.data.message);
      fetchDashboard();
    } catch (error) {
      alert(error.response?.data?.message || 'Failed to request payout');
    }
  };

  const updatePayoutSettings = async () => {
    try {
      await api.put('/referrals/payout/settings', payoutSettings);
      alert('Payout settings updated!');
      setShowPayoutSettings(false);
      fetchDashboard();
    } catch (error) {
      alert('Failed to update settings');
    }
  };

  const getPayoutMethodIcon = (method) => {
    switch (method) {
      case 'paypal': return <FaPaypal />;
      case 'stripe': return <FaCreditCard />;
      case 'bank_transfer': return <FaUniversity />;
      case 'platform_credit': return <FaCoins />;
      default: return <FaDollarSign />;
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">Referral Dashboard</h1>
          <p className="text-gray-400">Earn rewards by inviting friends to Vibe Coding Bros</p>
        </div>

        {/* Stats Cards */}
        <div className="grid md:grid-cols-4 gap-6 mb-8">
          <div className="card bg-gradient-to-br from-turquoise/10 to-transparent border-turquoise/30">
            <div className="flex items-center justify-between mb-2">
              <FaUsers className="text-turquoise text-3xl" />
              <span className="text-3xl font-bold">{stats?.totalReferrals || 0}</span>
            </div>
            <div className="text-sm text-gray-400">Total Referrals</div>
          </div>

          <div className="card bg-gradient-to-br from-green-500/10 to-transparent border-green-500/30">
            <div className="flex items-center justify-between mb-2">
              <FaCheck className="text-green-400 text-3xl" />
              <span className="text-3xl font-bold">{stats?.successfulReferrals || 0}</span>
            </div>
            <div className="text-sm text-gray-400">Successful</div>
          </div>

          <div className="card bg-gradient-to-br from-gold/10 to-transparent border-gold/30">
            <div className="flex items-center justify-between mb-2">
              <FaDollarSign className="text-gold text-3xl" />
              <span className="text-3xl font-bold">${stats?.pendingEarnings?.toFixed(2) || '0.00'}</span>
            </div>
            <div className="text-sm text-gray-400">Pending</div>
          </div>

          <div className="card bg-gradient-to-br from-purple/10 to-transparent border-purple/30">
            <div className="flex items-center justify-between mb-2">
              <FaChartLine className="text-purple text-3xl" />
              <span className="text-3xl font-bold">${stats?.totalEarned?.toFixed(2) || '0.00'}</span>
            </div>
            <div className="text-sm text-gray-400">Total Earned</div>
          </div>
        </div>

        {/* Referral Link */}
        <div className="card mb-8">
          <h2 className="text-2xl font-bold mb-4">Your Referral Link</h2>
          <div className="flex gap-3 items-center">
            <div className="flex-1 bg-gray-800 p-4 rounded-lg font-mono text-sm break-all">
              {stats?.referralLink}
            </div>
            <button
              onClick={copyReferralLink}
              className="btn-primary flex items-center gap-2 whitespace-nowrap"
            >
              {copied ? <><FaCheck /> Copied!</> : <><FaCopy /> Copy</>}
            </button>
          </div>
          <div className="mt-4 p-4 bg-turquoise/10 border border-turquoise/30 rounded-lg">
            <h3 className="font-semibold mb-2">Referral Code: <code className="bg-turquoise/20 px-3 py-1 rounded text-turquoise">{stats?.referralCode}</code></h3>
            <p className="text-sm text-gray-400">Share this link or code with friends. You'll earn rewards when they sign up and make purchases!</p>
          </div>
        </div>

        <div className="grid md:grid-cols-2 gap-8 mb-8">
          {/* Earnings */}
          <div className="card">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-bold">Earnings Breakdown</h2>
              <button
                onClick={() => setShowPayoutSettings(!showPayoutSettings)}
                className="btn-secondary text-sm"
              >
                Settings
              </button>
            </div>

            {showPayoutSettings && (
              <div className="mb-6 p-4 bg-gray-800 rounded-lg space-y-4">
                <div>
                  <label className="block text-sm font-medium mb-2">Payout Method</label>
                  <select
                    value={payoutSettings.payoutMethod}
                    onChange={(e) => setPayoutSettings({ ...payoutSettings, payoutMethod: e.target.value })}
                    className="w-full bg-gray-700 rounded-lg px-4 py-2"
                  >
                    <option value="platform_credit">Platform Credit</option>
                    <option value="paypal">PayPal</option>
                    <option value="stripe">Stripe</option>
                    <option value="bank_transfer">Bank Transfer</option>
                  </select>
                </div>

                {payoutSettings.payoutMethod !== 'platform_credit' && (
                  <div>
                    <label className="block text-sm font-medium mb-2">Payout Email</label>
                    <input
                      type="email"
                      value={payoutSettings.payoutEmail}
                      onChange={(e) => setPayoutSettings({ ...payoutSettings, payoutEmail: e.target.value })}
                      className="w-full bg-gray-700 rounded-lg px-4 py-2"
                      placeholder="email@example.com"
                    />
                  </div>
                )}

                <div>
                  <label className="block text-sm font-medium mb-2">Minimum Payout Amount</label>
                  <input
                    type="number"
                    value={payoutSettings.minimumPayout}
                    onChange={(e) => setPayoutSettings({ ...payoutSettings, minimumPayout: Number(e.target.value) })}
                    className="w-full bg-gray-700 rounded-lg px-4 py-2"
                    min="10"
                    step="10"
                  />
                </div>

                <button onClick={updatePayoutSettings} className="btn-primary w-full">
                  Save Settings
                </button>
              </div>
            )}

            <div className="space-y-4">
              <div className="flex justify-between items-center p-4 bg-gray-800 rounded-lg">
                <div>
                  <div className="text-sm text-gray-400">Pending Earnings</div>
                  <div className="text-2xl font-bold text-gold">${stats?.pendingEarnings?.toFixed(2) || '0.00'}</div>
                </div>
                <button
                  onClick={requestPayout}
                  disabled={stats?.pendingEarnings < payoutInfo.minimumPayout}
                  className="btn-primary disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Request Payout
                </button>
              </div>

              <div className="flex justify-between p-4 bg-gray-800 rounded-lg">
                <span className="text-gray-400">Paid Out</span>
                <span className="font-semibold text-green-400">${stats?.paidEarnings?.toFixed(2) || '0.00'}</span>
              </div>

              <div className="flex justify-between p-4 bg-gray-800 rounded-lg">
                <span className="text-gray-400">Total Earned</span>
                <span className="font-semibold text-turquoise">${stats?.totalEarned?.toFixed(2) || '0.00'}</span>
              </div>

              <div className="flex items-center gap-2 p-3 bg-blue-500/10 border border-blue-500/30 rounded-lg text-sm">
                {getPayoutMethodIcon(payoutInfo.method)}
                <span>Method: <strong>{payoutInfo.method?.replace('_', ' ').toUpperCase()}</strong></span>
              </div>

              {stats?.pendingEarnings < payoutInfo.minimumPayout && (
                <div className="flex items-start gap-2 p-3 bg-yellow-500/10 border border-yellow-500/30 rounded-lg text-sm">
                  <FaExclamationTriangle className="text-yellow-400 mt-0.5" />
                  <span>Minimum payout is ${payoutInfo.minimumPayout}. You need ${(payoutInfo.minimumPayout - stats.pendingEarnings).toFixed(2)} more.</span>
                </div>
              )}
            </div>
          </div>

          {/* By Type */}
          <div className="card">
            <h2 className="text-2xl font-bold mb-6">Referrals by Type</h2>
            <div className="space-y-3">
              {Object.entries(byType).map(([type, count]) => (
                <div key={type} className="flex justify-between items-center p-4 bg-gray-800 rounded-lg">
                  <span className="capitalize">{type.replace('_', ' ')}</span>
                  <span className="font-bold text-turquoise">{count}</span>
                </div>
              ))}
              {Object.keys(byType).length === 0 && (
                <div className="text-center text-gray-400 py-8">
                  No referrals yet. Share your link to get started!
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Recent Referrals */}
        <div className="card">
          <h2 className="text-2xl font-bold mb-6">Recent Referrals</h2>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="border-b border-gray-700">
                <tr className="text-left">
                  <th className="pb-3 font-semibold">User</th>
                  <th className="pb-3 font-semibold">Type</th>
                  <th className="pb-3 font-semibold">Reward</th>
                  <th className="pb-3 font-semibold">Status</th>
                  <th className="pb-3 font-semibold">Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-800">
                {recentReferrals.map((ref) => (
                  <tr key={ref.id}>
                    <td className="py-3">{ref.username}</td>
                    <td className="py-3 capitalize">{ref.conversionType.replace('_', ' ')}</td>
                    <td className="py-3 text-gold font-semibold">${ref.reward}</td>
                    <td className="py-3">
                      <span className={`px-3 py-1 rounded-full text-xs ${
                        ref.status === 'completed' ? 'bg-green-500/20 text-green-400' :
                        ref.status === 'paid' ? 'bg-blue-500/20 text-blue-400' :
                        ref.status === 'fraud' ? 'bg-red-500/20 text-red-400' :
                        'bg-yellow-500/20 text-yellow-400'
                      }`}>
                        {ref.status}
                      </span>
                      {ref.fraudScore > 30 && (
                        <span className="ml-2 text-xs text-orange-400">⚠️ Score: {ref.fraudScore}</span>
                      )}
                    </td>
                    <td className="py-3 text-gray-400 text-sm">
                      {new Date(ref.createdAt).toLocaleDateString()}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
            {recentReferrals.length === 0 && (
              <div className="text-center text-gray-400 py-12">
                No referrals yet. Start sharing your link!
              </div>
            )}
          </div>
        </div>

        {/* How It Works */}
        <div className="card mt-8 bg-gradient-to-br from-turquoise/5 to-transparent border-turquoise/20">
          <h2 className="text-2xl font-bold mb-4">How Referrals Work</h2>
          <div className="grid md:grid-cols-2 gap-6">
            <div>
              <h3 className="font-semibold mb-2 flex items-center gap-2">
                <span className="bg-turquoise text-vibe-black w-6 h-6 rounded-full flex items-center justify-center text-sm">1</span>
                Share Your Link
              </h3>
              <p className="text-sm text-gray-400">Send your unique referral link to friends via email, social media, or messaging apps.</p>
            </div>
            <div>
              <h3 className="font-semibold mb-2 flex items-center gap-2">
                <span className="bg-turquoise text-vibe-black w-6 h-6 rounded-full flex items-center justify-center text-sm">2</span>
                They Sign Up
              </h3>
              <p className="text-sm text-gray-400">When someone signs up using your link, both of you earn $5!</p>
            </div>
            <div>
              <h3 className="font-semibold mb-2 flex items-center gap-2">
                <span className="bg-turquoise text-vibe-black w-6 h-6 rounded-full flex items-center justify-center text-sm">3</span>
                Earn More Rewards
              </h3>
              <p className="text-sm text-gray-400">Get additional rewards when they make purchases ($10), subscribe ($20), or complete challenges ($15).</p>
            </div>
            <div>
              <h3 className="font-semibold mb-2 flex items-center gap-2">
                <span className="bg-turquoise text-vibe-black w-6 h-6 rounded-full flex items-center justify-center text-sm">4</span>
                Get Paid
              </h3>
              <p className="text-sm text-gray-400">Request payout once you reach ${payoutInfo.minimumPayout}. Choose from PayPal, Stripe, or platform credit.</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ReferralDashboard;
