import { Link } from 'react-router-dom';
import { FaDollarSign, FaUsers, FaRocket, FaChartLine, FaCheckCircle, FaGift, FaCopy } from 'react-icons/fa';
import { motion } from 'framer-motion';
import { useState } from 'react';
import { useAuth } from '../context/AuthContext';

const Referrals = () => {
  const { isAuthenticated } = useAuth();
  const [copied, setCopied] = useState(false);
  const exampleLink = 'https://vibecoding.com/signup?ref=YOURCODE';

  const copyLink = () => {
    navigator.clipboard.writeText(exampleLink);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  const rewards = [
    {
      icon: <FaGift className="text-5xl text-turquoise" />,
      title: 'Sign Up Bonus',
      amount: '$5 + $5',
      description: 'Both you and your friend get $5 when they sign up'
    },
    {
      icon: <FaDollarSign className="text-5xl text-gold" />,
      title: 'First Purchase',
      amount: '$10',
      description: 'Earn $10 when your referral makes their first purchase'
    },
    {
      icon: <FaRocket className="text-5xl text-purple" />,
      title: 'Subscription',
      amount: '$20',
      description: 'Get $20 when they subscribe to any paid plan'
    },
    {
      icon: <FaChartLine className="text-5xl text-green-400" />,
      title: 'Challenge Completion',
      amount: '$15',
      description: 'Earn $15 when they complete their first challenge'
    }
  ];

  const stats = [
    { number: '10K+', label: 'Active Members' },
    { number: '$500K+', label: 'Paid to Referrers' },
    { number: '$250', label: 'Average Earnings' },
    { number: '30 Days', label: 'Cookie Duration' }
  ];

  const benefits = [
    'No limit on referrals - invite unlimited friends',
    'Multiple payout options: PayPal, Stripe, or platform credit',
    'Real-time tracking dashboard',
    'Automatic fraud protection',
    'Minimum payout just $50',
    'Lifetime commissions on referrals'
  ];

  const howItWorks = [
    {
      step: '1',
      title: 'Get Your Link',
      description: 'Sign up and instantly get your unique referral link'
    },
    {
      step: '2',
      title: 'Share Everywhere',
      description: 'Share on social media, email, or embed on your website'
    },
    {
      step: '3',
      title: 'Earn Rewards',
      description: 'Get paid when friends sign up and engage with the platform'
    },
    {
      step: '4',
      title: 'Cash Out',
      description: 'Request payout once you hit $50 via your preferred method'
    }
  ];

  return (
    <div className="min-h-screen pt-24 pb-12">
      {/* Hero Section */}
      <section className="relative overflow-hidden mb-20">
        <div className="absolute inset-0 bg-gradient-to-br from-turquoise/10 via-vibe-black to-gold/10"></div>
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 text-center">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
          >
            <h1 className="text-5xl md:text-7xl font-bold mb-6">
              Earn Money by <span className="text-gradient">Sharing</span>
            </h1>
            <p className="text-xl md:text-2xl text-gray-300 mb-8 max-w-3xl mx-auto">
              Get paid for inviting developers to join Vibe Coding Bros. Earn up to $20 per referral with our generous reward program.
            </p>
            
            {isAuthenticated ? (
              <Link to="/referrals/dashboard" className="btn-primary text-lg inline-flex items-center gap-2">
                <FaChartLine /> View Your Dashboard
              </Link>
            ) : (
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Link to="/signup" className="btn-primary text-lg">
                  Start Earning Now
                </Link>
                <Link to="/signin" className="btn-outline text-lg">
                  Sign In to Dashboard
                </Link>
              </div>
            )}

            {/* Example Link Preview */}
            <div className="mt-12 max-w-2xl mx-auto">
              <div className="bg-gray-800/50 border border-turquoise/30 rounded-lg p-6">
                <div className="text-sm text-gray-400 mb-2">Your referral link will look like this:</div>
                <div className="flex items-center gap-3">
                  <code className="flex-1 bg-gray-900 px-4 py-3 rounded text-turquoise text-sm break-all">
                    {exampleLink}
                  </code>
                  <button
                    onClick={copyLink}
                    className="btn-secondary flex items-center gap-2 whitespace-nowrap"
                  >
                    {copied ? <><FaCheckCircle /> Copied!</> : <><FaCopy /> Copy</>}
                  </button>
                </div>
              </div>
            </div>
          </motion.div>
        </div>
      </section>

      {/* Stats Section */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mb-20">
        <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              viewport={{ once: true }}
              className="card text-center"
            >
              <div className="text-4xl font-bold text-turquoise mb-2">{stat.number}</div>
              <div className="text-sm text-gray-400">{stat.label}</div>
            </motion.div>
          ))}
        </div>
      </section>

      {/* Rewards Section */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mb-20">
        <motion.div
          initial={{ opacity: 0 }}
          whileInView={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
          viewport={{ once: true }}
          className="text-center mb-12"
        >
          <h2 className="text-4xl md:text-5xl font-bold mb-4">
            Earn <span className="text-gradient">Multiple Ways</span>
          </h2>
          <p className="text-xl text-gray-400">
            Get rewarded for every action your referrals take
          </p>
        </motion.div>

        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
          {rewards.map((reward, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              viewport={{ once: true }}
              className="card text-center hover:scale-105 transition-transform"
            >
              <div className="mb-4 flex justify-center">{reward.icon}</div>
              <h3 className="text-xl font-bold mb-2">{reward.title}</h3>
              <div className="text-3xl font-bold text-gold mb-3">{reward.amount}</div>
              <p className="text-sm text-gray-400">{reward.description}</p>
            </motion.div>
          ))}
        </div>
      </section>

      {/* How It Works */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mb-20">
        <motion.div
          initial={{ opacity: 0 }}
          whileInView={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
          viewport={{ once: true }}
          className="text-center mb-12"
        >
          <h2 className="text-4xl md:text-5xl font-bold mb-4">
            How It <span className="text-gradient">Works</span>
          </h2>
          <p className="text-xl text-gray-400">
            Start earning in 4 simple steps
          </p>
        </motion.div>

        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
          {howItWorks.map((item, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              viewport={{ once: true }}
              className="card text-center"
            >
              <div className="w-16 h-16 bg-gradient-to-br from-turquoise to-gold rounded-full flex items-center justify-center text-3xl font-bold text-vibe-black mx-auto mb-4">
                {item.step}
              </div>
              <h3 className="text-xl font-bold mb-2">{item.title}</h3>
              <p className="text-sm text-gray-400">{item.description}</p>
            </motion.div>
          ))}
        </div>
      </section>

      {/* Benefits Section */}
      <section className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mb-20">
        <div className="card bg-gradient-to-br from-turquoise/5 to-gold/5 border-turquoise/20">
          <div className="text-center mb-8">
            <h2 className="text-3xl md:text-4xl font-bold mb-4">
              Why Our <span className="text-gradient">Referral Program</span> Rocks
            </h2>
          </div>
          <div className="grid md:grid-cols-2 gap-4">
            {benefits.map((benefit, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, x: -20 }}
                whileInView={{ opacity: 1, x: 0 }}
                transition={{ duration: 0.5, delay: index * 0.05 }}
                viewport={{ once: true }}
                className="flex items-start gap-3"
              >
                <FaCheckCircle className="text-turquoise text-xl flex-shrink-0 mt-1" />
                <span className="text-gray-300">{benefit}</span>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          whileInView={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.8 }}
          viewport={{ once: true }}
          className="card bg-gradient-to-br from-turquoise/10 to-gold/10 border-turquoise/30 text-center"
        >
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Start Earning?
          </h2>
          <p className="text-xl text-gray-400 mb-8">
            Join thousands of developers already earning passive income
          </p>
          {isAuthenticated ? (
            <Link to="/referrals/dashboard" className="btn-primary text-lg inline-flex items-center gap-2">
              <FaChartLine /> Go to Dashboard
            </Link>
          ) : (
            <Link to="/signup" className="btn-primary text-lg inline-flex items-center gap-2">
              <FaUsers /> Create Free Account
            </Link>
          )}
        </motion.div>
      </section>
    </div>
  );
};

export default Referrals;
