import React, { useState, useEffect } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { FiSearch, FiFilter, FiX } from 'react-icons/fi';
import api from '../services/api';

const Search = () => {
  const navigate = useNavigate();
  const [searchParams, setSearchParams] = useSearchParams();
  
  const [searchQuery, setSearchQuery] = useState(searchParams.get('q') || '');
  const [results, setResults] = useState(null);
  const [loading, setLoading] = useState(false);
  const [showFilters, setShowFilters] = useState(false);
  const [filterOptions, setFilterOptions] = useState(null);
  
  const [filters, setFilters] = useState({
    type: searchParams.get('type') || 'all',
    category: searchParams.get('category') || '',
    difficulty: searchParams.get('difficulty') || '',
    level: searchParams.get('level') || '',
    jobType: searchParams.get('jobType') || '',
    sortBy: searchParams.get('sortBy') || 'date',
    order: searchParams.get('order') || 'desc',
  });

  useEffect(() => {
    fetchFilterOptions();
  }, []);

  useEffect(() => {
    const query = searchParams.get('q');
    if (query) {
      performSearch(query);
    }
  }, [searchParams]);

  const fetchFilterOptions = async () => {
    try {
      const response = await api.get('/search/filters');
      setFilterOptions(response.data.data);
    } catch (error) {
      console.error('Error fetching filter options:', error);
    }
  };

  const performSearch = async (query) => {
    if (!query.trim()) return;
    
    setLoading(true);
    try {
      const params = new URLSearchParams({
        query,
        ...filters,
      });
      
      const response = await api.get(`/search?${params.toString()}`);
      setResults(response.data.data);
    } catch (error) {
      console.error('Search error:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = (e) => {
    e.preventDefault();
    if (!searchQuery.trim()) return;
    
    const params = new URLSearchParams({
      q: searchQuery,
      ...filters,
    });
    
    setSearchParams(params);
  };

  const handleFilterChange = (key, value) => {
    const newFilters = { ...filters, [key]: value };
    setFilters(newFilters);
    
    if (searchQuery) {
      const params = new URLSearchParams({
        q: searchQuery,
        ...newFilters,
      });
      setSearchParams(params);
    }
  };

  const clearFilters = () => {
    setFilters({
      type: 'all',
      category: '',
      difficulty: '',
      level: '',
      jobType: '',
      sortBy: 'date',
      order: 'desc',
    });
  };

  const renderResultCard = (item, type) => {
    return (
      <div key={item._id} className="bg-gray-800 rounded-lg p-6 hover:bg-gray-750 transition-colors">
        <div className="flex items-start justify-between mb-2">
          <h3 className="text-xl font-semibold text-white">
            {item.title || item.name}
          </h3>
          <span className="px-3 py-1 bg-light-sea-green/20 text-light-sea-green rounded-full text-sm">
            {type}
          </span>
        </div>
        
        <p className="text-gray-400 mb-4 line-clamp-2">
          {item.description || item.content}
        </p>
        
        <div className="flex items-center justify-between text-sm">
          {type === 'challenge' && (
            <>
              <span className={`px-2 py-1 rounded ${
                item.difficulty === 'Easy' ? 'bg-green-500/20 text-green-400' :
                item.difficulty === 'Medium' ? 'bg-yellow-500/20 text-yellow-400' :
                'bg-red-500/20 text-red-400'
              }`}>
                {item.difficulty}
              </span>
              <span className="text-gray-500">{item.participants} participants</span>
            </>
          )}
          
          {type === 'job' && (
            <>
              <span className="text-gray-400">{item.company}</span>
              <span className="text-light-sea-green">{item.location}</span>
            </>
          )}
          
          {(type === 'workflow' || type === 'app') && (
            <>
              <span className="text-gray-400">{item.category}</span>
              <span className="text-light-sea-green">↑ {item.votes}</span>
            </>
          )}
        </div>
      </div>
    );
  };

  return (
    <div className="min-h-screen bg-dark-bg text-white py-8 px-4">
      <div className="max-w-6xl mx-auto">
        {/* Search Header */}
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-6 text-center">Search</h1>
          
          {/* Search Bar */}
          <form onSubmit={handleSearch} className="mb-6">
            <div className="relative">
              <FiSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 text-xl" />
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search challenges, jobs, workflows, apps..."
                className="w-full bg-gray-800 text-white pl-12 pr-4 py-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
              />
              <button
                type="submit"
                className="absolute right-2 top-1/2 transform -translate-y-1/2 bg-light-sea-green text-white px-6 py-2 rounded-lg hover:bg-opacity-90 transition-colors"
              >
                Search
              </button>
            </div>
          </form>

          {/* Filter Toggle */}
          <div className="flex items-center justify-between">
            <button
              onClick={() => setShowFilters(!showFilters)}
              className="flex items-center gap-2 text-light-sea-green hover:underline"
            >
              <FiFilter />
              {showFilters ? 'Hide Filters' : 'Show Filters'}
            </button>
            
            {results && (
              <span className="text-gray-400">
                {results.total} results found
              </span>
            )}
          </div>
        </div>

        {/* Filters Panel */}
        {showFilters && filterOptions && (
          <div className="bg-gray-800 rounded-lg p-6 mb-8">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-semibold">Filters</h3>
              <button
                onClick={clearFilters}
                className="text-sm text-gray-400 hover:text-white flex items-center gap-1"
              >
                <FiX /> Clear All
              </button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {/* Content Type */}
              <div>
                <label className="block text-sm font-medium mb-2">Type</label>
                <select
                  value={filters.type}
                  onChange={(e) => handleFilterChange('type', e.target.value)}
                  className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                >
                  <option value="all">All</option>
                  <option value="challenges">Challenges</option>
                  <option value="jobs">Jobs</option>
                  <option value="workflows">Workflows</option>
                  <option value="apps">Apps</option>
                </select>
              </div>

              {/* Challenge Difficulty */}
              {(filters.type === 'all' || filters.type === 'challenges') && (
                <div>
                  <label className="block text-sm font-medium mb-2">Difficulty</label>
                  <select
                    value={filters.difficulty}
                    onChange={(e) => handleFilterChange('difficulty', e.target.value)}
                    className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                  >
                    <option value="">All Levels</option>
                    {filterOptions.challenges.difficulties.map(diff => (
                      <option key={diff} value={diff}>{diff}</option>
                    ))}
                  </select>
                </div>
              )}

              {/* Job Level */}
              {(filters.type === 'all' || filters.type === 'jobs') && (
                <>
                  <div>
                    <label className="block text-sm font-medium mb-2">Job Level</label>
                    <select
                      value={filters.level}
                      onChange={(e) => handleFilterChange('level', e.target.value)}
                      className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                    >
                      <option value="">All Levels</option>
                      {filterOptions.jobs.levels.map(level => (
                        <option key={level} value={level}>{level.charAt(0).toUpperCase() + level.slice(1)}</option>
                      ))}
                    </select>
                  </div>
                  
                  <div>
                    <label className="block text-sm font-medium mb-2">Job Type</label>
                    <select
                      value={filters.jobType}
                      onChange={(e) => handleFilterChange('jobType', e.target.value)}
                      className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                    >
                      <option value="">All Types</option>
                      {filterOptions.jobs.types.map(type => (
                        <option key={type} value={type}>{type.charAt(0).toUpperCase() + type.slice(1)}</option>
                      ))}
                    </select>
                  </div>
                </>
              )}

              {/* Category */}
              {(filters.type === 'workflows' || filters.type === 'apps') && (
                <div>
                  <label className="block text-sm font-medium mb-2">Category</label>
                  <select
                    value={filters.category}
                    onChange={(e) => handleFilterChange('category', e.target.value)}
                    className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                  >
                    <option value="">All Categories</option>
                    {(filters.type === 'workflows' ? filterOptions.workflows.categories : filterOptions.apps.categories).map(cat => (
                      <option key={cat} value={cat}>{cat.charAt(0).toUpperCase() + cat.slice(1)}</option>
                    ))}
                  </select>
                </div>
              )}

              {/* Sort By */}
              <div>
                <label className="block text-sm font-medium mb-2">Sort By</label>
                <select
                  value={filters.sortBy}
                  onChange={(e) => handleFilterChange('sortBy', e.target.value)}
                  className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                >
                  {filterOptions.sortOptions.map(option => (
                    <option key={option.value} value={option.value}>{option.label}</option>
                  ))}
                </select>
              </div>

              {/* Order */}
              <div>
                <label className="block text-sm font-medium mb-2">Order</label>
                <select
                  value={filters.order}
                  onChange={(e) => handleFilterChange('order', e.target.value)}
                  className="w-full bg-gray-700 text-white px-4 py-2 rounded-lg focus:outline-none focus:ring-2 focus:ring-light-sea-green"
                >
                  {filterOptions.orderOptions.map(option => (
                    <option key={option.value} value={option.value}>{option.label}</option>
                  ))}
                </select>
              </div>
            </div>
          </div>
        )}

        {/* Loading State */}
        {loading && (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-light-sea-green mx-auto"></div>
            <p className="text-gray-400 mt-4">Searching...</p>
          </div>
        )}

        {/* Results */}
        {!loading && results && (
          <div className="space-y-8">
            {/* Challenges */}
            {results.challenges.length > 0 && (
              <div>
                <h2 className="text-2xl font-bold mb-4">Challenges ({results.challenges.length})</h2>
                <div className="grid gap-4">
                  {results.challenges.map(challenge => renderResultCard(challenge, 'challenge'))}
                </div>
              </div>
            )}

            {/* Jobs */}
            {results.jobs.length > 0 && (
              <div>
                <h2 className="text-2xl font-bold mb-4">Jobs ({results.jobs.length})</h2>
                <div className="grid gap-4">
                  {results.jobs.map(job => renderResultCard(job, 'job'))}
                </div>
              </div>
            )}

            {/* Workflows */}
            {results.workflows.length > 0 && (
              <div>
                <h2 className="text-2xl font-bold mb-4">Workflows ({results.workflows.length})</h2>
                <div className="grid gap-4">
                  {results.workflows.map(workflow => renderResultCard(workflow, 'workflow'))}
                </div>
              </div>
            )}

            {/* Apps */}
            {results.apps.length > 0 && (
              <div>
                <h2 className="text-2xl font-bold mb-4">Apps ({results.apps.length})</h2>
                <div className="grid gap-4">
                  {results.apps.map(app => renderResultCard(app, 'app'))}
                </div>
              </div>
            )}

            {/* No Results */}
            {results.total === 0 && (
              <div className="text-center py-12">
                <p className="text-gray-400 text-lg">No results found for "{searchQuery}"</p>
                <p className="text-gray-500 mt-2">Try adjusting your search or filters</p>
              </div>
            )}
          </div>
        )}

        {/* Empty State */}
        {!loading && !results && (
          <div className="text-center py-12">
            <FiSearch className="text-6xl text-gray-600 mx-auto mb-4" />
            <p className="text-gray-400 text-lg">Start searching to find content</p>
          </div>
        )}
      </div>
    </div>
  );
};

export default Search;
