import { useState, useEffect } from 'react';
import { FaCalendar, FaMapMarkerAlt, FaUsers, FaTicketAlt } from 'react-icons/fa';
import api from '../services/api';

const Summit = () => {
  const [events, setEvents] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchEvents();
  }, []);

  const fetchEvents = async () => {
    try {
      const response = await api.get('/events');
      setEvents(response.data.events || []);
    } catch (error) {
      console.error('Error fetching events:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen pt-24 pb-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Hero Section */}
        <div className="text-center mb-16">
          <h1 className="text-5xl md:text-6xl font-bold mb-4">
            Vibe Coding Bros© <span className="text-gradient">Summit</span>
          </h1>
          <p className="text-xl text-gray-400 max-w-3xl mx-auto">
            Join us for exclusive events, workshops, and networking opportunities with top developers and industry leaders
          </p>
        </div>

        {/* Upcoming Event Highlight */}
        <div className="glass-effect rounded-2xl p-8 mb-12">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8 items-center">
            <div>
              <span className="text-sm bg-gold/20 text-gold px-3 py-1 rounded-full">
                UPCOMING EVENT
              </span>
              <h2 className="text-3xl font-bold mt-4 mb-4">
                Annual Summit 2026
              </h2>
              <p className="text-gray-400 mb-6">
                Join us for our biggest event of the year! Three days of workshops, networking, and inspiration with the best developers in the community.
              </p>
              <div className="space-y-3 mb-6">
                <div className="flex items-center gap-3 text-gray-300">
                  <FaCalendar className="text-turquoise" />
                  <span>December 15-17, 2024</span>
                </div>
                <div className="flex items-center gap-3 text-gray-300">
                  <FaMapMarkerAlt className="text-turquoise" />
                  <span>Las Vegas, NV + Virtual</span>
                </div>
                <div className="flex items-center gap-3 text-gray-300">
                  <FaUsers className="text-turquoise" />
                  <span>500+ Expected Attendees</span>
                </div>
              </div>
              <button className="btn-primary">
                Register Now
              </button>
            </div>
            <div className="relative h-64 md:h-96 rounded-lg overflow-hidden">
              <div className="absolute inset-0 bg-gradient-to-br from-turquoise/20 to-gold/20 flex items-center justify-center">
                <FaTicketAlt className="text-8xl text-turquoise/30" />
              </div>
            </div>
          </div>
        </div>

        {/* Past Events */}
        <div>
          <h2 className="text-3xl font-bold mb-8">Past Events</h2>
          {loading ? (
            <div className="text-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise mx-auto"></div>
            </div>
          ) : events.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-400">No past events to display.</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {events.map((event) => (
                <div key={event._id} className="card">
                  <div className="mb-4">
                    <span className="text-xs bg-gray-700 text-gray-400 px-2 py-1 rounded">
                      PAST EVENT
                    </span>
                  </div>
                  <h3 className="text-xl font-bold mb-2">{event.name}</h3>
                  <p className="text-gray-400 mb-4 line-clamp-3">{event.description}</p>
                  <div className="space-y-2">
                    <div className="flex items-center gap-2 text-sm text-gray-400">
                      <FaCalendar className="text-turquoise" />
                      <span>{new Date(event.date).toLocaleDateString()}</span>
                    </div>
                    <div className="flex items-center gap-2 text-sm text-gray-400">
                      <FaUsers className="text-turquoise" />
                      <span>{event.attendees} attendees</span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* Newsletter Section */}
        <div className="glass-effect rounded-2xl p-8 mt-16 text-center">
          <h2 className="text-3xl font-bold mb-4">Stay Updated</h2>
          <p className="text-gray-400 mb-6">
            Get notified about upcoming events and summit announcements
          </p>
          <div className="flex flex-col sm:flex-row gap-4 max-w-md mx-auto">
            <input
              type="email"
              placeholder="Enter your email"
              className="input-field flex-1"
            />
            <button className="btn-primary">Subscribe</button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Summit;
