import { useEffect, useState } from 'react';
import { useParams, useNavigate, Link } from 'react-router-dom';
import { FaCheckCircle, FaTimesCircle, FaSpinner } from 'react-icons/fa';
import api from '../services/api';
import { useAuth } from '../context/AuthContext';

const VerifyEmail = () => {
  const { token } = useParams();
  const navigate = useNavigate();
  const { login } = useAuth();
  const [status, setStatus] = useState('verifying'); // verifying, success, error
  const [message, setMessage] = useState('');

  useEffect(() => {
    const verifyEmail = async () => {
      try {
        const response = await api.get(`/auth/verify-email/${token}`);
        
        // Auto-login user after verification
        if (response.data.token) {
          localStorage.setItem('token', response.data.token);
          localStorage.setItem('user', JSON.stringify(response.data.user));
        }

        setStatus('success');
        setMessage(response.data.message || 'Email verified successfully!');

        // Redirect to dashboard after 3 seconds
        setTimeout(() => {
          navigate('/dashboard');
        }, 3000);
      } catch (error) {
        setStatus('error');
        setMessage(
          error.response?.data?.message || 
          'Verification failed. The link may be invalid or expired.'
        );
      }
    };

    if (token) {
      verifyEmail();
    }
  }, [token, navigate, login]);

  return (
    <div className="min-h-screen flex items-center justify-center px-4 py-20">
      <div className="max-w-md w-full">
        <div className="card text-center">
          {status === 'verifying' && (
            <>
              <FaSpinner className="text-turquoise text-6xl mx-auto mb-6 animate-spin" />
              <h1 className="text-3xl font-bold mb-3">Verifying Your Email...</h1>
              <p className="text-gray-400">Please wait while we verify your email address.</p>
            </>
          )}

          {status === 'success' && (
            <>
              <FaCheckCircle className="text-green-500 text-6xl mx-auto mb-6" />
              <h1 className="text-3xl font-bold mb-3 text-green-400">Email Verified!</h1>
              <p className="text-gray-300 mb-6">{message}</p>
              <div className="bg-turquoise/10 border border-turquoise/30 rounded-lg p-4 mb-6">
                <p className="text-sm text-turquoise">
                  🎉 Welcome to Vibe Coding Bros! Redirecting you to your dashboard...
                </p>
              </div>
              <Link to="/dashboard" className="btn-primary inline-block">
                Go to Dashboard Now
              </Link>
            </>
          )}

          {status === 'error' && (
            <>
              <FaTimesCircle className="text-red-500 text-6xl mx-auto mb-6" />
              <h1 className="text-3xl font-bold mb-3 text-red-400">Verification Failed</h1>
              <p className="text-gray-300 mb-6">{message}</p>
              <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-lg p-4 mb-6">
                <p className="text-sm text-yellow-400">
                  💡 Verification links expire after 24 hours. You may need to request a new one.
                </p>
              </div>
              <div className="space-y-3">
                <Link to="/signup" className="btn-primary block">
                  Sign Up Again
                </Link>
                <Link to="/signin" className="text-turquoise hover:underline block">
                  Already verified? Sign In
                </Link>
              </div>
            </>
          )}
        </div>
      </div>
    </div>
  );
};

export default VerifyEmail;
