import { useEffect, useMemo, useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import { FaCheckCircle, FaCrown, FaGraduationCap, FaLock, FaCalendarAlt } from 'react-icons/fa';
import api from '../services/api';
import { useAuth } from '../context/AuthContext';

const SCHEDULER_SCRIPT_SRC = 'https://meeting.viralwebsem.com/assets/embed.js';
const SCHEDULER_URL = 'https://meeting.viralwebsem.com/matt/consulting-call';

const PRICING = [
  { hours: 1, label: '1 hour', price: 100 },
  { hours: 2, label: '2 hours', price: 200 },
  { hours: 3, label: '3 hours', price: 300 },
];

const useQuery = () => {
  const { search } = useLocation();
  return useMemo(() => new URLSearchParams(search), [search]);
};

const loadSchedulerScriptOnce = () => {
  if (document.querySelector(`script[src="${SCHEDULER_SCRIPT_SRC}"]`)) return;
  const script = document.createElement('script');
  script.src = SCHEDULER_SCRIPT_SRC;
  script.defer = true;
  document.body.appendChild(script);
};

const VibeClasses = () => {
  const navigate = useNavigate();
  const query = useQuery();
  const { isAuthenticated } = useAuth();

  const [hours, setHours] = useState(1);
  const [isPaying, setIsPaying] = useState(false);
  const [error, setError] = useState('');
  const [purchases, setPurchases] = useState([]);
  const [loadingPurchases, setLoadingPurchases] = useState(false);

  const success = query.get('success') === '1';
  const canceled = query.get('canceled') === '1';

  const selected = PRICING.find((p) => p.hours === hours) || PRICING[0];

  useEffect(() => {
    if (success) loadSchedulerScriptOnce();
  }, [success]);

  useEffect(() => {
    const loadPurchases = async () => {
      if (!success || !isAuthenticated) return;
      setLoadingPurchases(true);
      try {
        const res = await api.get('/vibe-classes/mine');
        setPurchases(Array.isArray(res?.data?.purchases) ? res.data.purchases : []);
      } catch {
        // Non-blocking: webhook may not have processed yet
      } finally {
        setLoadingPurchases(false);
      }
    };
    loadPurchases();
  }, [success, isAuthenticated]);

  const startCheckout = async () => {
    setError('');

    if (!isAuthenticated) {
      navigate('/signin');
      return;
    }

    setIsPaying(true);
    try {
      const res = await api.post('/stripe/create-vibe-class-checkout', { hours });
      const url = res?.data?.url;
      if (!url) throw new Error('Missing Stripe checkout URL');
      window.location.href = url;
    } catch (e) {
      const msg = e?.response?.data?.message || e?.message || 'Failed to start checkout';
      setError(msg);
    } finally {
      setIsPaying(false);
    }
  };

  return (
    <div className="min-h-screen bg-vibe-black text-white">
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 pt-28 pb-16">
        <div className="relative overflow-hidden rounded-3xl border border-gray-800 bg-gradient-to-br from-gray-950 via-gray-900 to-gray-950 p-8 md:p-12">
          <div className="absolute inset-0 pointer-events-none opacity-20 bg-[radial-gradient(circle_at_20%_20%,rgba(32,178,170,0.35),transparent_55%),radial-gradient(circle_at_85%_30%,rgba(255,215,0,0.22),transparent_45%),radial-gradient(circle_at_40%_85%,rgba(147,112,219,0.25),transparent_55%)]" />

          <div className="relative flex flex-col md:flex-row md:items-center md:justify-between gap-6">
            <div>
              <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-turquoise/10 text-turquoise border border-turquoise/20 text-sm font-semibold">
                <FaCrown /> Premium 1:1 Coaching
              </div>
              <h1 className="mt-4 text-4xl md:text-5xl font-extrabold leading-tight">
                Learn <span className="text-gradient">Vibe Coding</span> with Matt
              </h1>
              <p className="mt-4 text-gray-300 max-w-2xl">
                Book a focused 1:1 session to ship faster, debug smarter, and build real features end-to-end.
                Pick your duration, pay securely via Stripe, then book your time.
              </p>
            </div>

            <div className="bg-gray-900/60 border border-gray-800 rounded-2xl p-6 w-full md:w-[420px]">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2 text-lg font-bold">
                  <FaGraduationCap /> 1:1 Session
                </div>
                <div className="text-sm text-gray-400">Per hour pricing</div>
              </div>

              <div className="mt-4 grid grid-cols-3 gap-2">
                {PRICING.map((p) => (
                  <button
                    key={p.hours}
                    type="button"
                    onClick={() => setHours(p.hours)}
                    className={`rounded-xl border px-3 py-3 text-left transition-colors ${
                      p.hours === hours
                        ? 'border-turquoise bg-turquoise/10'
                        : 'border-gray-800 bg-gray-950 hover:bg-gray-900'
                    }`}
                  >
                    <div className="text-sm font-semibold">{p.label}</div>
                    <div className="text-xs text-gray-400">${p.price}</div>
                  </button>
                ))}
              </div>

              <div className="mt-4 flex items-center justify-between">
                <div className="text-gray-300">Total</div>
                <div className="text-2xl font-extrabold">${selected.price}</div>
              </div>

              <button
                type="button"
                onClick={startCheckout}
                disabled={isPaying}
                className="mt-5 w-full btn-primary flex items-center justify-center gap-2"
              >
                {isPaying ? 'Redirecting…' : 'Pay with Stripe'}
              </button>

              {error && <div className="mt-3 text-sm text-red-400">{error}</div>}

              {!isAuthenticated && (
                <div className="mt-3 text-xs text-gray-400 flex items-center gap-2">
                  <FaLock /> Sign in required to checkout
                </div>
              )}

              {canceled && (
                <div className="mt-3 text-sm text-yellow-300">
                  Checkout canceled. You can try again anytime.
                </div>
              )}

              {success && (
                <div className="mt-3 text-sm text-green-300 flex items-center gap-2">
                  <FaCheckCircle /> Payment received. Book your meeting below.
                </div>
              )}

              <div className="mt-5 border-t border-gray-800 pt-4">
                {success ? (
                  <button
                    data-scheduler-url={SCHEDULER_URL}
                    className="w-full rounded-xl bg-gradient-to-r from-turquoise to-gold text-black font-extrabold px-5 py-4 hover:opacity-90 transition-opacity flex items-center justify-center gap-2"
                  >
                    <FaCalendarAlt /> Schedule Your Class
                  </button>
                ) : (
                  <div className="rounded-xl border border-gray-800 bg-gray-950 px-4 py-3 text-sm text-gray-400">
                    Complete payment to unlock booking.
                  </div>
                )}
              </div>

              {success && (
                <div className="mt-4 rounded-xl border border-gray-800 bg-gray-950 px-4 py-3">
                  <div className="text-sm font-semibold text-gray-200">Your purchases</div>
                  {loadingPurchases ? (
                    <div className="mt-2 text-sm text-gray-400">Loading…</div>
                  ) : purchases.length === 0 ? (
                    <div className="mt-2 text-sm text-gray-400">
                      Payment received. Your receipt will appear here shortly.
                    </div>
                  ) : (
                    <div className="mt-2 space-y-1 text-sm text-gray-300">
                      {purchases.slice(0, 3).map((p) => (
                        <div key={p._id} className="flex items-center justify-between">
                          <div>
                            {p.hours} hour{p.hours === 1 ? '' : 's'}
                          </div>
                          <div className="text-gray-400">
                            ${p.amountUsd} • {new Date(p.createdAt).toLocaleDateString()}
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              <div className="mt-4 text-xs text-gray-500">
                Payments processed securely via Stripe. Booking runs on meeting.viralwebsem.com.
              </div>
            </div>
          </div>
        </div>

        <div className="mt-10 grid md:grid-cols-3 gap-6">
          {[
            {
              title: 'Ship real features',
              desc: 'Go from idea → PR-ready code. Focus on architecture, DX, and speed without the chaos.',
            },
            {
              title: 'Debug like a pro',
              desc: 'Triage issues, isolate root causes, and build guardrails so bugs don’t come back.',
            },
            {
              title: 'Level up your stack',
              desc: 'React, Node/Express, MongoDB, Stripe, deploys, testing, and modern workflows.',
            },
          ].map((f) => (
            <div key={f.title} className="card border-gray-800 bg-gray-900/40">
              <div className="text-xl font-bold mb-2">{f.title}</div>
              <div className="text-gray-300">{f.desc}</div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};

export default VibeClasses;
