import { useState } from 'react';
import { Link } from 'react-router-dom';
import { 
  BookOpen, 
  Code, 
  Play, 
  CheckCircle, 
  Copy, 
  ExternalLink,
  Zap,
  Users,
  Award,
  MessageSquare
} from 'lucide-react';

const WorkflowDemo = () => {
  const [activeStep, setActiveStep] = useState(0);
  const [copiedCode, setCopiedCode] = useState(null);
  const [demoInput, setDemoInput] = useState('');
  const [demoOutput, setDemoOutput] = useState('');
  const [isProcessing, setIsProcessing] = useState(false);

  const steps = [
    {
      title: 'Set Up OpenAI Account',
      duration: '5 minutes',
      content: 'Create your OpenAI account and obtain your API key from the developer dashboard.',
      code: `// Store your API key securely in environment variables
// Never commit API keys to version control!

// .env file
OPENAI_API_KEY=sk-your-api-key-here

// Access in your code
const apiKey = process.env.OPENAI_API_KEY;`,
      links: [
        { text: 'OpenAI Signup', url: 'https://platform.openai.com/signup' },
        { text: 'API Keys Dashboard', url: 'https://platform.openai.com/api-keys' }
      ]
    },
    {
      title: 'Install OpenAI SDK',
      duration: '2 minutes',
      content: 'Install the official OpenAI Node.js library using npm or yarn.',
      code: `# Using npm
npm install openai

# Using yarn
yarn add openai

# Using pnpm
pnpm add openai`,
      links: [
        { text: 'OpenAI Node.js Docs', url: 'https://github.com/openai/openai-node' }
      ]
    },
    {
      title: 'Initialize OpenAI Client',
      duration: '3 minutes',
      content: 'Create an OpenAI client instance with your API key.',
      code: `import OpenAI from 'openai';

// Initialize the OpenAI client
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Verify connection (optional)
async function testConnection() {
  try {
    const models = await openai.models.list();
    console.log('✅ Connected to OpenAI successfully!');
    return true;
  } catch (error) {
    console.error('❌ Connection failed:', error.message);
    return false;
  }
}`,
    },
    {
      title: 'Create Chat Completion Function',
      duration: '10 minutes',
      content: 'Build a function that sends messages to GPT-4 and receives intelligent responses.',
      code: `async function createChatCompletion(userMessage, systemPrompt = '') {
  try {
    const response = await openai.chat.completions.create({
      model: 'gpt-4-turbo-preview',
      messages: [
        {
          role: 'system',
          content: systemPrompt || 'You are a helpful customer support assistant.'
        },
        {
          role: 'user',
          content: userMessage
        }
      ],
      temperature: 0.7,
      max_tokens: 500,
    });

    return {
      success: true,
      message: response.choices[0].message.content,
      usage: response.usage
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}`,
    },
    {
      title: 'Handle Conversation Context',
      duration: '15 minutes',
      content: 'Implement conversation memory to maintain context across multiple messages.',
      code: `class ChatBot {
  constructor(systemPrompt) {
    this.messages = [
      { role: 'system', content: systemPrompt }
    ];
  }

  async sendMessage(userMessage) {
    // Add user message to conversation history
    this.messages.push({
      role: 'user',
      content: userMessage
    });

    try {
      const response = await openai.chat.completions.create({
        model: 'gpt-4-turbo-preview',
        messages: this.messages,
        temperature: 0.7,
        max_tokens: 500,
      });

      const assistantMessage = response.choices[0].message.content;

      // Add assistant response to conversation history
      this.messages.push({
        role: 'assistant',
        content: assistantMessage
      });

      return {
        success: true,
        message: assistantMessage,
        conversationLength: this.messages.length
      };
    } catch (error) {
      return {
        success: false,
        error: error.message
      };
    }
  }

  clearHistory() {
    // Keep only the system message
    this.messages = this.messages.slice(0, 1);
  }
}

// Usage example
const bot = new ChatBot('You are a helpful customer support assistant.');
await bot.sendMessage('How do I reset my password?');
await bot.sendMessage('What if I dont have access to my email?');`,
    },
    {
      title: 'Build Express API Endpoint',
      duration: '20 minutes',
      content: 'Create a production-ready REST API endpoint for your chatbot.',
      code: `import express from 'express';
import OpenAI from 'openai';

const app = express();
app.use(express.json());

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Store active conversations (use Redis in production)
const conversations = new Map();

app.post('/api/chat', async (req, res) => {
  try {
    const { message, conversationId } = req.body;

    if (!message) {
      return res.status(400).json({ error: 'Message is required' });
    }

    // Get or create conversation history
    let messages = conversations.get(conversationId) || [
      {
        role: 'system',
        content: 'You are a helpful customer support assistant.'
      }
    ];

    // Add user message
    messages.push({
      role: 'user',
      content: message
    });

    // Get GPT-4 response
    const response = await openai.chat.completions.create({
      model: 'gpt-4-turbo-preview',
      messages: messages,
      temperature: 0.7,
      max_tokens: 500,
    });

    const assistantMessage = response.choices[0].message.content;

    // Add assistant response
    messages.push({
      role: 'assistant',
      content: assistantMessage
    });

    // Store updated conversation
    conversations.set(conversationId, messages);

    res.json({
      success: true,
      message: assistantMessage,
      conversationId,
      usage: response.usage
    });
  } catch (error) {
    console.error('Chat error:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

app.listen(3000, () => {
  console.log('Chatbot API running on port 3000');
});`,
    },
    {
      title: 'Create React Frontend',
      duration: '30 minutes',
      content: 'Build a beautiful chat interface for your users.',
      code: `import { useState, useRef, useEffect } from 'react';
import axios from 'axios';

function ChatInterface() {
  const [messages, setMessages] = useState([]);
  const [input, setInput] = useState('');
  const [loading, setLoading] = useState(false);
  const [conversationId] = useState(\`conv-\${Date.now()}\`);
  const messagesEndRef = useRef(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const sendMessage = async (e) => {
    e.preventDefault();
    if (!input.trim() || loading) return;

    const userMessage = input.trim();
    setInput('');
    
    // Add user message to UI
    setMessages(prev => [...prev, { role: 'user', content: userMessage }]);
    setLoading(true);

    try {
      const response = await axios.post('/api/chat', {
        message: userMessage,
        conversationId
      });

      // Add assistant response to UI
      setMessages(prev => [...prev, {
        role: 'assistant',
        content: response.data.message
      }]);
    } catch (error) {
      console.error('Error:', error);
      setMessages(prev => [...prev, {
        role: 'assistant',
        content: 'Sorry, I encountered an error. Please try again.'
      }]);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="flex flex-col h-screen max-w-4xl mx-auto p-4">
      <div className="flex-1 overflow-y-auto space-y-4 mb-4">
        {messages.map((msg, idx) => (
          <div
            key={idx}
            className={\`flex \${msg.role === 'user' ? 'justify-end' : 'justify-start'}\`}
          >
            <div
              className={\`max-w-xs lg:max-w-md px-4 py-2 rounded-lg \${
                msg.role === 'user'
                  ? 'bg-blue-500 text-white'
                  : 'bg-gray-200 text-gray-900'
              }\`}
            >
              {msg.content}
            </div>
          </div>
        ))}
        {loading && (
          <div className="flex justify-start">
            <div className="bg-gray-200 text-gray-900 px-4 py-2 rounded-lg">
              <span className="animate-pulse">Thinking...</span>
            </div>
          </div>
        )}
        <div ref={messagesEndRef} />
      </div>

      <form onSubmit={sendMessage} className="flex gap-2">
        <input
          type="text"
          value={input}
          onChange={(e) => setInput(e.target.value)}
          placeholder="Type your message..."
          className="flex-1 px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
          disabled={loading}
        />
        <button
          type="submit"
          disabled={loading || !input.trim()}
          className="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Send
        </button>
      </form>
    </div>
  );
}

export default ChatInterface;`,
    },
  ];

  const handleCopyCode = (code, index) => {
    navigator.clipboard.writeText(code);
    setCopiedCode(index);
    setTimeout(() => setCopiedCode(null), 2000);
  };

  const handleDemoSubmit = async (e) => {
    e.preventDefault();
    if (!demoInput.trim()) return;

    setIsProcessing(true);
    setDemoOutput('');

    // Simulate API call with realistic delay
    await new Promise(resolve => setTimeout(resolve, 1500));

    // Simulated GPT-4 response
    const responses = {
      'hello': 'Hello! How can I assist you today?',
      'help': 'I\'m here to help! I can answer questions, provide guidance, and assist with customer support. What do you need help with?',
      'password': 'To reset your password, please follow these steps:\n1. Click "Forgot Password" on the login page\n2. Enter your email address\n3. Check your email for a reset link\n4. Create a new secure password\n\nIs there anything else I can help you with?',
      'default': `Thank you for your message: "${demoInput}"\n\nI'm a demo chatbot powered by GPT-4. In a production environment, I would use OpenAI's API to generate intelligent, context-aware responses to help with customer support, answer questions, and provide assistance.\n\nThis demo shows how the interface works. With a real API key, I could provide genuinely helpful responses!`
    };

    const responseKey = demoInput.toLowerCase().includes('password') ? 'password' 
      : demoInput.toLowerCase().includes('hello') || demoInput.toLowerCase().includes('hi') ? 'hello'
      : demoInput.toLowerCase().includes('help') ? 'help'
      : 'default';

    setDemoOutput(responses[responseKey]);
    setIsProcessing(false);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      {/* Header */}
      <div className="bg-white border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <Link to="/workflows" className="text-sm text-blue-600 hover:text-blue-700 mb-2 inline-block">
                ← Back to Workflows
              </Link>
              <h1 className="text-3xl font-bold text-gray-900">
                Build a Customer Support Chatbot with GPT-4
              </h1>
              <p className="text-gray-600 mt-2">
                Learn to create an intelligent chatbot using OpenAI's GPT-4 API
              </p>
            </div>
            <div className="flex items-center gap-2 bg-gradient-to-r from-blue-600 to-purple-600 text-white px-4 py-2 rounded-lg">
              <img src="https://openai.com/favicon.ico" alt="OpenAI" className="w-5 h-5" />
              <span className="font-semibold">Powered by OpenAI</span>
            </div>
          </div>

          {/* Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mt-6">
            <div className="bg-blue-50 rounded-lg p-4">
              <div className="flex items-center gap-2 text-blue-600 mb-1">
                <BookOpen className="w-4 h-4" />
                <span className="text-sm font-medium">Difficulty</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">Beginner</p>
            </div>
            <div className="bg-green-50 rounded-lg p-4">
              <div className="flex items-center gap-2 text-green-600 mb-1">
                <Zap className="w-4 h-4" />
                <span className="text-sm font-medium">Duration</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">1-2 hours</p>
            </div>
            <div className="bg-purple-50 rounded-lg p-4">
              <div className="flex items-center gap-2 text-purple-600 mb-1">
                <Users className="w-4 h-4" />
                <span className="text-sm font-medium">Completed</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">2,847</p>
            </div>
            <div className="bg-orange-50 rounded-lg p-4">
              <div className="flex items-center gap-2 text-orange-600 mb-1">
                <Award className="w-4 h-4" />
                <span className="text-sm font-medium">Rating</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">4.9/5</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Sidebar - Steps */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-xl shadow-sm p-6 sticky top-8">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">
                Learning Path
              </h3>
              <div className="space-y-2">
                {steps.map((step, index) => (
                  <button
                    key={index}
                    onClick={() => setActiveStep(index)}
                    className={`w-full text-left p-3 rounded-lg transition-all ${
                      activeStep === index
                        ? 'bg-blue-50 border-2 border-blue-500'
                        : 'hover:bg-gray-50 border-2 border-transparent'
                    }`}
                  >
                    <div className="flex items-start gap-3">
                      <div
                        className={`flex-shrink-0 w-6 h-6 rounded-full flex items-center justify-center text-sm font-semibold ${
                          activeStep === index
                            ? 'bg-blue-500 text-white'
                            : 'bg-gray-200 text-gray-600'
                        }`}
                      >
                        {index + 1}
                      </div>
                      <div className="flex-1 min-w-0">
                        <p
                          className={`text-sm font-medium ${
                            activeStep === index
                              ? 'text-blue-700'
                              : 'text-gray-700'
                          }`}
                        >
                          {step.title}
                        </p>
                        <p className="text-xs text-gray-500 mt-1">
                          {step.duration}
                        </p>
                      </div>
                      {activeStep === index && (
                        <CheckCircle className="w-5 h-5 text-blue-500 flex-shrink-0" />
                      )}
                    </div>
                  </button>
                ))}
              </div>

              <div className="mt-6 pt-6 border-t">
                <h4 className="text-sm font-semibold text-gray-900 mb-3">
                  What You'll Learn
                </h4>
                <ul className="space-y-2 text-sm text-gray-600">
                  <li className="flex items-start gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 flex-shrink-0" />
                    <span>OpenAI API authentication</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 flex-shrink-0" />
                    <span>GPT-4 chat completions</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 flex-shrink-0" />
                    <span>Conversation context management</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 flex-shrink-0" />
                    <span>Building REST APIs</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 flex-shrink-0" />
                    <span>React chat interfaces</span>
                  </li>
                </ul>
              </div>
            </div>
          </div>

          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            {/* Current Step Content */}
            <div className="bg-white rounded-xl shadow-sm p-8">
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-bold text-gray-900">
                  Step {activeStep + 1}: {steps[activeStep].title}
                </h2>
                <span className="text-sm text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                  {steps[activeStep].duration}
                </span>
              </div>

              <p className="text-gray-700 mb-6 leading-relaxed">
                {steps[activeStep].content}
              </p>

              {/* Code Block */}
              {steps[activeStep].code && (
                <div className="relative">
                  <div className="flex items-center justify-between bg-gray-800 text-white px-4 py-2 rounded-t-lg">
                    <div className="flex items-center gap-2">
                      <Code className="w-4 h-4" />
                      <span className="text-sm font-medium">Code Example</span>
                    </div>
                    <button
                      onClick={() => handleCopyCode(steps[activeStep].code, activeStep)}
                      className="flex items-center gap-2 text-sm hover:text-blue-300 transition-colors"
                    >
                      {copiedCode === activeStep ? (
                        <>
                          <CheckCircle className="w-4 h-4" />
                          <span>Copied!</span>
                        </>
                      ) : (
                        <>
                          <Copy className="w-4 h-4" />
                          <span>Copy</span>
                        </>
                      )}
                    </button>
                  </div>
                  <pre className="bg-gray-900 text-gray-100 p-4 rounded-b-lg overflow-x-auto">
                    <code className="text-sm">{steps[activeStep].code}</code>
                  </pre>
                </div>
              )}

              {/* External Links */}
              {steps[activeStep].links && steps[activeStep].links.length > 0 && (
                <div className="mt-6 p-4 bg-blue-50 rounded-lg">
                  <h4 className="text-sm font-semibold text-blue-900 mb-2">
                    Helpful Resources
                  </h4>
                  <div className="space-y-2">
                    {steps[activeStep].links.map((link, idx) => (
                      <a
                        key={idx}
                        href={link.url}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="flex items-center gap-2 text-blue-600 hover:text-blue-700 text-sm"
                      >
                        <ExternalLink className="w-4 h-4" />
                        <span>{link.text}</span>
                      </a>
                    ))}
                  </div>
                </div>
              )}

              {/* Navigation */}
              <div className="flex items-center justify-between mt-8 pt-6 border-t">
                <button
                  onClick={() => setActiveStep(Math.max(0, activeStep - 1))}
                  disabled={activeStep === 0}
                  className="px-4 py-2 text-gray-700 hover:bg-gray-100 rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Previous Step
                </button>
                <span className="text-sm text-gray-500">
                  Step {activeStep + 1} of {steps.length}
                </span>
                <button
                  onClick={() => setActiveStep(Math.min(steps.length - 1, activeStep + 1))}
                  disabled={activeStep === steps.length - 1}
                  className="px-4 py-2 bg-blue-500 text-white hover:bg-blue-600 rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Next Step
                </button>
              </div>
            </div>

            {/* Interactive Demo */}
            <div className="bg-gradient-to-br from-blue-500 to-purple-600 rounded-xl shadow-lg p-8 text-white">
              <div className="flex items-center gap-3 mb-4">
                <Play className="w-6 h-6" />
                <h3 className="text-2xl font-bold">Try It Live (Demo Mode)</h3>
              </div>
              <p className="text-blue-100 mb-6">
                Experience how the chatbot works. In production, this would use real GPT-4 API calls.
              </p>

              <form onSubmit={handleDemoSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium mb-2">
                    Ask the chatbot a question:
                  </label>
                  <textarea
                    value={demoInput}
                    onChange={(e) => setDemoInput(e.target.value)}
                    placeholder="Try: 'How do I reset my password?' or 'Hello!'"
                    className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/50 focus:outline-none focus:ring-2 focus:ring-white/50"
                    rows="3"
                    disabled={isProcessing}
                  />
                </div>

                <button
                  type="submit"
                  disabled={!demoInput.trim() || isProcessing}
                  className="w-full px-6 py-3 bg-white text-blue-600 font-semibold rounded-lg hover:bg-blue-50 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
                >
                  {isProcessing ? (
                    <>
                      <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                      <span>Processing...</span>
                    </>
                  ) : (
                    <>
                      <MessageSquare className="w-5 h-5" />
                      <span>Send Message</span>
                    </>
                  )}
                </button>
              </form>

              {demoOutput && (
                <div className="mt-6 p-4 bg-white/10 rounded-lg border border-white/20">
                  <p className="text-sm font-medium mb-2">Chatbot Response:</p>
                  <p className="text-white whitespace-pre-line">{demoOutput}</p>
                </div>
              )}

              <div className="mt-6 p-4 bg-white/5 rounded-lg border border-white/10">
                <p className="text-xs text-blue-100">
                  💡 <strong>Note:</strong> This is a demo interface. With your OpenAI API key, 
                  you'll get real GPT-4 responses that understand context, provide accurate 
                  information, and maintain conversation history.
                </p>
              </div>
            </div>

            {/* Next Steps */}
            <div className="bg-white rounded-xl shadow-sm p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-4">
                What's Next?
              </h3>
              <div className="space-y-4">
                <div className="flex items-start gap-4 p-4 bg-green-50 rounded-lg">
                  <CheckCircle className="w-6 h-6 text-green-600 flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-1">
                      Join the OpenAI Innovation Challenge
                    </h4>
                    <p className="text-sm text-gray-600 mb-2">
                      Win $10,000 by building the most innovative GPT-4 application.
                    </p>
                    <Link
                      to="/challenges"
                      className="text-sm text-green-600 hover:text-green-700 font-medium"
                    >
                      View Challenge →
                    </Link>
                  </div>
                </div>

                <div className="flex items-start gap-4 p-4 bg-blue-50 rounded-lg">
                  <Code className="w-6 h-6 text-blue-600 flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-1">
                      Explore GPT-4 Templates
                    </h4>
                    <p className="text-sm text-gray-600 mb-2">
                      Browse ready-to-use components and integrations in our marketplace.
                    </p>
                    <Link
                      to="/marketplace"
                      className="text-sm text-blue-600 hover:text-blue-700 font-medium"
                    >
                      Browse Templates →
                    </Link>
                  </div>
                </div>

                <div className="flex items-start gap-4 p-4 bg-purple-50 rounded-lg">
                  <Users className="w-6 h-6 text-purple-600 flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-1">
                      Get Hired as a GPT-4 Developer
                    </h4>
                    <p className="text-sm text-gray-600 mb-2">
                      Companies are hiring developers with OpenAI API experience.
                    </p>
                    <Link
                      to="/jobs"
                      className="text-sm text-purple-600 hover:text-purple-700 font-medium"
                    >
                      View Job Board →
                    </Link>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default WorkflowDemo;
