import { useState, useEffect } from 'react';
import { 
  FaArrowUp, FaArrowDown, FaPlus, FaSearch, FaTimes, FaBookmark, FaRegBookmark,
  FaShare, FaClock, FaCode, FaCheckCircle, FaEye, FaComment, FaStar,
  FaFilter, FaThLarge, FaList, FaCopy, FaDownload, FaGithub, FaLinkedin, FaTwitter,
  FaTrophy, FaFire, FaHeart, FaChartLine, FaHistory, FaUser, FaCodeBranch, FaProjectDiagram
} from 'react-icons/fa';
import { useAuth } from '../context/AuthContext';
import MessageButton from '../components/MessageButton';
import VisualWorkflowBuilder from '../components/VisualWorkflowBuilder';
import api from '../services/api';

const Workflows = () => {
  const [workflows, setWorkflows] = useState([]);
  const [loading, setLoading] = useState(true);
  const [category, setCategory] = useState('all');
  const [searchQuery, setSearchQuery] = useState('');
  const [sortBy, setSortBy] = useState('trending');
  const [viewMode, setViewMode] = useState('grid');
  const [showFilters, setShowFilters] = useState(false);
  const [selectedDifficulty, setSelectedDifficulty] = useState([]);
  const [selectedTools, setSelectedTools] = useState([]);
  const [selectedTime, setSelectedTime] = useState([]);
  const [bookmarkedWorkflows, setBookmarkedWorkflows] = useState([]);
  const [selectedWorkflow, setSelectedWorkflow] = useState(null);
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [recentlyViewed, setRecentlyViewed] = useState([]);
  const [showPersonalized, setShowPersonalized] = useState(false);
  const { isAuthenticated, user } = useAuth();

  useEffect(() => {
    fetchWorkflows();
  }, []);

  const fetchWorkflows = async () => {
    try {
      const response = await api.get('/workflows');
      setWorkflows(response.data.workflows || []);
    } catch (error) {
      console.error('Error fetching workflows:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleVote = async (workflowId, voteType) => {
    if (!isAuthenticated) {
      alert('Please sign in to vote');
      return;
    }
    try {
      await api.post(`/workflows/${workflowId}/vote`, { voteType });
      fetchWorkflows();
    } catch (error) {
      console.error('Error voting:', error);
    }
  };

  const categories = ['All', 'Frontend', 'Backend', 'DevOps', 'Productivity', 'Tools', 'Testing', 'CI/CD', 'Git'];
  const difficulties = ['Beginner', 'Intermediate', 'Advanced'];
  const tools = ['VS Code', 'Git', 'Docker', 'GitHub Actions', 'Jest', 'Webpack', 'npm', 'Node.js'];
  const timeEstimates = ['< 5 min', '5-15 min', '15-30 min', '30+ min'];
  const sortOptions = [
    { value: 'trending', label: 'Trending' },
    { value: 'newest', label: 'Newest' },
    { value: 'top-rated', label: 'Top Rated' },
    { value: 'most-bookmarked', label: 'Most Bookmarked' }
  ];

  // Featured/Template workflows
  const templates = [
    { id: 1, title: 'Git Workflow for Teams', category: 'Git', difficulty: 'Intermediate', time: '15-30 min', icon: FaGithub },
    { id: 2, title: 'CI/CD Pipeline Setup', category: 'CI/CD', difficulty: 'Advanced', time: '30+ min', icon: FaCode },
    { id: 3, title: 'Testing Best Practices', category: 'Testing', difficulty: 'Beginner', time: '5-15 min', icon: FaCheckCircle },
    { id: 4, title: 'Docker Dev Environment', category: 'DevOps', difficulty: 'Intermediate', time: '15-30 min', icon: FaCode }
  ];

  // Filter and sort workflows
  const getFilteredWorkflows = () => {
    let filtered = workflows;

    // Search
    if (searchQuery) {
      filtered = filtered.filter(w => 
        w.title?.toLowerCase().includes(searchQuery.toLowerCase()) ||
        w.content?.toLowerCase().includes(searchQuery.toLowerCase())
      );
    }

    // Category
    if (category !== 'all') {
      filtered = filtered.filter(w => w.category?.toLowerCase() === category);
    }

    // Difficulty
    if (selectedDifficulty.length > 0) {
      filtered = filtered.filter(w => selectedDifficulty.includes(w.difficulty));
    }

    // Tools
    if (selectedTools.length > 0) {
      filtered = filtered.filter(w => 
        w.tools?.some(tool => selectedTools.includes(tool))
      );
    }

    // Time estimate
    if (selectedTime.length > 0) {
      filtered = filtered.filter(w => selectedTime.includes(w.timeEstimate));
    }

    // Sort
    switch (sortBy) {
      case 'newest':
        filtered.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        break;
      case 'top-rated':
        filtered.sort((a, b) => (b.votes || 0) - (a.votes || 0));
        break;
      case 'most-bookmarked':
        filtered.sort((a, b) => (b.bookmarks || 0) - (a.bookmarks || 0));
        break;
      case 'trending':
      default:
        // Trending algorithm: combine votes, views, and recency
        filtered.sort((a, b) => {
          const scoreA = (a.votes || 0) * 2 + (a.views || 0) * 0.1 + (a.bookmarks || 0) * 3;
          const scoreB = (b.votes || 0) * 2 + (b.views || 0) * 0.1 + (b.bookmarks || 0) * 3;
          return scoreB - scoreA;
        });
    }

    return filtered;
  };

  const filteredWorkflows = getFilteredWorkflows();

  const activeFiltersCount = 
    selectedDifficulty.length + 
    selectedTools.length + 
    selectedTime.length + 
    (category !== 'all' ? 1 : 0) +
    (searchQuery ? 1 : 0);

  const clearAllFilters = () => {
    setCategory('all');
    setSearchQuery('');
    setSelectedDifficulty([]);
    setSelectedTools([]);
    setSelectedTime([]);
  };

  const toggleBookmark = async (workflowId) => {
    if (!isAuthenticated) {
      alert('Please sign in to bookmark');
      return;
    }
    const isBookmarked = bookmarkedWorkflows.includes(workflowId);
    if (isBookmarked) {
      setBookmarkedWorkflows(bookmarkedWorkflows.filter(id => id !== workflowId));
    } else {
      setBookmarkedWorkflows([...bookmarkedWorkflows, workflowId]);
    }
    // API call would go here
  };

  const handleOpenDetail = (workflow) => {
    setSelectedWorkflow(workflow);
    
    // Track recently viewed
    if (!recentlyViewed.includes(workflow._id)) {
      const updated = [workflow._id, ...recentlyViewed].slice(0, 5); // Keep last 5
      setRecentlyViewed(updated);
      localStorage.setItem('recentlyViewedWorkflows', JSON.stringify(updated));
    }
    
    // Increment view count (would be API call in production)
    setWorkflows(workflows.map(w => 
      w._id === workflow._id ? { ...w, views: (w.views || 0) + 1 } : w
    ));
  };

  const handleForkWorkflow = async (workflowId) => {
    if (!isAuthenticated) {
      alert('Please sign in to fork workflows');
      return;
    }
    
    try {
      const response = await api.post(`/workflows/${workflowId}/fork`);
      alert('Workflow forked successfully! Check your workflows.');
      fetchWorkflows();
    } catch (error) {
      console.error('Error forking workflow:', error);
      alert('Failed to fork workflow');
    }
  };

  // Load recently viewed from localStorage on mount
  useEffect(() => {
    const stored = localStorage.getItem('recentlyViewedWorkflows');
    if (stored) {
      setRecentlyViewed(JSON.parse(stored));
    }
  }, []);

  const handleShare = (workflow, platform) => {
    const url = window.location.href;
    const text = `Check out this workflow: ${workflow.title}`;
    
    const shareUrls = {
      twitter: `https://twitter.com/intent/tweet?text=${encodeURIComponent(text)}&url=${encodeURIComponent(url)}`,
      linkedin: `https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(url)}`,
      copy: url
    };

    if (platform === 'copy') {
      navigator.clipboard.writeText(url);
      alert('Link copied to clipboard!');
    } else {
      window.open(shareUrls[platform], '_blank');
    }
  };

  return (
    <div className="min-h-screen pt-24 pb-12 bg-gradient-to-b from-vibe-black to-gray-900">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Authentication Gate */}
        {!isAuthenticated ? (
          <div className="flex flex-col items-center justify-center py-20">
            <div className="bg-gray-900 rounded-2xl p-12 border border-gray-800 max-w-2xl w-full text-center">
              <div className="mb-6">
                <FaProjectDiagram className="text-turquoise mx-auto mb-4" size={64} />
                <h2 className="text-3xl font-bold mb-3">
                  Unlock <span className="text-gradient">Premium Workflows</span>
                </h2>
                <p className="text-gray-400 text-lg mb-2">
                  Access exclusive coding workflows shared by our community
                </p>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-8 text-left">
                <div className="bg-gray-800/50 rounded-lg p-4 border border-gray-700">
                  <FaCheckCircle className="text-turquoise mb-2" size={24} />
                  <h3 className="font-semibold mb-1">Visual Workflow Builder</h3>
                  <p className="text-sm text-gray-400">Create and share step-by-step workflows</p>
                </div>
                <div className="bg-gray-800/50 rounded-lg p-4 border border-gray-700">
                  <FaCodeBranch className="text-gold mb-2" size={24} />
                  <h3 className="font-semibold mb-1">Fork & Customize</h3>
                  <p className="text-sm text-gray-400">Build on workflows from other developers</p>
                </div>
                <div className="bg-gray-800/50 rounded-lg p-4 border border-gray-700">
                  <FaTrophy className="text-orange-500 mb-2" size={24} />
                  <h3 className="font-semibold mb-1">Trending Workflows</h3>
                  <p className="text-sm text-gray-400">Discover top-rated productivity tips</p>
                </div>
                <div className="bg-gray-800/50 rounded-lg p-4 border border-gray-700">
                  <FaBookmark className="text-purple-500 mb-2" size={24} />
                  <h3 className="font-semibold mb-1">Save Favorites</h3>
                  <p className="text-sm text-gray-400">Bookmark workflows for quick access</p>
                </div>
              </div>

              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <a 
                  href="/signup" 
                  className="btn-primary px-8 py-3 text-lg"
                >
                  Create Free Account
                </a>
                <a 
                  href="/signin" 
                  className="px-8 py-3 text-lg bg-gray-800 hover:bg-gray-700 rounded-lg transition-colors border border-gray-700"
                >
                  Sign In
                </a>
              </div>

              <p className="text-sm text-gray-500 mt-6">
                Join {workflows.length > 0 ? workflows.length : '100+'} workflows already shared by our community
              </p>
            </div>
          </div>
        ) : (
          <>
        {/* Header */}
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center mb-8">
          <div>
            <h1 className="text-4xl font-bold mb-2">
              Coding <span className="text-gradient">Workflows</span>
            </h1>
            <p className="text-gray-400">Discover and share productivity workflows • {filteredWorkflows.length} workflows</p>
          </div>
          {isAuthenticated && (
            <button 
              onClick={() => setShowCreateModal(true)}
              className="btn-primary mt-4 md:mt-0 flex items-center gap-2"
            >
              <FaPlus /> Share Workflow
            </button>
          )}
        </div>

        {/* Featured Templates Section */}
        <div className="mb-8 bg-gradient-to-r from-turquoise/10 to-gold/10 rounded-xl p-6 border border-turquoise/20">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-2xl font-bold flex items-center gap-2">
              <FaStar className="text-gold" /> Featured Workflow Templates
            </h2>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            {templates.map(template => (
              <div key={template.id} className="bg-gray-800/50 rounded-lg p-4 hover:bg-gray-800 transition-all cursor-pointer border border-gray-700 hover:border-turquoise">
                <div className="flex items-center gap-3 mb-3">
                  <div className="p-2 bg-turquoise/20 rounded-lg">
                    <template.icon className="text-turquoise" size={20} />
                  </div>
                  <div className="flex-1">
                    <h3 className="font-bold text-sm">{template.title}</h3>
                    <p className="text-xs text-gray-400">{template.category}</p>
                  </div>
                </div>
                <div className="flex items-center gap-2 text-xs">
                  <span className="px-2 py-1 bg-blue-500/20 text-blue-400 rounded">{template.difficulty}</span>
                  <span className="text-gray-500 flex items-center gap-1">
                    <FaClock size={10} /> {template.time}
                  </span>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Analytics & Personalization Section */}
        {isAuthenticated && (
          <div className="mb-8 grid grid-cols-1 md:grid-cols-3 gap-4">
            {/* Trending This Week */}
            <div className="bg-gradient-to-br from-orange-500/10 to-red-500/10 border border-orange-500/20 rounded-xl p-6">
              <div className="flex items-center gap-2 mb-4">
                <FaFire className="text-orange-500" size={24} />
                <h3 className="text-xl font-bold">Trending This Week</h3>
              </div>
              <div className="space-y-3">
                {workflows.slice(0, 3).map((w, idx) => (
                  <div key={w._id} className="flex items-center gap-3 text-sm">
                    <span className="flex-shrink-0 w-6 h-6 bg-orange-500 text-vibe-black rounded-full flex items-center justify-center font-bold text-xs">
                      {idx + 1}
                    </span>
                    <div className="flex-1 truncate">
                      <p className="font-semibold truncate">{w.title}</p>
                      <p className="text-xs text-gray-400">{w.votes || 0} votes • {w.views || 0} views</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Your Saved Workflows */}
            <div className="bg-gradient-to-br from-gold/10 to-yellow-500/10 border border-gold/20 rounded-xl p-6">
              <div className="flex items-center gap-2 mb-4">
                <FaBookmark className="text-gold" size={20} />
                <h3 className="text-xl font-bold">Your Saved</h3>
              </div>
              <div className="space-y-3">
                {bookmarkedWorkflows.length > 0 ? (
                  workflows
                    .filter(w => bookmarkedWorkflows.includes(w._id))
                    .slice(0, 3)
                    .map(w => (
                      <div key={w._id} className="text-sm cursor-pointer hover:text-turquoise" onClick={() => setSelectedWorkflow(w)}>
                        <p className="font-semibold truncate">{w.title}</p>
                        <p className="text-xs text-gray-400">{w.category}</p>
                      </div>
                    ))
                ) : (
                  <p className="text-sm text-gray-400">No saved workflows yet</p>
                )}
              </div>
              <div className="mt-4 pt-4 border-t border-gray-700">
                <p className="text-lg font-bold text-gold">{bookmarkedWorkflows.length}</p>
                <p className="text-xs text-gray-400">Total Saved</p>
              </div>
            </div>

            {/* Success Metrics */}
            <div className="bg-gradient-to-br from-green-500/10 to-turquoise/10 border border-green-500/20 rounded-xl p-6">
              <div className="flex items-center gap-2 mb-4">
                <FaChartLine className="text-green-500" size={20} />
                <h3 className="text-xl font-bold">Your Impact</h3>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-2xl font-bold text-turquoise">{workflows.filter(w => w.creator?.username === user?.username).length}</p>
                  <p className="text-xs text-gray-400">Workflows Shared</p>
                </div>
                <div>
                  <p className="text-2xl font-bold text-turquoise">
                    {workflows
                      .filter(w => w.creator?.username === user?.username)
                      .reduce((sum, w) => sum + (w.votes || 0), 0)}
                  </p>
                  <p className="text-xs text-gray-400">Total Votes</p>
                </div>
                <div>
                  <p className="text-2xl font-bold text-turquoise">
                    {workflows
                      .filter(w => w.creator?.username === user?.username)
                      .reduce((sum, w) => sum + (w.views || 0), 0)}
                  </p>
                  <p className="text-xs text-gray-400">Total Views</p>
                </div>
                <div>
                  <p className="text-2xl font-bold text-turquoise">{bookmarkedWorkflows.length}</p>
                  <p className="text-xs text-gray-400">Bookmarks</p>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Recommended For You Section */}
        {isAuthenticated && workflows.length > 0 && (
          <div className="mb-8 bg-gray-800/30 rounded-xl p-6 border border-gray-700">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center gap-2">
                <FaHeart className="text-pink-500" size={20} />
                <h2 className="text-xl font-bold">Recommended For You</h2>
                <span className="text-xs bg-pink-500/20 text-pink-400 px-2 py-1 rounded">Personalized</span>
              </div>
              <button
                onClick={() => setShowPersonalized(!showPersonalized)}
                className="text-sm text-turquoise hover:underline"
              >
                {showPersonalized ? 'Hide' : 'Show All'}
              </button>
            </div>
            <div className={`grid grid-cols-1 md:grid-cols-${showPersonalized ? '3' : '4'} gap-4`}>
              {workflows
                .filter(w => !bookmarkedWorkflows.includes(w._id))
                .slice(0, showPersonalized ? 9 : 4)
                .map(workflow => (
                  <div
                    key={workflow._id}
                    className="bg-gray-800/50 rounded-lg p-4 hover:bg-gray-800 transition-all cursor-pointer border border-gray-700 hover:border-pink-500"
                    onClick={() => setSelectedWorkflow(workflow)}
                  >
                    <div className="flex items-center gap-2 mb-2">
                      <span className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded">
                        {workflow.category}
                      </span>
                      {workflow.difficulty && (
                        <span className="text-xs bg-blue-500/20 text-blue-400 px-2 py-1 rounded">
                          {workflow.difficulty}
                        </span>
                      )}
                    </div>
                    <h3 className="font-bold text-sm mb-2 line-clamp-2">{workflow.title}</h3>
                    <div className="flex items-center gap-3 text-xs text-gray-400">
                      <span className="flex items-center gap-1">
                        <FaArrowUp className="text-turquoise" size={10} /> {workflow.votes || 0}
                      </span>
                      <span className="flex items-center gap-1">
                        <FaEye size={10} /> {workflow.views || 0}
                      </span>
                    </div>
                  </div>
                ))}
            </div>
          </div>
        )}

        {/* Recently Viewed */}
        {isAuthenticated && recentlyViewed.length > 0 && (
          <div className="mb-8 bg-gray-800/30 rounded-xl p-6 border border-gray-700">
            <div className="flex items-center gap-2 mb-4">
              <FaHistory className="text-purple-500" size={20} />
              <h2 className="text-xl font-bold">Recently Viewed</h2>
            </div>
            <div className="flex gap-4 overflow-x-auto pb-2">
              {recentlyViewed.map(workflowId => {
                const workflow = workflows.find(w => w._id === workflowId);
                if (!workflow) return null;
                return (
                  <div
                    key={workflowId}
                    className="flex-shrink-0 w-64 bg-gray-800/50 rounded-lg p-4 hover:bg-gray-800 transition-all cursor-pointer border border-gray-700 hover:border-purple-500"
                    onClick={() => setSelectedWorkflow(workflow)}
                  >
                    <h3 className="font-bold text-sm mb-2 line-clamp-2">{workflow.title}</h3>
                    <p className="text-xs text-gray-400">{workflow.category}</p>
                  </div>
                );
              })}
            </div>
          </div>
        )}

        {/* Search and Filter Bar */}
        <div className="mb-6 space-y-4">
          <div className="flex flex-col md:flex-row gap-4">
            {/* Search */}
            <div className="flex-1 relative">
              <FaSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400" />
              <input
                type="text"
                placeholder="Search workflows by title, content, or tools..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pl-12 pr-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 placeholder-gray-500 focus:outline-none focus:border-turquoise"
              />
              {searchQuery && (
                <button
                  onClick={() => setSearchQuery('')}
                  className="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-white"
                >
                  <FaTimes />
                </button>
              )}
            </div>

            {/* Sort */}
            <select
              value={sortBy}
              onChange={(e) => setSortBy(e.target.value)}
              className="px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
            >
              {sortOptions.map(option => (
                <option key={option.value} value={option.value}>{option.label}</option>
              ))}
            </select>

            {/* View Mode */}
            <div className="flex items-center gap-2 bg-gray-800 border border-gray-700 rounded-lg p-1">
              <button
                onClick={() => setViewMode('grid')}
                className={`p-2 rounded ${viewMode === 'grid' ? 'bg-turquoise text-white' : 'text-gray-400 hover:text-white'}`}
              >
                <FaThLarge />
              </button>
              <button
                onClick={() => setViewMode('list')}
                className={`p-2 rounded ${viewMode === 'list' ? 'bg-turquoise text-white' : 'text-gray-400 hover:text-white'}`}
              >
                <FaList />
              </button>
            </div>

            {/* Filter Toggle */}
            <button
              onClick={() => setShowFilters(!showFilters)}
              className={`flex items-center gap-2 px-4 py-3 rounded-lg transition-colors ${
                showFilters ? 'bg-turquoise text-white' : 'bg-gray-800 text-gray-300 hover:bg-gray-700'
              }`}
            >
              <FaFilter /> Filters {activeFiltersCount > 0 && `(${activeFiltersCount})`}
            </button>
          </div>

          {/* Active Filters Chips */}
          {activeFiltersCount > 0 && (
            <div className="flex flex-wrap items-center gap-2">
              <span className="text-sm text-gray-400">Active filters:</span>
              {category !== 'all' && (
                <span className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                  Category: {category}
                  <button onClick={() => setCategory('all')} className="hover:text-white">
                    <FaTimes size={12} />
                  </button>
                </span>
              )}
              {searchQuery && (
                <span className="px-3 py-1 bg-turquoise/20 text-turquoise rounded-full text-sm flex items-center gap-2">
                  Search: "{searchQuery}"
                  <button onClick={() => setSearchQuery('')} className="hover:text-white">
                    <FaTimes size={12} />
                  </button>
                </span>
              )}
              {selectedDifficulty.map(diff => (
                <span key={diff} className="px-3 py-1 bg-blue-500/20 text-blue-400 rounded-full text-sm flex items-center gap-2">
                  {diff}
                  <button onClick={() => setSelectedDifficulty(selectedDifficulty.filter(d => d !== diff))} className="hover:text-white">
                    <FaTimes size={12} />
                  </button>
                </span>
              ))}
              {selectedTools.map(tool => (
                <span key={tool} className="px-3 py-1 bg-purple-500/20 text-purple-400 rounded-full text-sm flex items-center gap-2">
                  {tool}
                  <button onClick={() => setSelectedTools(selectedTools.filter(t => t !== tool))} className="hover:text-white">
                    <FaTimes size={12} />
                  </button>
                </span>
              ))}
              {selectedTime.map(time => (
                <span key={time} className="px-3 py-1 bg-green-500/20 text-green-400 rounded-full text-sm flex items-center gap-2">
                  {time}
                  <button onClick={() => setSelectedTime(selectedTime.filter(t => t !== time))} className="hover:text-white">
                    <FaTimes size={12} />
                  </button>
                </span>
              ))}
              <button
                onClick={clearAllFilters}
                className="px-3 py-1 bg-red-500/20 text-red-400 rounded-full text-sm hover:bg-red-500/30"
              >
                Clear All
              </button>
            </div>
          )}

          {/* Advanced Filters Panel */}
          {showFilters && (
            <div className="bg-gray-800 border border-gray-700 rounded-lg p-6 space-y-6">
              {/* Category Filters */}
              <div>
                <h3 className="text-sm font-semibold text-gray-300 mb-3">Category</h3>
                <div className="flex flex-wrap gap-2">
                  {categories.map((cat) => (
                    <button
                      key={cat}
                      onClick={() => setCategory(cat.toLowerCase())}
                      className={`px-4 py-2 rounded-lg transition-colors ${
                        category === cat.toLowerCase()
                          ? 'bg-turquoise text-white'
                          : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                      }`}
                    >
                      {cat}
                    </button>
                  ))}
                </div>
              </div>

              {/* Difficulty Filters */}
              <div>
                <h3 className="text-sm font-semibold text-gray-300 mb-3">Difficulty Level</h3>
                <div className="flex flex-wrap gap-2">
                  {difficulties.map(diff => (
                    <button
                      key={diff}
                      onClick={() => {
                        if (selectedDifficulty.includes(diff)) {
                          setSelectedDifficulty(selectedDifficulty.filter(d => d !== diff));
                        } else {
                          setSelectedDifficulty([...selectedDifficulty, diff]);
                        }
                      }}
                      className={`px-4 py-2 rounded-lg transition-colors ${
                        selectedDifficulty.includes(diff)
                          ? 'bg-blue-500 text-white'
                          : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                      }`}
                    >
                      {diff}
                    </button>
                  ))}
                </div>
              </div>

              {/* Tools Filters */}
              <div>
                <h3 className="text-sm font-semibold text-gray-300 mb-3">Tools Used</h3>
                <div className="flex flex-wrap gap-2">
                  {tools.map(tool => (
                    <button
                      key={tool}
                      onClick={() => {
                        if (selectedTools.includes(tool)) {
                          setSelectedTools(selectedTools.filter(t => t !== tool));
                        } else {
                          setSelectedTools([...selectedTools, tool]);
                        }
                      }}
                      className={`px-4 py-2 rounded-lg transition-colors ${
                        selectedTools.includes(tool)
                          ? 'bg-purple-500 text-white'
                          : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                      }`}
                    >
                      {tool}
                    </button>
                  ))}
                </div>
              </div>

              {/* Time Estimate Filters */}
              <div>
                <h3 className="text-sm font-semibold text-gray-300 mb-3">Time to Complete</h3>
                <div className="flex flex-wrap gap-2">
                  {timeEstimates.map(time => (
                    <button
                      key={time}
                      onClick={() => {
                        if (selectedTime.includes(time)) {
                          setSelectedTime(selectedTime.filter(t => t !== time));
                        } else {
                          setSelectedTime([...selectedTime, time]);
                        }
                      }}
                      className={`px-4 py-2 rounded-lg transition-colors ${
                        selectedTime.includes(time)
                          ? 'bg-green-500 text-white'
                          : 'bg-gray-700 text-gray-400 hover:bg-gray-600'
                      }`}
                    >
                      <FaClock className="inline mr-1" size={12} /> {time}
                    </button>
                  ))}
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Workflows List */}
        {loading ? (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-turquoise mx-auto"></div>
          </div>
        ) : filteredWorkflows.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-400 text-lg mb-4">No workflows found matching your filters</p>
            {activeFiltersCount > 0 && (
              <button onClick={clearAllFilters} className="btn-primary">
                Clear Filters
              </button>
            )}
          </div>
        ) : (
          <div className={viewMode === 'grid' ? 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6' : 'space-y-4'}>
            {filteredWorkflows.map((workflow) => (
              <WorkflowCard
                key={workflow._id}
                workflow={workflow}
                viewMode={viewMode}
                isBookmarked={bookmarkedWorkflows.includes(workflow._id)}
                onVote={handleVote}
                onBookmark={toggleBookmark}
                onShare={handleShare}
                onOpenDetail={() => handleOpenDetail(workflow)}
              />
            ))}
          </div>
        )}

        {/* Workflow Detail Modal */}
        {selectedWorkflow && (
          <WorkflowDetailModal
            workflow={selectedWorkflow}
            onClose={() => setSelectedWorkflow(null)}
            onVote={handleVote}
            isBookmarked={bookmarkedWorkflows.includes(selectedWorkflow._id)}
            onBookmark={toggleBookmark}
            onFork={handleForkWorkflow}
          />
        )}

        {/* Create Workflow Modal */}
        {showCreateModal && (
          <CreateWorkflowModal
            onClose={() => setShowCreateModal(false)}
            onSubmit={(newWorkflow) => {
              setWorkflows([newWorkflow, ...workflows]);
              setShowCreateModal(false);
            }}
          />
        )}
      </>
        )}
      </div>
    </div>
  );
};

// Enhanced Workflow Card Component
const WorkflowCard = ({ workflow, viewMode, isBookmarked, onVote, onBookmark, onShare, onOpenDetail }) => {
  const difficultyColors = {
    'Beginner': 'bg-green-500/20 text-green-400',
    'Intermediate': 'bg-blue-500/20 text-blue-400',
    'Advanced': 'bg-red-500/20 text-red-400'
  };

  if (viewMode === 'list') {
    return (
      <div className="card flex gap-4 hover:border-turquoise/50 transition-all cursor-pointer" onClick={onOpenDetail}>
        {/* Vote Section */}
        <div className="flex flex-col items-center gap-2">
          <button
            onClick={(e) => { e.stopPropagation(); onVote(workflow._id, 'up'); }}
            className="text-gray-400 hover:text-turquoise transition-colors"
          >
            <FaArrowUp size={20} />
          </button>
          <span className="text-xl font-bold">{workflow.votes || 0}</span>
          <button
            onClick={(e) => { e.stopPropagation(); onVote(workflow._id, 'down'); }}
            className="text-gray-400 hover:text-red-500 transition-colors"
          >
            <FaArrowDown size={20} />
          </button>
        </div>

        {/* Content */}
        <div className="flex-1">
          <div className="flex items-center justify-between mb-2">
            <div className="flex items-center gap-2 flex-wrap">
              <span className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded">
                {workflow.category}
              </span>
              {workflow.difficulty && (
                <span className={`text-xs px-2 py-1 rounded ${difficultyColors[workflow.difficulty] || 'bg-gray-700 text-gray-400'}`}>
                  {workflow.difficulty}
                </span>
              )}
              {workflow.timeEstimate && (
                <span className="text-xs text-gray-400 flex items-center gap-1">
                  <FaClock size={10} /> {workflow.timeEstimate}
                </span>
              )}
              {workflow.verified && (
                <span className="text-xs bg-gold/20 text-gold px-2 py-1 rounded flex items-center gap-1">
                  <FaCheckCircle size={10} /> Verified
                </span>
              )}
            </div>
          </div>

          <h3 className="text-xl font-bold mb-2">{workflow.title}</h3>
          <p className="text-gray-400 line-clamp-2">{workflow.content}</p>

          {/* Tools/Tech Stack */}
          {workflow.tools && workflow.tools.length > 0 && (
            <div className="flex flex-wrap gap-2 mt-3">
              {workflow.tools.slice(0, 4).map(tool => (
                <span key={tool} className="text-xs px-2 py-1 bg-purple-500/20 text-purple-400 rounded">
                  {tool}
                </span>
              ))}
              {workflow.tools.length > 4 && (
                <span className="text-xs text-gray-500">+{workflow.tools.length - 4} more</span>
              )}
            </div>
          )}

          {/* Meta Info */}
          <div className="flex items-center gap-4 mt-3 text-sm text-gray-500">
            <span className="flex items-center gap-1">
              <FaEye /> {workflow.views || 0}
            </span>
            <span className="flex items-center gap-1">
              <FaComment /> {workflow.comments?.length || 0}
            </span>
            <span className="flex items-center gap-1">
              <FaBookmark /> {workflow.bookmarks || 0}
            </span>
            <span className="text-gray-600">•</span>
            <span>by {workflow.creator?.username || 'Anonymous'}</span>
          </div>
        </div>

        {/* Actions */}
        <div className="flex flex-col gap-2">
          <button
            onClick={(e) => { e.stopPropagation(); onBookmark(workflow._id); }}
            className={`p-2 rounded hover:bg-gray-700 ${isBookmarked ? 'text-gold' : 'text-gray-400'}`}
          >
            {isBookmarked ? <FaBookmark /> : <FaRegBookmark />}
          </button>
          <button
            onClick={(e) => { e.stopPropagation(); onShare(workflow, 'copy'); }}
            className="p-2 text-gray-400 rounded hover:bg-gray-700 hover:text-white"
          >
            <FaShare />
          </button>
        </div>
      </div>
    );
  }

  // Grid View
  return (
    <div className="card hover:border-turquoise/50 transition-all cursor-pointer group" onClick={onOpenDetail}>
      {/* Header with badges */}
      <div className="flex items-start justify-between mb-3">
        <div className="flex flex-wrap gap-2">
          <span className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded">
            {workflow.category}
          </span>
          {workflow.difficulty && (
            <span className={`text-xs px-2 py-1 rounded ${difficultyColors[workflow.difficulty] || 'bg-gray-700 text-gray-400'}`}>
              {workflow.difficulty}
            </span>
          )}
        </div>
        {workflow.verified && (
          <FaCheckCircle className="text-gold" title="Verified Workflow" />
        )}
      </div>

      {/* Title */}
      <h3 className="text-xl font-bold mb-2 group-hover:text-turquoise transition-colors">{workflow.title}</h3>

      {/* Description */}
      <p className="text-gray-400 text-sm mb-4 line-clamp-3">{workflow.content}</p>

      {/* Tools */}
      {workflow.tools && workflow.tools.length > 0 && (
        <div className="flex flex-wrap gap-1 mb-4">
          {workflow.tools.slice(0, 3).map(tool => (
            <span key={tool} className="text-xs px-2 py-1 bg-purple-500/20 text-purple-400 rounded">
              {tool}
            </span>
          ))}
          {workflow.tools.length > 3 && (
            <span className="text-xs text-gray-500">+{workflow.tools.length - 3}</span>
          )}
        </div>
      )}

      {/* Time estimate */}
      {workflow.timeEstimate && (
        <div className="text-sm text-gray-400 mb-4 flex items-center gap-2">
          <FaClock className="text-turquoise" size={14} />
          {workflow.timeEstimate}
        </div>
      )}

      {/* Stats */}
      <div className="flex items-center justify-between pt-4 border-t border-gray-700">
        <div className="flex items-center gap-3 text-sm text-gray-500">
          <span className="flex items-center gap-1">
            <FaArrowUp className="text-turquoise" /> {workflow.votes || 0}
          </span>
          <span className="flex items-center gap-1">
            <FaEye /> {workflow.views || 0}
          </span>
          <span className="flex items-center gap-1">
            <FaComment /> {workflow.comments?.length || 0}
          </span>
        </div>

        {/* Actions */}
        <div className="flex items-center gap-2">
          <button
            onClick={(e) => { e.stopPropagation(); onBookmark(workflow._id); }}
            className={`p-1.5 rounded hover:bg-gray-700 ${isBookmarked ? 'text-gold' : 'text-gray-400'}`}
          >
            {isBookmarked ? <FaBookmark size={14} /> : <FaRegBookmark size={14} />}
          </button>
          <button
            onClick={(e) => { e.stopPropagation(); onShare(workflow, 'copy'); }}
            className="p-1.5 text-gray-400 rounded hover:bg-gray-700 hover:text-white"
          >
            <FaShare size={14} />
          </button>
        </div>
      </div>

      {/* Creator */}
      <div className="mt-3 pt-3 border-t border-gray-700 text-xs text-gray-500">
        by {workflow.creator?.username || 'Anonymous'}
      </div>
    </div>
  );
};

// Workflow Detail Modal Component
const WorkflowDetailModal = ({ workflow, onClose, onVote, isBookmarked, onBookmark, onFork }) => {
  const [activeTab, setActiveTab] = useState('overview');
  const [copied, setCopied] = useState(false);

  const copyCode = (code) => {
    navigator.clipboard.writeText(code);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  const exportWorkflow = (format) => {
    let content = '';
    const filename = `${workflow.title.replace(/\s+/g, '-').toLowerCase()}`;
    
    if (format === 'markdown') {
      content = `# ${workflow.title}\n\n**Category:** ${workflow.category}\n**Difficulty:** ${workflow.difficulty || 'N/A'}\n**Time:** ${workflow.timeEstimate || 'N/A'}\n\n## Description\n${workflow.content}\n\n## Steps\n${workflow.steps?.map((step, i) => `${i + 1}. ${step.title}\n   ${step.description || ''}`).join('\n\n') || 'No steps available'}\n`;
    } else if (format === 'json') {
      content = JSON.stringify(workflow, null, 2);
    }

    const blob = new Blob([content], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${filename}.${format === 'markdown' ? 'md' : 'json'}`;
    a.click();
    URL.revokeObjectURL(url);
  };

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4 overflow-y-auto" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto border border-gray-700" onClick={e => e.stopPropagation()}>
        {/* Header */}
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-start justify-between z-10">
          <div className="flex-1">
            <div className="flex items-center gap-2 mb-3">
              <span className="text-xs bg-turquoise/20 text-turquoise px-2 py-1 rounded">
                {workflow.category}
              </span>
              {workflow.difficulty && (
                <span className="text-xs bg-blue-500/20 text-blue-400 px-2 py-1 rounded">
                  {workflow.difficulty}
                </span>
              )}
              {workflow.timeEstimate && (
                <span className="text-xs text-gray-400 flex items-center gap-1">
                  <FaClock size={10} /> {workflow.timeEstimate}
                </span>
              )}
              {workflow.verified && (
                <span className="text-xs bg-gold/20 text-gold px-2 py-1 rounded flex items-center gap-1">
                  <FaCheckCircle size={10} /> Verified
                </span>
              )}
            </div>
            <h2 className="text-3xl font-bold mb-2">{workflow.title}</h2>
            <p className="text-gray-400">by {workflow.creator?.username || 'Anonymous'}</p>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-white p-2">
            <FaTimes size={20} />
          </button>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-800 px-6">
          <div className="flex gap-4">
            {['overview', 'diagram', 'steps', 'code', 'comments', 'versions'].map(tab => (
              <button
                key={tab}
                onClick={() => setActiveTab(tab)}
                className={`py-3 px-4 border-b-2 transition-colors ${
                  activeTab === tab
                    ? 'border-turquoise text-turquoise'
                    : 'border-transparent text-gray-400 hover:text-white'
                }`}
              >
                {tab.charAt(0).toUpperCase() + tab.slice(1)}
              </button>
            ))}
          </div>
        </div>

        {/* Content */}
        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div>
                <h3 className="text-lg font-semibold mb-3">Description</h3>
                <p className="text-gray-400 whitespace-pre-line">{workflow.content}</p>
              </div>

              {workflow.tools && workflow.tools.length > 0 && (
                <div>
                  <h3 className="text-lg font-semibold mb-3">Tools & Technologies</h3>
                  <div className="flex flex-wrap gap-2">
                    {workflow.tools.map(tool => (
                      <span key={tool} className="px-3 py-2 bg-purple-500/20 text-purple-400 rounded-lg">
                        {tool}
                      </span>
                    ))}
                  </div>
                </div>
              )}

              <div className="grid grid-cols-4 gap-4">
                <div className="bg-gray-800 p-4 rounded-lg text-center">
                  <div className="text-2xl font-bold text-turquoise">{workflow.votes || 0}</div>
                  <div className="text-sm text-gray-400">Upvotes</div>
                </div>
                <div className="bg-gray-800 p-4 rounded-lg text-center">
                  <div className="text-2xl font-bold text-turquoise">{workflow.views || 0}</div>
                  <div className="text-sm text-gray-400">Views</div>
                </div>
                <div className="bg-gray-800 p-4 rounded-lg text-center">
                  <div className="text-2xl font-bold text-turquoise">{workflow.bookmarks || 0}</div>
                  <div className="text-sm text-gray-400">Bookmarks</div>
                </div>
                <div className="bg-gray-800 p-4 rounded-lg text-center">
                  <div className="text-2xl font-bold text-turquoise">{workflow.comments?.length || 0}</div>
                  <div className="text-sm text-gray-400">Comments</div>
                </div>
              </div>

              <div className="flex flex-wrap gap-3">
                <button onClick={() => onVote(workflow._id, 'up')} className="btn-primary flex items-center gap-2">
                  <FaArrowUp /> Upvote
                </button>
                <button
                  onClick={() => onBookmark(workflow._id)}
                  className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                    isBookmarked ? 'bg-gold text-vibe-black' : 'bg-gray-800 text-gray-300 hover:bg-gray-700'
                  }`}
                >
                  {isBookmarked ? <FaBookmark /> : <FaRegBookmark />}
                  {isBookmarked ? 'Bookmarked' : 'Bookmark'}
                </button>
                <button onClick={() => onFork(workflow._id)} className="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 flex items-center gap-2">
                  <FaCodeBranch /> Fork Workflow
                </button>
                <button onClick={() => exportWorkflow('markdown')} className="px-4 py-2 bg-gray-800 text-gray-300 rounded-lg hover:bg-gray-700 flex items-center gap-2">
                  <FaDownload /> Markdown
                </button>
                <button onClick={() => exportWorkflow('json')} className="px-4 py-2 bg-gray-800 text-gray-300 rounded-lg hover:bg-gray-700 flex items-center gap-2">
                  <FaDownload /> JSON
                </button>
              </div>
            </div>
          )}

          {activeTab === 'diagram' && (
            <div className="space-y-4">
              <div className="bg-blue-500/10 border border-blue-500/30 rounded-lg p-3 mb-4">
                <p className="text-sm text-blue-300 flex items-center gap-2">
                  <FaProjectDiagram />
                  Interactive visual workflow diagram - drag nodes to reposition, click connectors to link steps
                </p>
              </div>
              <VisualWorkflowBuilder 
                steps={workflow.steps || []} 
                readOnly={true}
              />
            </div>
          )}

          {activeTab === 'steps' && (
            <div className="space-y-4">
              {workflow.steps && workflow.steps.length > 0 ? (
                workflow.steps.map((step, index) => (
                  <div key={index} className="bg-gray-800 p-4 rounded-lg border-l-4 border-turquoise">
                    <div className="flex items-start gap-3">
                      <span className="flex-shrink-0 w-8 h-8 bg-turquoise text-vibe-black rounded-full flex items-center justify-center font-bold">
                        {index + 1}
                      </span>
                      <div className="flex-1">
                        <h4 className="font-bold mb-2">{step.title}</h4>
                        <p className="text-gray-400">{step.description}</p>
                        {step.code && (
                          <div className="mt-3 relative">
                            <pre className="bg-vibe-black p-4 rounded overflow-x-auto text-sm">
                              <code className="text-turquoise">{step.code}</code>
                            </pre>
                            <button onClick={() => copyCode(step.code)} className="absolute top-2 right-2 p-2 bg-gray-800 hover:bg-gray-700 rounded">
                              {copied ? <FaCheckCircle className="text-green-400" /> : <FaCopy />}
                            </button>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                ))
              ) : (
                <p className="text-gray-400 text-center py-8">No detailed steps available</p>
              )}
            </div>
          )}

          {activeTab === 'code' && (
            <div className="space-y-4">
              <p className="text-gray-400 text-center py-8">Code snippets feature coming soon</p>
            </div>
          )}

          {activeTab === 'comments' && (
            <div className="space-y-4">
              <div className="bg-gray-800 p-4 rounded-lg">
                <textarea placeholder="Add a comment..." className="w-full bg-gray-900 border border-gray-700 rounded-lg p-3 text-gray-300 resize-none" rows="3" />
                <div className="flex justify-end mt-2">
                  <button className="btn-primary">Post Comment</button>
                </div>
              </div>
              <p className="text-gray-400 text-center py-8">No comments yet. Be the first to comment!</p>
            </div>
          )}

          {activeTab === 'versions' && (
            <div className="space-y-6">
              {/* Fork Information */}
              {workflow.forkedFrom && (
                <div className="bg-blue-500/10 border border-blue-500/30 rounded-lg p-4">
                  <div className="flex items-center gap-2 mb-2">
                    <FaCodeBranch className="text-blue-400" />
                    <span className="text-sm font-medium text-blue-400">Forked From</span>
                  </div>
                  <a href="#" className="text-white hover:text-turquoise">
                    {workflow.forkedFrom.title} by {workflow.forkedFrom.creator?.username || 'Anonymous'}
                  </a>
                </div>
              )}

              {/* Fork Statistics */}
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-gray-800 p-4 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <FaCodeBranch className="text-purple-400" />
                    <span className="text-sm text-gray-400">Total Forks</span>
                  </div>
                  <div className="text-2xl font-bold text-white">{workflow.forkCount || 0}</div>
                </div>
                <div className="bg-gray-800 p-4 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <FaProjectDiagram className="text-turquoise" />
                    <span className="text-sm text-gray-400">Version</span>
                  </div>
                  <div className="text-2xl font-bold text-white">{workflow.version || '1.0.0'}</div>
                </div>
              </div>

              {/* Version History */}
              <div>
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <FaHistory className="text-turquoise" />
                  Version History
                </h3>
                <div className="space-y-3">
                  {(workflow.versionHistory || [
                    { version: '1.0.0', date: new Date().toISOString(), changes: 'Initial release', author: workflow.creator?.username }
                  ]).map((ver, idx) => (
                    <div key={idx} className="bg-gray-800 p-4 rounded-lg">
                      <div className="flex items-start justify-between mb-2">
                        <div className="flex items-center gap-2">
                          <span className="text-sm font-mono bg-turquoise/20 text-turquoise px-2 py-1 rounded">
                            v{ver.version}
                          </span>
                          <span className="text-xs text-gray-400">
                            by {ver.author || 'Unknown'}
                          </span>
                        </div>
                        <span className="text-xs text-gray-500">
                          {new Date(ver.date).toLocaleDateString()}
                        </span>
                      </div>
                      <p className="text-sm text-gray-300">{ver.changes}</p>
                      {idx === 0 && (
                        <div className="mt-2">
                          <span className="text-xs bg-green-500/20 text-green-400 px-2 py-1 rounded">
                            Current Version
                          </span>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              </div>

              {/* Forks List */}
              {workflow.forks && workflow.forks.length > 0 && (
                <div>
                  <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                    <FaCodeBranch className="text-purple-400" />
                    Popular Forks
                  </h3>
                  <div className="space-y-3">
                    {workflow.forks.slice(0, 5).map((fork, idx) => (
                      <div key={idx} className="bg-gray-800 p-4 rounded-lg hover:bg-gray-750 transition-colors cursor-pointer">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h4 className="font-medium text-white hover:text-turquoise">
                              {fork.title}
                            </h4>
                            <p className="text-xs text-gray-400 mt-1">
                              by {fork.creator?.username || 'Anonymous'} • {fork.votes || 0} votes
                            </p>
                          </div>
                          <span className="text-xs text-gray-500">
                            {new Date(fork.createdAt).toLocaleDateString()}
                          </span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

// Create Workflow Modal Component
const CreateWorkflowModal = ({ onClose, onSubmit }) => {
  const [step, setStep] = useState(1);
  const [submitting, setSubmitting] = useState(false);
  const [formData, setFormData] = useState({
    title: '',
    category: 'Frontend',
    difficulty: 'Beginner',
    timeEstimate: '< 5 min',
    content: '',
    tools: [],
    steps: []
  });

  const handleSubmit = async () => {
    if (!formData.title || !formData.content) {
      alert('Please fill in all required fields');
      return;
    }

    setSubmitting(true);
    try {
      const response = await api.post('/workflows', formData);
      onSubmit(response.data.workflow);
    } catch (error) {
      console.error('Create workflow error:', error);
      alert(error.response?.data?.message || 'Failed to create workflow');
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <div className="fixed inset-0 bg-black/80 z-50 flex items-center justify-center p-4" onClick={onClose}>
      <div className="bg-gray-900 rounded-xl max-w-3xl w-full max-h-[90vh] overflow-y-auto border border-gray-700" onClick={e => e.stopPropagation()}>
        <div className="sticky top-0 bg-gray-900 border-b border-gray-800 p-6 flex items-center justify-between z-10">
          <div>
            <h2 className="text-2xl font-bold">Create New Workflow</h2>
            <p className="text-sm text-gray-400">Step {step} of 3</p>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-white">
            <FaTimes size={20} />
          </button>
        </div>

        <div className="px-6 pt-4">
          <div className="flex items-center gap-2">
            {[1, 2, 3].map(s => (
              <div key={s} className={`flex-1 h-2 rounded-full ${s <= step ? 'bg-turquoise' : 'bg-gray-700'}`} />
            ))}
          </div>
        </div>

        <div className="p-6 space-y-6">
          {step === 1 && (
            <>
              <div>
                <label className="block text-sm font-semibold mb-2">Workflow Title *</label>
                <input
                  type="text"
                  value={formData.title}
                  onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                  placeholder="e.g., Git Workflow for Teams"
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                />
              </div>

              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-semibold mb-2">Category *</label>
                  <select
                    value={formData.category}
                    onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                    className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                  >
                    <option>Frontend</option>
                    <option>Backend</option>
                    <option>DevOps</option>
                    <option>Testing</option>
                    <option>Git</option>
                    <option>CI/CD</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-semibold mb-2">Difficulty *</label>
                  <select
                    value={formData.difficulty}
                    onChange={(e) => setFormData({ ...formData, difficulty: e.target.value })}
                    className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                  >
                    <option>Beginner</option>
                    <option>Intermediate</option>
                    <option>Advanced</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-semibold mb-2">Time *</label>
                  <select
                    value={formData.timeEstimate}
                    onChange={(e) => setFormData({ ...formData, timeEstimate: e.target.value })}
                    className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 focus:outline-none focus:border-turquoise"
                  >
                    <option>{'< 5 min'}</option>
                    <option>5-15 min</option>
                    <option>15-30 min</option>
                    <option>30+ min</option>
                  </select>
                </div>
              </div>

              <div>
                <label className="block text-sm font-semibold mb-2">Description *</label>
                <textarea
                  value={formData.content}
                  onChange={(e) => setFormData({ ...formData, content: e.target.value })}
                  placeholder="Describe your workflow..."
                  className="w-full px-4 py-3 bg-gray-800 border border-gray-700 rounded-lg text-gray-300 resize-none focus:outline-none focus:border-turquoise"
                  rows="6"
                />
              </div>
            </>
          )}

          {step === 2 && (
            <div>
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <FaProjectDiagram className="text-turquoise" />
                Build Workflow Diagram
              </h3>
              <p className="text-gray-400 text-sm mb-4">
                Create your workflow visually - add steps, connect them, and arrange the flow
              </p>
              <VisualWorkflowBuilder 
                steps={formData.steps}
                onChange={(updatedSteps) => setFormData({ ...formData, steps: updatedSteps })}
                readOnly={false}
              />
            </div>
          )}

          {step === 3 && (
            <div>
              <h3 className="text-lg font-semibold mb-4">Review & Publish</h3>
              <div className="bg-gray-800 p-4 rounded-lg space-y-3">
                <div><span className="text-sm text-gray-400">Title:</span><p className="font-semibold">{formData.title || 'Untitled'}</p></div>
                <div><span className="text-sm text-gray-400">Category:</span><p>{formData.category}</p></div>
                <div><span className="text-sm text-gray-400">Difficulty:</span><p>{formData.difficulty}</p></div>
              </div>
            </div>
          )}
        </div>

        <div className="sticky bottom-0 bg-gray-900 border-t border-gray-800 p-6 flex justify-between">
          {step > 1 && <button onClick={() => setStep(step - 1)} className="px-6 py-2 bg-gray-800 rounded-lg hover:bg-gray-700" disabled={submitting}>Back</button>}
          <div className="flex-1" />
          {step < 3 ? (
            <button onClick={() => setStep(step + 1)} className="btn-primary px-6 py-2">Next</button>
          ) : (
            <button 
              onClick={handleSubmit} 
              disabled={submitting}
              className="btn-primary px-6 py-2 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {submitting ? 'Publishing...' : 'Publish Workflow'}
            </button>
          )}
        </div>
      </div>
    </div>
  );
};

export default Workflows;
