import axios from 'axios';
import { getCSRFToken, refreshCSRFToken, clearCSRFToken } from '../utils/csrf.js';

const api = axios.create({
  baseURL: import.meta.env.VITE_API_URL || '/api',
  headers: {
    'Content-Type': 'application/json',
  },
  withCredentials: true, // Important for session cookies and CSRF
});

// Request interceptor to add token and CSRF token
api.interceptors.request.use(
  async (config) => {
    // Add JWT token
    const token = localStorage.getItem('token');
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }

    // Add CSRF token for state-changing methods
    if (['post', 'put', 'delete', 'patch'].includes(config.method.toLowerCase())) {
      try {
        const csrfToken = await getCSRFToken();
        config.headers['X-CSRF-Token'] = csrfToken;
      } catch (error) {
        console.error('Failed to get CSRF token:', error);
      }
    }

    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// Response interceptor to handle errors
api.interceptors.response.use(
  (response) => response,
  async (error) => {
    const originalRequest = error.config;

    // Handle 401 Unauthorized (expired/invalid JWT)
    if (error.response?.status === 401) {
      localStorage.removeItem('token');
      clearCSRFToken();
      window.location.href = '/signin';
      return Promise.reject(error);
    }

    // Handle 403 CSRF token errors (retry once with fresh token)
    if (error.response?.status === 403 && 
        error.response?.data?.error === 'CSRF_INVALID' && 
        !originalRequest._retry) {
      originalRequest._retry = true;
      
      try {
        // Refresh CSRF token and retry request
        const newCsrfToken = await refreshCSRFToken();
        originalRequest.headers['X-CSRF-Token'] = newCsrfToken;
        return api(originalRequest);
      } catch (retryError) {
        console.error('Failed to refresh CSRF token:', retryError);
        return Promise.reject(retryError);
      }
    }

    // Handle rate limiting (429)
    if (error.response?.status === 429) {
      const retryAfter = error.response.headers['retry-after'];
      const message = retryAfter 
        ? `Rate limit exceeded. Please try again in ${retryAfter} seconds.`
        : 'Too many requests. Please slow down.';
      
      // You can dispatch a notification here if you have a toast/notification system
      console.warn(message);
    }

    return Promise.reject(error);
  }
);

export default api;
