/**
 * CSRF Token Management Utility
 * Handles fetching and storing CSRF tokens for secure API requests
 */

let csrfToken = null;

/**
 * Fetch CSRF token from server
 * @returns {Promise<string>} The CSRF token
 */
export const fetchCSRFToken = async () => {
  try {
    // Use relative URL to go through Vite proxy in development
    const API_URL = import.meta.env.VITE_API_URL || '/api';
    const response = await fetch(`${API_URL}/csrf-token`, {
      method: 'GET',
      credentials: 'include', // Important for session cookies
    });

    if (!response.ok) {
      throw new Error('Failed to fetch CSRF token');
    }

    const data = await response.json();
    csrfToken = data.csrfToken;
    return csrfToken;
  } catch (error) {
    console.error('Error fetching CSRF token:', error);
    throw error;
  }
};

/**
 * Get current CSRF token (fetch if not available)
 * @returns {Promise<string>} The CSRF token
 */
export const getCSRFToken = async () => {
  if (!csrfToken) {
    await fetchCSRFToken();
  }
  return csrfToken;
};

/**
 * Refresh CSRF token (force new fetch)
 * @returns {Promise<string>} The new CSRF token
 */
export const refreshCSRFToken = async () => {
  csrfToken = null;
  return await fetchCSRFToken();
};

/**
 * Clear stored CSRF token (call on logout)
 */
export const clearCSRFToken = () => {
  csrfToken = null;
};

/**
 * Check if a CSRF token is currently stored
 * @returns {boolean} True if token exists
 */
export const hasCSRFToken = () => {
  return csrfToken !== null;
};
